# 通知設計書 40-page.scheduled

## 概要

本ドキュメントは、Ghost CMSにおける `page.scheduled` Webhookイベントの設計仕様を定義する。このWebhookはページがスケジュール公開に設定された際にトリガーされる。

### 本通知の処理概要

**業務上の目的・背景**：ページのスケジュール公開は、特定の日時にコンテンツを自動公開する機能である。外部システム（カレンダー連携、静的サイト生成パイプライン、SNS予約投稿ツールなど）がスケジュールされたページを事前に把握し、公開に備えた準備処理を行う必要がある。このWebhookにより、ページがスケジュールされたタイミングで外部システムへリアルタイムに通知を送信する。

**通知の送信タイミング**：ページのステータスが `draft` や他の状態から `scheduled` に変更された際、または新規ページが `scheduled` ステータスで作成された際にトリガーされる。具体的には、Postモデルの `onSaved`（新規作成時）または `onUpdated`（更新時）イベントハンドラ内で、ステータスが `scheduled` に変化したことを検知し、`page.scheduled` イベントを発火する。

**通知の受信者**：`webhooks` テーブルに登録されている `event = 'page.scheduled'` のWebhookエンドポイント全て。各Webhookは特定のIntegration（連携アプリケーション）に紐づいており、カスタムIntegration制限がある場合は内部Integrationのみに制限される。

**通知内容の概要**：ページの現在の状態（`current`）と変更前の状態（`previous`）を含むJSONペイロード。ページデータには、タイトル、スラグ、HTML/プレーンテキストコンテンツ、公開予定日時、著者情報、タグ情報、ティア情報などが含まれる。

**期待されるアクション**：受信システムは、スケジュールされたページの情報を基に、公開予定日時に合わせた準備処理（プレビュー生成、カレンダー登録、通知予約など）を実行することが期待される。

## 通知種別

Webhook（HTTP POST リクエスト）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（イベント駆動） |
| 優先度 | 中 |
| リトライ | 有・5回（テスト環境では0回） |

### 送信先決定ロジック

1. `webhooks` テーブルから `event = 'page.scheduled'` に一致するレコードを取得
2. カスタムIntegration制限（`customIntegrations` limit）が有効な場合、`integration.type = 'internal'` のWebhookのみに絞り込み
3. 各Webhookの `target_url` に対してHTTP POSTリクエストを送信

## 通知テンプレート

### Webhook通知の場合

| 項目 | 内容 |
|-----|------|
| HTTPメソッド | POST |
| Content-Type | application/json |
| Content-Version | v{Ghost APIバージョン} |
| タイムアウト | 2000ms |

### ペイロード構造

```json
{
  "page": {
    "current": {
      "id": "ページID",
      "uuid": "UUID",
      "title": "ページタイトル",
      "slug": "スラグ",
      "html": "HTMLコンテンツ",
      "plaintext": "プレーンテキスト",
      "status": "scheduled",
      "published_at": "公開予定日時（ISO8601形式）",
      "created_at": "作成日時",
      "updated_at": "更新日時",
      "url": "ページURL",
      "authors": [...],
      "tags": [...],
      "tiers": [...],
      "primary_author": {...}
    },
    "previous": {
      "status": "変更前のステータス",
      "updated_at": "変更前の更新日時"
    }
  }
}
```

### 署名ヘッダー

Webhookに `secret` が設定されている場合、以下の形式で署名ヘッダーが付与される：

```
X-Ghost-Signature: sha256={HMAC-SHA256署名}, t={タイムスタンプ}
```

### 添付ファイル

なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| page.current.id | ページID | posts.id | Yes |
| page.current.uuid | UUID | posts.uuid | Yes |
| page.current.title | タイトル | posts.title | Yes |
| page.current.slug | スラグ | posts.slug | Yes |
| page.current.status | ステータス | posts.status | Yes |
| page.current.published_at | 公開予定日時 | posts.published_at | Yes |
| page.current.html | HTMLコンテンツ | posts.html | No |
| page.current.authors | 著者リスト | posts_authors + users | Yes |
| page.previous | 変更前の値 | モデル._previousAttributes | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 画面操作/API | Page作成（status=scheduled） | 新規ページがscheduledステータスで作成 | Admin APIまたは管理画面からページを作成 |
| 画面操作/API | Page更新（status変更→scheduled） | ステータスがscheduledに変更 | 既存ページのステータスをscheduledに変更 |
| 画面操作/API | リソースタイプ変更 | post→pageの変換でscheduled状態 | 投稿からページへの変換時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| options.importing = true | データインポート中は送信しない |
| webhooks テーブルに該当レコードなし | 登録されたWebhookがない場合 |
| customIntegrations制限超過 | 内部Integration以外のWebhookは無効化 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ページ作成/更新] --> B{ステータス変更?}
    B -->|Yes| C{scheduled に変更?}
    B -->|No| D{既にscheduled?}
    C -->|Yes| E[page.scheduled イベント発火]
    C -->|No| Z[終了]
    D -->|Yes| F{published_at変更?}
    D -->|No| Z
    F -->|Yes| G[page.rescheduled イベント発火]
    F -->|No| Z
    E --> H[WebhookTrigger.trigger 呼び出し]
    H --> I[webhooks テーブル検索]
    I --> J{Webhook存在?}
    J -->|No| Z
    J -->|Yes| K[各Webhookにペイロード送信]
    K --> L{送信結果}
    L -->|成功| M[last_triggered_at/status更新]
    L -->|失敗| N[エラー記録・リトライ]
    M --> Z
    N --> Z
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| posts | ページデータ取得 | type='page'、currentとpreviousの値を取得 |
| webhooks | Webhook設定取得 | event='page.scheduled'で検索 |
| integrations | Integration情報 | Webhook経由で関連取得 |
| users | 著者情報 | posts_authors経由 |
| tags | タグ情報 | posts_tags経由 |
| products | ティア情報 | posts_products経由 |

### テーブル別参照項目詳細

#### posts

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| type | ページ判定 | type = 'page' |
| status | ステータス確認 | scheduled であること |
| published_at | 公開予定日時 | 必須 |
| 全カラム | ペイロード生成 | id指定 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| webhooks | UPDATE | 送信結果の記録 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 送信失敗 | ネットワークエラー、タイムアウト | リトライ実行、エラーログ記録 |
| 410 Gone | 送信先が恒久的に削除 | Webhookレコードを自動削除 |
| 4xx/5xx | サーバーエラー | ステータス記録、リトライ実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 5回（本番環境）、0回（テスト環境） |
| リトライ間隔 | @tryghost/requestライブラリのデフォルト設定に依存 |
| リトライ対象エラー | ネットワークエラー、一時的なサーバーエラー |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント発生ベース） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（イベント発生時に即座に送信）

## セキュリティ考慮事項

- Webhookに `secret` を設定することで、HMAC-SHA256署名による送信元検証が可能
- HTTPS URLの使用を推奨
- 署名検証により、受信側でリクエストの正当性を確認可能

## 備考

- `page.scheduled` はページが初めてスケジュールされた時にトリガー
- 既にスケジュール済みのページの公開日時を変更した場合は `page.rescheduled` がトリガー
- スケジュールを解除（draft等に戻す）した場合は `page.unscheduled` がトリガー
- 新規作成時に `status=scheduled` の場合、`page.added` と `page.scheduled` の両方が発火
- ページタイプは `posts` テーブルの `type='page'` で識別される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | schema.js | `ghost/core/core/server/data/schema/schema.js` | postsテーブル定義、type='page'の定義 |
| 1-2 | listen.js | `ghost/core/core/server/services/webhooks/listen.js` | WEBHOOKS配列で'page.scheduled'が定義されている（29行目） |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | post.js | `ghost/core/core/server/models/post.js` | `onSaved`（378-392行目）と`onUpdated`（394-464行目）でscheduledイベント発火を確認 |

**主要処理フロー**:
1. **389-391行目**: 新規作成時、statusが'scheduled'なら`model.emitChange('scheduled')`を呼び出し
2. **399行目**: `isScheduled = model.get('status') === 'scheduled'` で判定
3. **439-441行目**: 更新時、statusが'scheduled'に変化したら`model.emitChange('scheduled')`を呼び出し
4. **356-367行目**: `emitChange`メソッドで`page.scheduled`イベントを発火

#### Step 3: Webhook送信処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | triggerメソッド |
| 3-2 | serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | ペイロード生成 |

### プログラム呼び出し階層図

```
ページスケジュール（Admin API / 管理画面）
    │
    └─ Post.edit() / Post.add() (type='page')
           │
           ├─ onSaved() (新規作成時)
           │      └─ if (status === 'scheduled')
           │              └─ emitChange('scheduled')
           │
           └─ onUpdated() (更新時)
                  │
                  └─ if (statusChanging && isScheduled)
                         └─ emitChange('scheduled')
                                │
                                └─ resourceType = 'page'
                                └─ events.emit('page.scheduled', model)
                                       │
                                       └─ WebhookTrigger.trigger('page.scheduled', model)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ページスケジュール ───▶ Postモデル (type='page')
リクエスト               │ (ステータス変更検知)
                        │ status → 'scheduled'
                        ▼
                      emitChange('scheduled')
                        │ resourceType = 'page'
                        ▼
                      WebhookTrigger ───▶ HTTP POST
                        │                 {page: {current: {status: 'scheduled', ...},
                        │                         previous: {...}}}
                        ▼
                      webhooksテーブル更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| post.js | `ghost/core/core/server/models/post.js` | ソース | ページモデル、イベント発火 |
| listen.js | `ghost/core/core/server/services/webhooks/listen.js` | ソース | イベントリスナー登録（29行目） |
| webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | ソース | Webhook送信処理 |
| serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | ソース | シリアライズ |
| schema.js | `ghost/core/core/server/data/schema/schema.js` | 設定 | DBスキーマ定義 |
| pages.test.js | `ghost/core/test/e2e-webhooks/pages.test.js` | テスト | E2Eテスト（309-359行目） |
