# セキュリティ設計書

## 概要

本ドキュメントは、GitLabプラットフォームにおけるセキュリティ設計を定義するものです。GitLabはDevOpsプラットフォームとして、ソースコード管理、CI/CD、イシュー管理など多岐にわたる機能を提供しており、エンタープライズレベルのセキュリティ対策が実装されています。

## 認証設計

### 認証方式

GitLabでは、Deviseフレームワークを基盤として複数の認証方式をサポートしています。

| 認証方式 | 説明 | 設定ファイル |
| --- | --- | --- |
| データベース認証 | ユーザー名/メール + パスワードによる標準認証 | `config/initializers/8_devise.rb` |
| LDAP認証 | 外部LDAPサーバーとの連携認証 | `config/initializers/8_devise.rb` |
| OmniAuth | OAuth/SAML等の外部認証プロバイダー連携 | `app/controllers/omniauth_callbacks_controller.rb` |
| 二要素認証（2FA） | TOTP/WebAuthn/パスキーによる追加認証 | `app/controllers/concerns/authenticates_with_two_factor.rb` |
| パスワードレス認証 | パスキーによるパスワードなし認証 | `app/controllers/concerns/authenticates_with_two_factor.rb` |
| アクセストークン認証 | 個人アクセストークン/デプロイトークン | `app/models/personal_access_token.rb`, `app/models/deploy_token.rb` |

### パスワード要件

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| 最小文字数 | 8文字 | `config/initializers/8_devise.rb` |
| 最大文字数 | 128文字 | Devise標準設定 |
| ハッシュアルゴリズム | BCrypt (stretches: 13) または PBKDF2-SHA512 | 本番環境では13ラウンド |
| ソルト長 | 64バイト（PBKDF2） | `app/models/concerns/encrypted_user_password.rb` |

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | 10080分（デフォルト7日） | `config/initializers/session_store.rb` |
| セッション固定化対策 | 実装済み | Devise標準機能 |
| Cookie暗号化 | 有効 | Rails標準のsigned/encrypted cookie |
| セッションストア | Redis | Redisベースのセッション管理 |
| ログアウト方式 | POST | `config.sign_out_via = :post` |

### アカウントロック機能

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| ロック戦略 | 失敗回数ベース | `config.lock_strategy = :failed_attempts` |
| 最大失敗回数 | 10回（デフォルト） | カスタマイズ可能 |
| アンロック方式 | メール + 時間 | `config.unlock_strategy = :both` |
| アンロック待機時間 | 10分（デフォルト） | カスタマイズ可能 |

## 認可設計

### 権限体系

GitLabはDeclarativePolicyフレームワークを使用したロールベースアクセス制御（RBAC）を実装しています。

| ロール | 権限 | 説明 |
| --- | --- | --- |
| Guest | read_project, create_issue, create_note | 閲覧と限定的な投稿が可能 |
| Planner | guest_access + admin_issue, create_design | プロジェクト計画関連の管理が可能 |
| Reporter | planner_access + read_code, download_code | コードの閲覧とダウンロードが可能 |
| Developer | reporter_access + push_code, create_merge_request | コード変更とマージリクエスト作成が可能 |
| Maintainer | developer_access + admin_project, admin_pipeline | プロジェクト設定の管理が可能 |
| Owner | maintainer_access + change_namespace, remove_project | プロジェクトの完全な制御が可能 |
| Admin | admin_all_resources, read_all_resources | インスタンス全体の管理が可能 |

### アクセス制御

アクセス制御はポリシークラス（`app/policies/`）で実装されています。

| ポリシークラス | 対象リソース | 主な制御内容 |
| --- | --- | --- |
| `BasePolicy` | 共通 | 管理者権限、組織所属チェック |
| `ProjectPolicy` | プロジェクト | プロジェクト機能へのアクセス制御 |
| `GroupPolicy` | グループ | グループリソースへのアクセス制御 |
| `MergeRequestPolicy` | マージリクエスト | MR操作の権限制御 |
| `IssuePolicy` | イシュー | イシュー操作の権限制御 |

### 外部認可

| 機能 | 説明 |
| --- | --- |
| 外部認可サービス | 外部サービスによるアクセス制御が可能 |
| 分類ラベル認可 | プロジェクトの分類ラベルによるアクセス制御 |
| CIジョブトークンスコープ | CI/CDジョブからのアクセス範囲制御 |

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | 設定可能（`config/environments/production.rb`でforce_ssl設定可能） |
| HSTS | Let's Encrypt連携によるHTTPS自動設定をサポート |
| WebSocket | WSS（セキュアWebSocket）サポート |
| TLS | TLS 1.2以上を推奨（OpenSSL設定に依存） |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | TLS 1.2+ |
| パスワード | BCrypt（stretches: 13）または PBKDF2-SHA512 |
| 保存データ（機密情報） | AES-256-GCM（attr_encrypted使用） |
| トークン | ハッシュ化保存（PBKDF2-SHA512） |
| Terraformステート | Lockbox暗号化 |
| CI/CDセキュアファイル | Lockbox暗号化 |

### 暗号化対象フィールド

| モデル | フィールド | 暗号化方式 |
| --- | --- | --- |
| ApplicationSetting | external_auth_client_key | AES-256-GCM |
| ApplicationSetting | lets_encrypt_private_key | AES-256-GCM |
| ApplicationSetting | akismet_api_key | AES-256-GCM |
| ApplicationSetting | recaptcha_private_key | AES-256-GCM |
| RemoteMirror | credentials | AES-256-GCM |
| ErrorTrackingProjectSetting | token | AES-256-GCM |
| SlackIntegration | bot_access_token | AES-256-GCM |

### 機密情報管理

| 項目 | 対策 |
| --- | --- |
| シークレットトークン | `config/initializers/2_secret_token.rb`で管理 |
| Workhorse共有シークレット | `config/initializers/gitlab_workhorse_secret.rb` |
| KAS共有シークレット | `config/initializers/gitlab_kas_secret.rb` |
| 環境変数 | 機密情報は環境変数経由で設定 |

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | Content Security Policy（CSP）実装、テンプレートでの自動エスケープ |
| SQLインジェクション | ActiveRecordのパラメータ化クエリ、`sanitize_sql_like`使用 |
| CSRF | `protect_from_forgery with: :exception`（ApplicationController） |
| クリックジャッキング | CSP frame-ancestors、X-Frame-Options |
| オープンリダイレクト | リダイレクト先の検証 |

### Content Security Policy（CSP）設定

| ディレクティブ | 設定値 |
| --- | --- |
| default_src | 'self' |
| script_src | 'self'（nonce付き） |
| style_src | 'self' |
| img_src | 'self' data: blob: http: https: |
| object_src | 'none' |
| frame_ancestors | 'self' |
| form_action | 'self' https: http: |

CSP設定は `config/initializers/content_security_policy.rb` および `lib/gitlab/content_security_policy/config_loader.rb` で管理されています。

### CSRF保護

| コントローラー | 設定 |
| --- | --- |
| ApplicationController | `protect_from_forgery with: :exception, prepend: true` |
| SessionsController | `protect_from_forgery with: :exception, prepend: true, except: :destroy` |
| OmniauthCallbacksController | SAMLプロバイダーは例外設定 |
| HealthController | `protect_from_forgery with: :exception` |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| 認証イベント | ログイン成功/失敗、認証方式、IPアドレス | 設定による |
| 監査イベント | ユーザー操作、権限変更、設定変更 | パーティション管理（月次） |
| セキュリティログ | セキュリティ関連イベント、警告 | 設定による |
| アプリケーションログ | 一般的な操作ログ | 設定による |

### 監査イベント実装

| コンポーネント | ファイル | 説明 |
| --- | --- | --- |
| AuditEvent | `app/models/audit_event.rb` | 監査イベントモデル |
| AuditEventService | `app/services/audit_event_service.rb` | 監査イベント記録サービス |
| AuthenticationEvent | `app/models/authentication_event.rb` | 認証イベント記録 |

### 監査ログの特徴

| 項目 | 内容 |
| --- | --- |
| メッセージサニタイズ | Sanitize.cleanによるXSS対策 |
| IPアドレス記録 | IPアドレスの検証と記録 |
| 作成者追跡 | ユーザー/トークン/APIキーの追跡 |
| パーティション管理 | 月次でのテーブルパーティション |

## 追加セキュリティ機能

### レート制限

| 機能 | 説明 |
| --- | --- |
| IPブロック | 悪意あるIPアドレスのブロック |
| 一意IPアドレス制限 | 同一アカウントからの接続IP数制限 |
| リクエストレート制限 | API/Webリクエストのレート制限 |

### ボット検出・対策

| 機能 | 説明 |
| --- | --- |
| Akismet連携 | スパムコンテンツの検出 |
| reCAPTCHA | 自動化攻撃の防止 |
| invisible_captcha | フォームボット対策 |

### 追加認証機能

| 機能 | 説明 |
| --- | --- |
| Admin Mode | 管理者操作時の追加認証要求 |
| Step-up認証 | 機密操作時の追加認証 |
| パスワード変更通知 | パスワード変更時のメール通知 |
| メールアドレス変更通知 | メール変更時の通知 |

## 備考

- GitLabはRuby on Rails 7.xフレームワーク上で構築されており、Railsのセキュリティ機能を最大限に活用しています
- セキュリティ設定の多くは`ApplicationSettings`で管理者がカスタマイズ可能です
- Enterprise Edition（EE）では追加のセキュリティ機能（監査ストリーミング、外部認可など）が利用可能です
- 定期的なセキュリティアップデートとパッチリリースが行われています
