# バッチ設計書 34-Authn::DataRetention::AuthenticationEventArchiveWorker

## 概要

本ドキュメントは、認証イベントのアーカイブ（削除）を行うバッチ処理「Authn::DataRetention::AuthenticationEventArchiveWorker」の設計書です。

### 本バッチの処理概要

**業務上の目的・背景**：GitLabでは、ユーザーの認証イベント（ログイン、ログアウト、認証失敗など）がauthentication_eventsテーブルに記録されます。これらのデータは監査やセキュリティ分析に重要ですが、長期間蓄積されるとデータベースのパフォーマンスやストレージに影響を与えます。本バッチは、保持期間（1年）を超えた古い認証イベントを削除することで、データベースの健全性を維持しながらコンプライアンス要件を満たします。

**バッチの実行タイミング**：日次で午前6時10分に実行されます。cronスケジュールは`10 6 * * *`で設定されています。

**主要な処理内容**：
1. 機能フラグとアプリケーション設定の確認
2. authentication_eventsテーブルを10,000件単位でバッチ処理
3. 各バッチをさらに1,000件単位のサブバッチに分割
4. 保持期間（1年）を超えたイベントをCTEを使用して削除
5. 実行時間が3分を超えた場合は3分後に継続処理をスケジュール

**前後の処理との関連**：OauthAccessGrantArchiveWorker、OauthAccessTokenArchiveWorkerと同じデータ保持ポリシーの一環として実行され、認証関連データのライフサイクル管理を担います。

**影響範囲**：authentication_eventsテーブルのデータ削除、データベースパフォーマンス

## バッチ種別

データクレンジング

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 日次 |
| 実行時刻 | 06:10 |
| 実行曜日 | 毎日 |
| 実行日 | - |
| トリガー | cron |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Sidekiq稼働 | Sidekiqワーカーが正常に稼働していること |
| データベース接続 | メインデータベースへの接続が確立されていること |
| 機能フラグ有効 | archive_authentication_eventsフィーチャーフラグが有効であること |
| 設定有効 | authn_data_retention_cleanup_enabledが有効であること |

### 実行可否判定

- authn_data_retention_cleanup_enabledがfalseの場合は処理をスキップ
- archive_authentication_eventsフィーチャーフラグが無効の場合は処理をスキップ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| cursor | Integer | No | nil | 前回処理の続きを示すイベントID |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| authentication_events | DB | 認証イベントテーブル |
| application_settings | DB | authn_data_retention_cleanup_enabled設定を取得 |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| authentication_events | DB | 古いイベントを削除 |
| Sidekiqキュー | ジョブ | 継続処理が必要な場合は自己スケジュール |
| ログ | Stdout | 削除件数、カットオフ日時などをログ出力 |

### 出力ファイル仕様

ファイル出力はありません。

## 処理フロー

### 処理シーケンス

```
1. バッチ開始
   └─ cursorパラメータを受け取り（初回はnil）
2. 実行条件チェック
   └─ authn_data_retention_cleanup_enabled確認
   └─ archive_authentication_eventsフィーチャーフラグ確認
3. ランタイムリミッター初期化
   └─ 最大実行時間3分を設定
4. バッチ処理（10,000件単位）
   └─ cursor以降のイベントをeach_batchで処理
5. サブバッチ処理（1,000件単位）
   └─ CTEを使用して古いイベントを削除
6. ログ出力
   └─ 削除件数をログ記録
7. スリープ
   └─ 0.1秒の遅延を挿入
8. 実行時間チェック
   └─ 3分超過時は3分後に自己スケジュール
9. メタデータ記録
   └─ over_time、total_deleted、cutoff_timeを記録
10. バッチ終了
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B{設定が有効?}
    B -->|いいえ| C[処理スキップ]
    B -->|はい| D{機能フラグ有効?}
    D -->|いいえ| C
    D -->|はい| E[ランタイムリミッター初期化]
    E --> F[10,000件バッチ取得]
    F --> G{イベントが存在?}
    G -->|なし| H[メタデータ記録]
    G -->|あり| I[1,000件サブバッチ処理]
    I --> J[CTEでold_eventsを削除]
    J --> K[削除件数をログ]
    K --> L[0.1秒スリープ]
    L --> M{サブバッチ完了?}
    M -->|いいえ| I
    M -->|はい| N{実行時間3分超過?}
    N -->|はい| O[3分後に自己スケジュール]
    O --> H
    N -->|いいえ| P{次のバッチあり?}
    P -->|あり| F
    P -->|なし| H
    H --> Q[バッチ終了]
    C --> Q
```

## データベース操作仕様

### 操作別データベース影響一覧

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| イベント取得 | authentication_events | SELECT | バッチ処理用にイベントを取得 |
| イベント削除 | authentication_events | DELETE | CTEを使用して古いイベントを削除 |
| 設定取得 | application_settings | SELECT | データ保持設定を取得 |

### テーブル別操作詳細

#### authentication_events

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | id, created_at | id > cursor | バッチ処理用 |
| DELETE | - | created_at <= retention_period_cutoff | CTEで条件指定 |

**削除SQL構造（CTE使用）**:
```sql
WITH batch AS MATERIALIZED (
    SELECT id, created_at FROM authentication_events LIMIT 1000
),
filtered_batch AS MATERIALIZED (
    SELECT id, created_at FROM batch
    WHERE created_at <= '保持期間カットオフ日時' LIMIT 1000
)
DELETE FROM authentication_events
WHERE id IN (SELECT id FROM filtered_batch)
```

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | DB接続エラー | データベースへの接続失敗 | Sidekiqの標準リトライで再実行 |
| - | 設定無効 | authn_data_retention_cleanup_enabledがfalse | 処理をスキップ |
| - | 機能フラグ無効 | archive_authentication_eventsが無効 | 処理をスキップ |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | Sidekiqデフォルト（25回） |
| リトライ間隔 | 指数バックオフ |
| リトライ対象エラー | すべての例外 |

### 障害時対応

- idempotent!が設定されているため、再実行しても問題なし
- cursorによる継続処理により、中断時も途中から再開可能
- defer_on_database_health_signalによりDB負荷時は5分延期

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | サブバッチ単位（1,000件） |
| コミットタイミング | DELETE文実行時（暗黙コミット） |
| ロールバック条件 | DELETE文失敗時 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 数万〜数十万件/日（環境依存） |
| 目標処理時間 | 3分/バッチ（MAX_RUNTIME制限） |
| メモリ使用量上限 | 制限なし（バッチ処理により制御） |

## 排他制御

- CronjobQueueにより同時実行は自動的に防止される
- deduplicate :until_executing, including_scheduled: trueにより重複排除
- concurrency_limit 1により同時実行数を制限

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 開始ログ | バッチ開始時 | Sidekiq標準ログ |
| 進捗ログ | サブバッチ削除後 | "Deleted {count} authentication events" |
| 終了ログ | バッチ終了時 | over_time, total_deleted, cutoff_time |
| エラーログ | エラー発生時 | 例外情報 |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| ジョブ失敗 | 1件以上 | Sidekiqダッシュボード |
| 処理時間 | 3分 | 自動で次バッチにスケジュール |
| DB健全性 | authentication_eventsシグナル | 5分延期 |

## 備考

- feature_category: system_access
- data_consistency: sticky
- BATCH_SIZE: 10,000件
- SUB_BATCH_SIZE: 1,000件
- MAX_RUNTIME: 3分
- REQUEUE_DELAY: 3分
- ITERATION_DELAY: 0.1秒
- RETENTION_PERIOD: 1年（AuthenticationEvent::RETENTION_PERIOD）
- defer_on_database_health_signal: gitlab_main, [:authentication_events], 5分
