# 機能設計書 127-CI/CDアナリティクス

## 概要

本ドキュメントは、GitLabにおけるCI/CDアナリティクス機能の設計仕様を記述したものである。本機能は、パイプライン実行統計、ジョブ成功率、コードカバレッジの推移などCI/CDに関する分析データを提供し、継続的インテグレーション・デリバリーの改善を支援する。

### 本機能の処理概要

**業務上の目的・背景**：DevOpsプラクティスの成熟度向上には、CI/CDパイプラインのパフォーマンスを継続的に測定・改善することが重要である。CI/CDアナリティクスは、パイプラインの実行頻度、成功率、所要時間、コードカバレッジの推移などを可視化し、ボトルネックの特定や品質改善の効果測定を可能にする。

**機能の利用シーン**：プロジェクトのAnalyticsメニューから「CI/CD analytics」を選択した際に表示される。パイプラインの健全性を評価する場面、カバレッジ向上の取り組みを追跡する場面、DevOpsメトリクスをレポートする場面などで利用される。

**主要な処理内容**：
1. 日次ビルドグループレポート結果の取得
2. コードカバレッジの時系列データ集計
3. パイプライン実行統計の計算
4. ジョブグループ別のカバレッジ比較
5. CSVエクスポート機能
6. グラフデータのJSON提供

**関連システム・外部連携**：CI/CDパイプラインからのカバレッジ結果収集。DevOpsレポート機能との連携。

**権限による制御**：ビルドレポート結果の読み取り権限（authorize_read_build_report_results!）が必要。プロジェクトへの読み取りアクセス権が前提。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 269 | DevOpsレポート | 主画面 | DevOpsレポート表示 |

## 機能種別

データ取得・分析（READ操作）/ 統計レポート生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| param_type | String | Yes | パラメータ種別（coverage等） | PARAM_TYPESに含まれること |
| start_date | String | No | 集計開始日（YYYY-MM-DD） | 有効な日付形式 |
| end_date | String | No | 集計終了日（YYYY-MM-DD） | 有効な日付形式 |
| ref_path | String | No | 分析対象ブランチパス | リポジトリに存在するref |
| format | String | No | 出力形式（csv/json） | csv, json |

### 入力データソース

- ci_daily_build_group_report_results テーブル
- projects テーブル
- ci_pipelines テーブル（参照）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| date | Date | 集計日 |
| group_name | String | ジョブグループ名 |
| coverage | Float | カバレッジ値（%） |
| data | Hash | 各種メトリクスデータ |

### 出力先

- CSV形式（ダウンロード）
- JSON API（グラフ描画用）

## 処理フロー

### 処理シーケンス

```
1. 権限チェック
   └─ authorize_read_build_report_results!
2. パラメータ検証
   └─ validate_param_type!
3. レポート結果取得
   └─ DailyBuildGroupReportResultsFinder実行
4. フィルタ適用
   └─ ref_path、日付範囲でフィルタ
5. レスポンス生成
   └─ CSV形式: CsvBuilder
   └─ JSON形式: Serializer
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{権限チェック}
    B -->|No| C[403 Forbidden]
    B -->|Yes| D{param_type検証}
    D -->|Invalid| E[422 Unprocessable]
    D -->|Valid| F[Finder実行]
    F --> G[フィルタ適用]
    G --> H{出力形式}
    H -->|CSV| I[CsvBuilder生成]
    H -->|JSON| J[Serializer生成]
    I --> K[ダウンロード]
    J --> L[JSONレスポンス]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-127-01 | レポート期間 | 最大90日間のデータを取得可能 | 日付範囲指定時 |
| BR-127-02 | デフォルトブランチ | ref_path未指定時はデフォルトブランチを使用 | ref_path未指定時 |
| BR-127-03 | 最大件数 | MAX_ITEMS = 1000件まで取得 | 常時 |
| BR-127-04 | ソート順 | 日付とグループ名で昇順ソート | sort: true時 |

### 計算ロジック

**日付範囲制限**:
```ruby
REPORT_WINDOW = 90.days
start_date = [start_date, end_date - REPORT_WINDOW].max
# 開始日は終了日から最大90日前まで
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| レポート取得 | ci_daily_build_group_report_results | SELECT | 日次カバレッジ結果取得 |
| プロジェクト確認 | projects | SELECT | プロジェクト情報取得 |

### テーブル別操作詳細

#### ci_daily_build_group_report_results

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | project_id, ref_path, date, group_name, data | project_id = @project.id AND date BETWEEN start_date AND end_date | カバレッジ取得 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 403 | Forbidden | 権限不足 | アクセス拒否 |
| 422 | Unprocessable Entity | 不正なparam_type | エラーメッセージ返却 |

### リトライ仕様

データベースタイムアウト時は標準的なリトライ機構に従う。

## トランザクション仕様

読み取り専用のため、明示的なトランザクション管理は不要。

## パフォーマンス要件

- feature_category: code_testing
- MAX_ITEMS = 1000件の制限
- インデックス: (project_id, ref_path, date)

## セキュリティ考慮事項

- ビルドレポート読み取り権限の確認
- プロジェクト可視性に基づくアクセス制御

## 備考

- PARAM_TYPESはモデルで定義（coverageなど）
- GraphsController#chartsからも参照される（get_daily_coverage_options）
- EE版で追加の分析機能が提供される可能性あり

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

日次ビルドレポート結果の構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Ci::DailyBuildGroupReportResult | `app/models/ci/daily_build_group_report_result.rb` | レポート結果モデル |

**読解のコツ**: dataカラムにはJSONB形式でcoverage等のメトリクスが格納される。PARAM_TYPESで許可される種別が定義されている。

#### Step 2: エントリーポイントを理解する

コントローラーの処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Projects::Ci::DailyBuildGroupReportResultsController | `app/controllers/projects/ci/daily_build_group_report_results_controller.rb` | indexアクションの処理フロー |

**主要処理フロー**:
1. **4-5行目**: before_actionでの権限・パラメータチェック
2. **9-14行目**: indexアクション - CSV/JSON分岐
3. **22-31行目**: render_csvメソッド - CSVビルダー
4. **33-35行目**: render_jsonメソッド - シリアライザー
5. **37-42行目**: report_resultsメソッド - Finder呼び出し

#### Step 3: Finderを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Ci::DailyBuildGroupReportResultsFinder | `app/finders/ci/daily_build_group_report_results_finder.rb` | 検索ロジック |

**主要処理フロー**:
- **23-24行目**: MAX_ITEMS = 1000, REPORT_WINDOW = 90日定義
- **34-39行目**: executeメソッド - 権限チェックとフィルタ適用
- **47-54行目**: filter_report_resultsメソッド - フィルタチェーン
- **82-91行目**: start_date計算 - 90日制限ロジック

### プログラム呼び出し階層図

```
Projects::Ci::DailyBuildGroupReportResultsController#index
    │
    ├─ validate_param_type!
    │      └─ PARAM_TYPES確認
    │
    └─ report_results
           │
           └─ Ci::DailyBuildGroupReportResultsFinder.new(params, current_user).execute
                  │
                  ├─ query_allowed? (権限チェック)
                  │
                  └─ filter_report_results
                         ├─ by_coverage
                         ├─ by_ref_path
                         ├─ by_dates
                         ├─ sort
                         └─ limit_by (MAX_ITEMS)
```

### データフロー図

```
[入力]                    [処理]                              [出力]

params ─────────────────▶ Controller ──────────────────────▶ CSV/JSON
                               │
                         ┌─────┴─────┐
                         │           │
                   validate      report_results
                         │           │
                         │     Finder.execute
                         │           │
                         │     filter_report_results
                         │           │
                         │     ┌─────┼─────┐
                         │     │     │     │
                         │  by_ref by_dates limit
                         │     │     │     │
                         │     └─────┴─────┘
                         │           │
                         └─────┬─────┘
                               │
                         ┌─────┴─────┐
                         │           │
                   CsvBuilder    Serializer
                         │           │
                         ▼           ▼
                       CSV         JSON
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Projects::Ci::DailyBuildGroupReportResultsController | `app/controllers/projects/ci/daily_build_group_report_results_controller.rb` | コントローラー | エントリーポイント |
| Ci::DailyBuildGroupReportResultsFinder | `app/finders/ci/daily_build_group_report_results_finder.rb` | ファインダー | 検索ロジック |
| Ci::DailyBuildGroupReportResult | `app/models/ci/daily_build_group_report_result.rb` | モデル | レポート結果データ |
| Ci::DailyBuildGroupReportResultSerializer | `app/serializers/ci/daily_build_group_report_result_serializer.rb` | シリアライザー | JSON変換 |
| CsvBuilder | `lib/gitlab/csv_builder.rb` | ライブラリ | CSV生成 |
