# 機能設計書：50-CI変数管理

## 1. 機能概要

### 1.1 機能の目的
CI変数管理機能は、CI/CDパイプラインで使用する環境変数を管理する機能である。プロジェクトレベルとグループレベルで変数を定義でき、マスク、保護、暗号化などのセキュリティ機能を提供する。環境スコープによる変数の分離も可能で、本番・ステージング環境ごとに異なる値を設定できる。

### 1.2 関連画面
| 画面ID | 画面名 | パス |
|--------|--------|------|
| SCR-PROJECT-CICD | プロジェクトCI/CD設定 | `/:namespace/:project/-/settings/ci_cd` |
| SCR-GROUP-CICD | グループCI/CD設定 | `/groups/:group/-/settings/ci_cd` |

### 1.3 関連API
| APIエンドポイント | メソッド | 概要 |
|-------------------|----------|------|
| `/projects/:id/variables` | GET | プロジェクト変数一覧取得 |
| `/projects/:id/variables` | POST | プロジェクト変数作成 |
| `/projects/:id/variables/:key` | GET | プロジェクト変数詳細取得 |
| `/projects/:id/variables/:key` | PUT | プロジェクト変数更新 |
| `/projects/:id/variables/:key` | DELETE | プロジェクト変数削除 |
| `/groups/:id/variables` | GET | グループ変数一覧取得 |
| `/groups/:id/variables` | POST | グループ変数作成 |
| `/groups/:id/variables/:key` | GET | グループ変数詳細取得 |
| `/groups/:id/variables/:key` | PUT | グループ変数更新 |
| `/groups/:id/variables/:key` | DELETE | グループ変数削除 |

## 2. データモデル

### 2.1 主要エンティティ

#### Ci::Variable（プロジェクト変数）
```
ci_variables テーブル
├── id: bigint (PK)
├── project_id: bigint (FK)
├── key: varchar(255)
├── value: text (encrypted)
├── encrypted_value: text
├── encrypted_value_salt: varchar
├── encrypted_value_iv: varchar
├── variable_type: smallint
├── protected: boolean
├── masked: boolean
├── raw: boolean
├── hidden: boolean
├── environment_scope: varchar
├── description: varchar(255)
├── created_at: timestamp
└── updated_at: timestamp

制限:
- limit_name = 'project_ci_variables'
- limit_scope = :project
```

#### Ci::GroupVariable（グループ変数）
```
ci_group_variables テーブル
├── id: bigint (PK)
├── group_id: bigint (FK)
├── key: varchar(255)
├── value: text (encrypted)
├── encrypted_value: text
├── encrypted_value_salt: varchar
├── encrypted_value_iv: varchar
├── variable_type: smallint
├── protected: boolean
├── masked: boolean
├── raw: boolean
├── hidden: boolean
├── environment_scope: varchar
├── description: varchar(255)
├── created_at: timestamp
└── updated_at: timestamp

制限:
- limit_name = 'group_ci_variables'
- limit_scope = :group
```

#### variable_type enum
```
- env_var: 1（環境変数）
- file: 2（ファイル変数）
```

### 2.2 共通モジュール

#### Ci::HasVariable
```ruby
# 共通機能:
- 暗号化（attr_encrypted）
- キー検証（英数字とアンダースコアのみ、最大255文字）
- ソート（order_key_asc, order_key_desc）
- to_hash_variable（ジョブ実行時の変換）
```

#### Ci::Maskable
```ruby
# マスク機能:
REGEX = /\A[a-zA-Z0-9_+=/@:.~-]{8,}\z/  # 通常
MASK_AND_RAW_REGEX = /\A\S{8,}\z/        # raw時

# 条件:
- 8文字以上
- 単一行
- 空白なし
- Base64アルファベット + 特殊文字（通常時）
```

#### Ci::RawVariable
```ruby
# raw変数:
# $VAR展開を行わない変数
```

#### Ci::HidableVariable
```ruby
# 隠し変数:
# hidden: true で非表示
```

### 2.3 ER図
```
┌─────────────────────┐      ┌──────────────────┐
│      Project        │      │      Group       │
├─────────────────────┤      ├──────────────────┤
│ id                  │      │ id               │
│ ...                 │      │ ...              │
└─────────────────────┘      └──────────────────┘
         │                            │
         │1                           │1
         ▼ *                          ▼ *
┌─────────────────────┐      ┌──────────────────────┐
│   Ci::Variable      │      │  Ci::GroupVariable   │
├─────────────────────┤      ├──────────────────────┤
│ id                  │      │ id                   │
│ project_id          │      │ group_id             │
│ key, value(enc)     │      │ key, value(enc)      │
│ variable_type       │      │ variable_type        │
│ protected, masked   │      │ protected, masked    │
│ raw, hidden         │      │ raw, hidden          │
│ environment_scope   │      │ environment_scope    │
│ description         │      │ description          │
└─────────────────────┘      └──────────────────────┘
```

## 3. 処理フロー

### 3.1 変数作成フロー
```
[ユーザー]
     │
     │ 変数作成リクエスト
     ▼
[VariablesController / CI/CD Settings]
     │
     ├── 1. 権限チェック
     │   ├── admin_cicd_variables（プロジェクト）
     │   └── admin_group（グループ）
     │
     ├── 2. パラメータ検証
     │   ├── キー形式チェック
     │   ├── ユニーク制約チェック
     │   ├── マスク条件チェック
     │   └── 制限数チェック
     │
     ├── 3. 暗号化
     │   └── attr_encrypted（AES-256-CBC）
     │
     └── 4. 保存
         └── Ci::ChangeVariablesService
```

### 3.2 変数利用フロー（パイプライン実行時）
```
[Ci::CreatePipelineService]
     │
     ▼
[Ci::Build#scoped_variables]
     │
     ├── プロジェクト変数
     │   └── Ci::Variable.for_ref(ref)
     │
     ├── グループ変数（継承）
     │   └── project.group.self_and_ancestors
     │       └── Ci::GroupVariable.for_ref(ref)
     │
     ├── 環境スコープマッチング
     │   └── for_environment(environment_name)
     │
     └── 保護ブランチチェック
         └── protected == true の場合
             └── protected ref のみ使用可能
```

### 3.3 マスク処理フロー
```
[Ci::Build実行]
     │
     ▼
[ジョブログ出力]
     │
     ├── マスク対象変数の収集
     │   └── masked? == true
     │
     └── ログのマスク処理
         └── 変数値を [MASKED] に置換
```

## 4. ビジネスルール

### 4.1 変数キーの制約
| ルールID | ルール | 実装箇所 |
|----------|--------|----------|
| CV-R01 | キーは必須 | validates :key, presence: true |
| CV-R02 | キーは最大255文字 | validates :key, length: { maximum: 255 } |
| CV-R03 | キーは英数字とアンダースコアのみ | format: { with: /\A[a-zA-Z0-9_]+\z/ } |
| CV-R04 | キーはスコープ内でユニーク | uniqueness: { scope: [:project_id, :environment_scope] } |

### 4.2 マスク条件
| ルールID | ルール | 実装箇所 |
|----------|--------|----------|
| CV-R05 | 8文字以上 | REGEX, MASK_AND_RAW_REGEX |
| CV-R06 | 単一行 | REGEX, MASK_AND_RAW_REGEX |
| CV-R07 | 空白なし | REGEX, MASK_AND_RAW_REGEX |
| CV-R08 | Base64 + 特殊文字（非raw時） | REGEX |

### 4.3 保護変数
| ルールID | ルール | 実装箇所 |
|----------|--------|----------|
| CV-R09 | protected=trueは保護ブランチ/タグのみ | for_ref フィルタリング |
| CV-R10 | 非保護ブランチでは使用不可 | pipeline変数構築時 |

### 4.4 権限
| 操作 | プロジェクト変数 | グループ変数 |
|------|------------------|--------------|
| 表示 | admin_cicd_variables | admin_group |
| 作成 | admin_cicd_variables | admin_group |
| 更新 | admin_cicd_variables | admin_cicd_variables |
| 削除 | admin_cicd_variables | admin_group |

### 4.5 環境スコープ
```ruby
# 環境名とスコープのマッチング
# scope: "production" -> env: "production" (マッチ)
# scope: "prod*"      -> env: "production" (マッチ)
# scope: "*"          -> すべての環境 (マッチ)
```

### 4.6 変数の優先順位
```
1. ジョブ変数（.gitlab-ci.yml内）
2. パイプライン変数（手動実行時）
3. プロジェクト変数
4. グループ変数（親グループ順）
5. インスタンス変数
6. 定義済み変数
```

## 5. 暗号化

### 5.1 暗号化方式
```ruby
attr_encrypted :value,
  mode: :per_attribute_iv_and_salt,
  insecure_mode: true,
  key: :db_key_base,
  algorithm: 'aes-256-cbc'
```

### 5.2 保存カラム
- `encrypted_value`: 暗号化された値
- `encrypted_value_salt`: ソルト
- `encrypted_value_iv`: 初期化ベクトル

## 6. コードリーディングガイド

### 6.1 読み解く順序
1. **Concern層**: Ci::HasVariable, Ci::Maskable, Ci::RawVariable
2. **モデル層**: Ci::Variable, Ci::GroupVariable
3. **コントローラー層**: Projects::Settings::CiCdController, Groups::VariablesController
4. **サービス層**: Ci::ChangeVariablesService

### 6.2 プログラム呼び出し階層図
```
Projects::Settings::CiCdController#show
├── define_ci_variables
│   └── project.variables.order_key_asc
├── @variable_limit
│   └── Plan.default.actual_limits.project_ci_variables
└── audit_project_cicd_settings_access

Groups::VariablesController#update
├── Ci::ChangeVariablesService
│   └── container.update(params)
└── Ci::GroupVariableSerializer

Ci::Build#scoped_variables
├── Ci::Variable.for_ref(ref)
├── Ci::GroupVariable.for_ref(ref)
├── HasEnvironmentScope
│   └── for_environment(environment_name)
└── to_hash_variable
    └── { key:, value:, public:, file:, masked: }
```

### 6.3 データフロー図
```
[設定画面]
     │
     │ key, value, protected, masked, raw
     ▼
[Ci::Variable / Ci::GroupVariable]
     │
     ├── 検証
     │   ├── キー形式
     │   ├── ユニーク制約
     │   └── マスク条件
     │
     ├── 暗号化
     │   └── attr_encrypted (AES-256-CBC)
     │
     └── 保存
         └── ci_variables / ci_group_variables
     │
     ▼
[パイプライン作成時]
     │
     ├── 環境スコープマッチング
     ├── 保護ブランチチェック
     └── to_hash_variable
         │
         ▼
[Ci::Build実行]
     │
     ├── 環境変数として注入
     │   └── env_var: 環境変数
     │   └── file: ファイルとして書き込み
     │
     └── ログマスク
         └── masked: true の値を [MASKED] に置換
```

### 6.4 関連ファイル一覧
| ファイルパス | 責務 |
|--------------|------|
| `app/models/ci/variable.rb` | プロジェクト変数モデル |
| `app/models/ci/group_variable.rb` | グループ変数モデル |
| `app/models/concerns/ci/has_variable.rb` | 変数共通機能 |
| `app/models/concerns/ci/maskable.rb` | マスク機能 |
| `app/models/concerns/ci/raw_variable.rb` | raw変数機能 |
| `app/models/concerns/ci/hidable_variable.rb` | 隠し変数機能 |
| `app/models/concerns/has_environment_scope.rb` | 環境スコープ |
| `app/controllers/projects/settings/ci_cd_controller.rb` | プロジェクト設定 |
| `app/controllers/groups/variables_controller.rb` | グループ変数コントローラー |
| `app/services/ci/change_variables_service.rb` | 変数変更サービス |
| `app/serializers/ci/variable_serializer.rb` | 変数シリアライザー |
| `app/serializers/ci/group_variable_serializer.rb` | グループ変数シリアライザー |

## 7. 拡張ポイント

### 7.1 EE拡張
- **インスタンス変数**: 管理者が設定するインスタンス全体の変数
- **外部シークレット管理**: Vault、AWS Secrets Manager等との連携
- **変数監査ログ**: 変数へのアクセス・変更の監査

### 7.2 新規変数タイプ追加
1. `Ci::HasVariable` のenum追加
2. `to_hash_variable` で適切な変換
3. Runner側での対応

### 7.3 新規スコープ追加
1. `HasEnvironmentScope` の拡張
2. マッチングロジックの実装
3. UIでの選択肢追加
