# 機能設計書 55-環境管理

## 概要

本ドキュメントは、GitLabのデプロイ先環境（development、staging、production等）の管理機能について記述した機能設計書である。

### 本機能の処理概要

環境管理機能は、アプリケーションをデプロイする先の環境を定義・管理し、各環境へのデプロイ状態や履歴を追跡するための機能である。

**業務上の目的・背景**：ソフトウェア開発では、開発・テスト・ステージング・本番など複数の環境を使い分けてアプリケーションをデプロイする。本機能により、各環境の状態を一元管理し、どのバージョンがどの環境にデプロイされているかを可視化できる。また、環境ごとの自動停止設定やKubernetesクラスターとの連携も可能となる。

**機能の利用シーン**：
- 新しいデプロイ環境を定義する場合
- 環境の状態（利用可能/停止中）を確認する場合
- レビューアプリ用の動的環境を管理する場合
- 環境を手動で停止する場合
- Kubernetes上の環境を監視する場合
- 環境の外部URLを設定する場合

**主要な処理内容**：
1. 環境の作成
2. 環境の編集（外部URL等）
3. 環境の停止（stop）
4. 自動停止のキャンセル
5. 環境一覧・フォルダ表示
6. ターミナルアクセス（Kubernetes環境）
7. Kubernetes k8sビュー

**関連システム・外部連携**：
- Kubernetesクラスター
- GitLab Agent for Kubernetes
- 外部URL（デプロイ先アプリケーション）
- Prometheus（メトリクス）

**権限による制御**：
- 環境閲覧: `read_environment`権限が必要
- 環境作成: `create_environment`権限が必要
- 環境更新: `update_environment`権限が必要
- 環境停止: `stop_environment`権限が必要
- ターミナルアクセス: `admin_environment`権限が必要

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 77 | 環境一覧 | 主画面 | デプロイ環境の一覧表示 |
| 78 | 環境詳細 | 主画面 | 環境の詳細表示 |
| 79 | 環境新規作成 | 主画面 | 新規環境の作成 |
| 80 | 環境編集 | 主画面 | 環境の編集 |

## 機能種別

CRUD操作 / 状態管理 / 外部連携

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | String | Yes | 環境名 | 最大255文字、正規表現チェック |
| external_url | String | No | 外部URL | 最大255文字、URL形式チェック |
| tier | String | No | 環境層（production/staging等） | enum値 |

### 入力データソース

- 画面入力（環境設定フォーム）
- .gitlab-ci.yml（environment定義）
- APIリクエスト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| id | Integer | 環境ID |
| name | String | 環境名 |
| slug | String | URL用スラッグ |
| external_url | String | 外部URL |
| state | String | 状態（available/stopping/stopped） |
| tier | String | 環境層 |
| last_deployment | Object | 最新デプロイ情報 |
| auto_stop_at | DateTime | 自動停止予定日時 |

### 出力先

- 画面表示（環境一覧、詳細）
- JSONレスポンス（API、ポーリング）
- リダイレクト

## 処理フロー

### 処理シーケンス

```
1. 認証・権限チェック
   └─ read_environment等の権限確認

2. 環境データ取得
   └─ 検索条件に応じた環境一覧/詳細取得

3. 操作実行（作成/更新/停止等）
   └─ 該当するサービスクラスを呼び出し

4. Kubernetesデータ取得（該当時）
   └─ クラスターからPod情報等を取得

5. レスポンス生成
   └─ HTML/JSONレスポンス
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{権限チェック}
    B -->|権限なし| C[アクセス拒否]
    B -->|権限あり| D{アクション種別}
    D -->|一覧| E[環境一覧取得]
    D -->|作成| F[環境作成]
    D -->|更新| G[環境更新]
    D -->|停止| H[StopService実行]
    E --> I[レスポンス生成]
    F --> I
    G --> I
    H --> J{stop_action存在?}
    J -->|Yes| K[stop_action実行]
    J -->|No| L[状態をstoppedに変更]
    K --> I
    L --> I
    I --> M[終了]
    C --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-55-01 | 環境名一意性 | 環境名はプロジェクト内で一意 | 作成/更新時 |
| BR-55-02 | スラッグ自動生成 | slugは環境名から自動生成（最大24文字） | 作成時 |
| BR-55-03 | 環境層自動推定 | tierは環境名から自動推定（prod, staging等） | 作成時 |
| BR-55-04 | 自動停止 | auto_stop_atに達した環境は自動停止 | バックグラウンドジョブ |
| BR-55-05 | フォルダ表示 | 環境名に/が含まれる場合はフォルダ化 | 一覧表示時 |
| BR-55-06 | 停止待機 | stop_actionがある場合はstoppingを経てstopped | 停止操作時 |

### 計算ロジック

- 環境層推定ロジック（guess_tier）:
  - `/(dev|review|trunk)/i` → development
  - `/(test|tst|int|ac(ce|)pt|qa|qc|control|quality)/i` → testing
  - `/(st(a|)g|mod(e|)l|pre|demo|non)/i` → staging
  - `/(pr(o|)d|live)/i` → production
  - それ以外 → other

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 環境作成 | environments | INSERT | 新規環境レコード作成 |
| 環境更新 | environments | UPDATE | external_url等の更新 |
| 環境停止 | environments | UPDATE | stateの更新 |
| デプロイ連携 | deployments | SELECT | 最新デプロイ情報取得 |

### テーブル別操作詳細

#### environments

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | name | ユーザー入力値 | 一意制約あり |
| INSERT | slug | 自動生成 | 最大24文字 |
| INSERT | tier | 自動推定または指定値 | enum |
| UPDATE | state | available/stopping/stopped | state_machine管理 |
| UPDATE | auto_stop_at | null（キャンセル時） | - |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 403 | Forbidden | 権限不足 | 必要な権限を取得 |
| 404 | Not Found | 環境未検出 | 正しい環境IDを確認 |
| 400 | Bad Request | バリデーションエラー | 入力値を修正 |

### リトライ仕様

- 停止失敗時は手動で再実行
- 自動停止失敗時はAutoStopServiceが再試行

## トランザクション仕様

- 環境作成/更新は単一トランザクション
- 停止操作時のstop_action実行は非同期（ジョブとして実行）

## パフォーマンス要件

- 環境一覧取得: 3秒以内（ポーリング間隔考慮）
- 環境作成/更新: 2秒以内
- Kubernetesデータ取得: ReactiveCacheで55秒キャッシュ

## セキュリティ考慮事項

- プロジェクト単位での権限チェック
- 外部URLのスキームチェック（unsafe protocolを拒否）
- ターミナルアクセスの管理者権限制限
- Kas Cookie設定（Kubernetes認証）

## 備考

- LONG_STOP: stopping状態が1週間以上続くと異常とみなす
- ポーリング間隔: 3秒
- ReactiveCacheリフレッシュ間隔: 1分

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | environment.rb | `app/models/environment.rb` | 環境の属性、関連、状態遷移を理解する |

**読解のコツ**:
- **12行目**: `LONG_STOP = 1.week` で長期停止の閾値を確認
- **197-203行目**: `enum :tier` で環境層の種類を確認
- **210-239行目**: `state_machine :state` で状態遷移を確認
- **614-627行目**: `guess_tier`メソッドで環境層自動推定ロジックを確認

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | environments_controller.rb | `app/controllers/projects/environments_controller.rb` | 各アクションの処理フローを確認 |

**主要処理フロー**:
- **36-57行目**: indexアクションで環境一覧取得（HTML/JSON対応）
- **62-82行目**: folderアクションでフォルダ内環境一覧取得
- **97-104行目**: createアクションで環境作成
- **107-112行目**: updateアクションで環境更新
- **115-133行目**: stopアクションでStopService呼び出し
- **136-154行目**: cancel_auto_stopアクションで自動停止キャンセル

#### Step 3: サービス層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stop_service.rb | `app/services/environments/stop_service.rb` | 環境停止のビジネスロジック |

**主要処理フロー**:
- **7-16行目**: executeメソッドで権限チェックと停止処理
- **22-40行目**: unsafe_execute!で実際の停止処理
- **43-48行目**: execute_for_branchでブランチに紐づく環境を停止

### プログラム呼び出し階層図

```
Projects::EnvironmentsController
    │
    ├─ #index
    │      └─ Environments::EnvironmentsFinder#execute
    │
    ├─ #create
    │      └─ project.environments.create
    │
    ├─ #update
    │      └─ environment.update
    │
    └─ #stop
           └─ Environments::StopService#execute
                  ├─ environment.stop_with_actions!
                  └─ Environments::DeleteManagedResourcesService
```

### データフロー図

```
[入力]               [処理]                        [出力]

環境設定
フォーム    ───▶ EnvironmentsController ───▶ 環境一覧/詳細画面
                      │
                      ▼
              Environment (Model)
                      │
        ┌─────────────┼─────────────┐
        ▼             ▼             ▼
  environments   deployments   Kubernetes
   テーブル       テーブル       クラスター
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| environment.rb | `app/models/environment.rb` | モデル | 環境のデータモデル |
| environments_controller.rb | `app/controllers/projects/environments_controller.rb` | コントローラー | 環境操作のエンドポイント |
| stop_service.rb | `app/services/environments/stop_service.rb` | サービス | 環境停止ロジック |
| auto_stop_service.rb | `app/services/environments/auto_stop_service.rb` | サービス | 自動停止処理 |
| environments_finder.rb | `app/finders/environments/environments_finder.rb` | Finder | 環境検索ロジック |
| environment_serializer.rb | `app/serializers/environment_serializer.rb` | シリアライザー | JSON出力フォーマット |
