# 機能設計書 75-リリースエビデンス

## 概要

本ドキュメントは、GitLabのリリースエビデンス機能について、その処理概要、入出力仕様、処理フロー、データベース操作仕様を定義する。

### 本機能の処理概要

リリースエビデンス機能は、リリース時点のプロジェクト状態を記録・保存するスナップショット機能である。エビデンスには、リリース情報、関連マイルストーン、プロジェクト情報などがJSON形式で記録され、改ざん検知のためのSHA256ハッシュが付与される。

**業務上の目的・背景**：ソフトウェアリリースにおいて、リリース時点でのプロジェクト状態を証跡として残すことは、監査やコンプライアンス要件を満たす上で重要である。リリースエビデンスは、リリース時点のマイルストーン、タグ、パイプライン情報などを不変の記録として保存し、後からリリース時の状態を確認・検証できるようにする。

**機能の利用シーン**：
- 監査時にリリースの証跡を提示する場合
- リリース時点のプロジェクト状態を後から確認する場合
- コンプライアンス要件でリリース記録が必要な場合
- セキュリティ監査でリリースの完全性を検証する場合

**主要な処理内容**：
1. リリース作成時のエビデンス自動生成（非同期）
2. エビデンスサマリーの収集（リリース情報、マイルストーン、プロジェクト情報）
3. SHA256ハッシュによる改ざん検知用署名の生成
4. エビデンスの閲覧（JSON形式）

**関連システム・外部連携**：
- リリース機能（親エンティティ）
- CI/CDパイプライン（パイプライン情報の記録）
- マイルストーン（関連マイルストーン情報の記録）

**権限による制御**：
- エビデンスの閲覧: リリースエビデンス閲覧権限
- エビデンスは自動生成されるため、手動での作成/編集/削除は不可

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 87 | リリース詳細 | 参照画面 | リリースエビデンスの表示 |

## 機能種別

データ収集 / スナップショット / 監査ログ

## 入力仕様

### 入力パラメータ

エビデンスは自動生成されるため、ユーザーからの入力はない。

内部的に以下のデータが収集される：

| データ項目 | 型 | 説明 |
|-----------|-----|------|
| release_id | Integer | 対象リリースID |
| pipeline_id | Integer | 関連パイプラインID（任意） |

### 入力データソース

- リリース作成処理（Releases::CreateService）
- CI/CDパイプライン情報

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| id | Integer | エビデンスID |
| release_id | Integer | 対象リリースID |
| summary | JSON | エビデンスサマリー |
| summary_sha | String | サマリーのSHA256ハッシュ |
| collected_at | DateTime | 収集日時（created_atのエイリアス） |

#### summary構造

```json
{
  "release": {
    "id": 1,
    "tag": "v1.0.0",
    "name": "Release 1.0.0",
    "description": "...",
    "created_at": "...",
    "project": {
      "id": 1,
      "name": "...",
      "description": "..."
    },
    "milestones": [
      {
        "id": 1,
        "title": "...",
        "state": "..."
      }
    ]
  }
}
```

### 出力先

- JSON API レスポンス
- 画面表示

## 処理フロー

### 処理シーケンス

```
1. リリース作成検知
   └─ CreateServiceからのトリガー
2. エビデンス生成条件チェック
   └─ historical/upcomingリリースでないことを確認
3. 非同期ワーカー起動
   └─ CreateEvidenceWorkerをキューに投入
4. サマリー収集
   └─ EvidenceSerializerでデータを収集
5. ハッシュ生成
   └─ サマリーのSHA256ハッシュを計算
6. DB保存
   └─ release_evidencesテーブルへの保存
```

### フローチャート

```mermaid
flowchart TD
    A[リリース作成] --> B{historical?}
    B -->|Yes| C[エビデンス生成スキップ]
    B -->|No| D{upcoming?}
    D -->|Yes| C
    D -->|No| E[CreateEvidenceWorker.perform_async]
    E --> F[EvidenceSerializer.represent]
    F --> G[summary収集]
    G --> H[SHA256ハッシュ計算]
    H --> I[evidence.save!]
    I --> J[完了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-75-01 | 自動生成 | エビデンスはリリース作成時に自動生成される | リリース作成時 |
| BR-75-02 | historical除外 | 過去日付のリリース（historical）にはエビデンスを生成しない | released_at < created_at |
| BR-75-03 | upcoming除外 | 将来日付のリリース（upcoming）にはエビデンスを生成しない | released_at > 現在時刻 |
| BR-75-04 | issues除外 | サマリーからissues情報は除外される（機密保護） | サマリー取得時 |
| BR-75-05 | 不変性 | 生成されたエビデンスは編集不可 | 常時 |

### 計算ロジック

- `summary_sha`: `Gitlab::CryptoHelper.sha256(summary_json)`

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| エビデンス作成 | release_evidences | INSERT | エビデンスレコード作成 |
| エビデンス取得 | release_evidences | SELECT | エビデンス取得 |

### テーブル別操作詳細

#### release_evidences

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | release_id | リリースID | 必須、外部キー |
| INSERT | summary | シリアライズされたJSON | エビデンスデータ |
| INSERT | summary_sha | SHA256ハッシュ | 改ざん検知用 |
| INSERT | created_at | 現在日時 | 収集日時として使用 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 403 | 権限エラー | エビデンス閲覧権限なし | エラーメッセージ表示 |
| 404 | NotFound | 存在しないエビデンスへのアクセス | 404ページ表示 |

### リトライ仕様

- エビデンス生成は非同期ワーカーで実行され、失敗時は自動リトライ

## トランザクション仕様

- エビデンス作成は非同期のため、リリース作成とは別トランザクション

## パフォーマンス要件

- エビデンス生成は非同期処理のためリリース作成に影響しない
- エビデンス取得は即時

## セキュリティ考慮事項

- issues情報はサマリーから除外（機密イシューの保護）
- SHA256ハッシュによる改ざん検知
- 権限ベースのアクセス制御

## 備考

- エビデンスはリリースごとに複数存在する可能性がある（再生成時など）
- `collected_at`は`created_at`のエイリアス
- `sha`は`summary_sha`のエイリアス

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | evidence.rb | `app/models/releases/evidence.rb` | Evidenceモデルの属性、関連定義を確認 |

**読解のコツ**:
- **8行目**: `belongs_to :release`で親リレーションを確認
- **12-14行目**: `sha_attribute :summary_sha`とエイリアス定義を確認
- **25-33行目**: `summary`メソッドでissues除外処理を確認

#### Step 2: サービス層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | create_evidence_service.rb | `app/services/releases/create_evidence_service.rb` | エビデンス生成のビジネスロジック |

**主要処理フロー**:
1. **5-8行目**: コンストラクタでリリースとパイプラインを受け取る
2. **10-18行目**: `execute`メソッドでエビデンス生成
3. **13行目**: EvidenceSerializerでサマリー収集
4. **16行目**: SHA256ハッシュ計算

#### Step 3: 呼び出し元を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | create_service.rb | `app/services/releases/create_service.rb` | エビデンス生成のトリガー |

**主要処理フロー（create_service.rb）**:
- **107-111行目**: `create_evidence!`メソッドでエビデンス生成条件判定
- **108行目**: historical/upcomingリリースの除外
- **110行目**: CreateEvidenceWorkerへの非同期ジョブ投入

### プログラム呼び出し階層図

```
Releases::CreateService
    │
    └─ create_evidence!
           │
           ├─ release.historical_release? チェック
           ├─ release.upcoming_release? チェック
           │
           └─ Releases::CreateEvidenceWorker.perform_async
                  │
                  └─ Releases::CreateEvidenceService
                         │
                         ├─ EvidenceSerializer.represent
                         │
                         ├─ Gitlab::CryptoHelper.sha256
                         │
                         └─ Evidence.save!
```

### データフロー図

```
[入力]               [処理]                    [出力]

リリース作成 ───▶ CreateService ───▶ リリースレコード
                       │
                       └─▶ CreateEvidenceWorker
                                  │
                                  └─▶ CreateEvidenceService
                                             │
                                             ├─▶ EvidenceSerializer
                                             │         │
                                             │         └─▶ summary JSON
                                             │
                                             ├─▶ CryptoHelper.sha256
                                             │         │
                                             │         └─▶ summary_sha
                                             │
                                             └─▶ Evidence.save!
                                                       │
                                                       └─▶ release_evidences
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| evidence.rb | `app/models/releases/evidence.rb` | モデル | エビデンスのデータ構造定義 |
| create_evidence_service.rb | `app/services/releases/create_evidence_service.rb` | サービス | エビデンス生成処理 |
| create_service.rb | `app/services/releases/create_service.rb` | サービス | リリース作成（エビデンス生成トリガー） |
| evidences_controller.rb | `app/controllers/projects/releases/evidences_controller.rb` | コントローラー | エビデンス閲覧API |
| evidence_serializer.rb | `app/serializers/evidences/evidence_serializer.rb` | シリアライザー | エビデンスJSON形式定義 |
| create_evidence_worker.rb | `app/workers/releases/create_evidence_worker.rb` | ワーカー | 非同期エビデンス生成 |
