# 機能設計書 88-OAuth2プロバイダ

## 概要

本ドキュメントは、GitLabにおけるOAuth2プロバイダ機能の設計を記述する。GitLabはOAuth2認可サーバーとして機能し、サードパーティアプリケーションがGitLab APIにアクセスするための認可を提供する。Doorkeeperを基盤として実装されている。

### 本機能の処理概要

**業務上の目的・背景**：外部アプリケーションやサービスがGitLabのリソース（リポジトリ、イシュー、マージリクエストなど）にアクセスする際、ユーザーの認証情報を直接共有することなく、安全なアクセス許可を実現する必要がある。OAuth2プロバイダ機能により、ユーザーは自分のデータへのアクセスを制御しながら、外部アプリケーションとの連携が可能になる。

**機能の利用シーン**：
- CI/CDツールがGitLabリポジトリにアクセスする場合
- IDE拡張機能がGitLab APIを利用する場合
- サードパーティサービスがGitLabデータを取得する場合
- カスタムアプリケーションからGitLab APIを呼び出す場合
- MCP（Model Context Protocol）サーバーとの連携

**主要な処理内容**：
1. OAuthアプリケーションの登録・管理
2. 認可リクエストの処理（Authorization Code Grant）
3. アクセストークンの発行
4. スコープベースの権限制御
5. トークンの更新・失効

**関連システム・外部連携**：
- Doorkeeper gem（OAuth2サーバー実装）
- 監査ログシステム
- 組織管理システム

**権限による制御**：
- アプリケーション作成はuser_oauth_applications設定に依存
- 管理者向けスコープは設定で制限可能
- PKCEがダイナミックアプリケーションで必須

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 117 | OAuth2アプリケーション一覧 | 主画面 | アプリケーション管理 |
| 118 | OAuth2認可画面 | 主画面 | 認可同意 |

## 機能種別

認証・認可 / CRUD操作

## 入力仕様

### 入力パラメータ（アプリケーション作成）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | String | Yes | アプリケーション名 | 必須 |
| redirect_uri | String | Yes | リダイレクトURI | 有効なURI形式 |
| scopes | Array | Yes | 許可スコープ | 有効なスコープ |
| confidential | Boolean | No | 機密アプリケーション | - |

### 入力パラメータ（認可リクエスト）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| client_id | String | Yes | アプリケーションID | 有効なID |
| redirect_uri | String | Yes | リダイレクトURI | 登録URIと一致 |
| response_type | String | Yes | レスポンスタイプ | code |
| scope | String | No | 要求スコープ | 有効なスコープ |
| state | String | No | CSRF防止用状態値 | - |
| code_challenge | String | Conditional | PKCEチャレンジ | ダイナミックアプリで必須 |
| code_challenge_method | String | No | PKCEメソッド | S256 |

### 入力データソース

- 画面入力
- API経由

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| access_token | String | アクセストークン |
| token_type | String | トークンタイプ（Bearer） |
| expires_in | Integer | 有効期限（秒） |
| refresh_token | String | リフレッシュトークン |
| scope | String | 付与されたスコープ |
| created_at | Integer | 作成日時（Unix timestamp） |

### 出力先

- APIレスポンス
- リダイレクト（認可コード）

## 処理フロー

### 処理シーケンス

```
1. アプリケーション登録
   └─ Applications::CreateService#execute
2. 認可リクエスト受信
   └─ Oauth::AuthorizationsController#new
3. 認可可能性チェック
   └─ pre_auth.authorizable?
4. ユーザー同意
   └─ 認可画面表示
5. 認可コード発行
   └─ authorization.authorize
6. トークン交換
   └─ Doorkeeper::TokensController
7. 監査ログ記録
   └─ audit_oauth_authorization
```

### フローチャート

```mermaid
flowchart TD
    A[認可リクエスト] --> B{メール確認済み?}
    B -->|No| C[エラー: unconfirmed_email]
    B -->|Yes| D{管理者制限?}
    D -->|Yes| E[forbidden画面]
    D -->|No| F{ダイナミックアプリ?}
    F -->|Yes| G{PKCE必須チェック}
    G -->|No PKCE| H[エラー: pkce_required]
    G -->|OK| I{認可可能?}
    F -->|No| I
    I -->|No| J[エラー画面]
    I -->|Yes| K{既存トークンあり?}
    K -->|Yes| L[自動承認]
    K -->|No| M[認可画面表示]
    L --> N[認可コード発行]
    M --> O{ユーザー承認?}
    O -->|Yes| N
    O -->|No| P[認可拒否]
    N --> Q[リダイレクト]
    Q --> R[トークン交換]
    R --> S[アクセストークン発行]
    S --> T[監査ログ記録]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-88-01 | メール確認必須 | 認可前にメール確認が必要 | 常時 |
| BR-88-02 | スコープダウングレード | ログイン用途ではread_userに制限 | gl_auth_type=login時 |
| BR-88-03 | 管理者スコープ制限 | 管理者は危険なスコープを持つ非信頼アプリへの認可制限可能 | disable_admin_oauth_scopes有効時 |
| BR-88-04 | PKCE必須 | ダイナミックアプリケーションはPKCE（S256）が必須 | ダイナミックアプリ |
| BR-88-05 | トークン有効期限必須 | アクセストークンには必ず有効期限が設定される | 常時 |
| BR-88-06 | 機密アプリ自動承認 | 機密アプリで既存トークンがあれば自動承認 | matching_token? && confidential |

### 計算ロジック

**スコープ検証**：`doorkeeper_application.includes_scope?`メソッドで要求スコープがアプリケーションに許可されているか確認。

**危険なスコープ判定**：API、READ_API、ADMIN、REPOSITORY、REGISTRYスコープを含む非信頼アプリは管理者に危険と判定。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| アプリ作成 | oauth_applications | INSERT | アプリケーション登録 |
| 認可 | oauth_access_grants | INSERT | 認可コード発行 |
| トークン発行 | oauth_access_tokens | INSERT | アクセストークン生成 |
| トークン更新 | oauth_access_tokens | UPDATE/INSERT | リフレッシュ |
| シークレット更新 | oauth_applications | UPDATE | renew_secret |

### テーブル別操作詳細

#### oauth_applications

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | name | アプリケーション名 | |
| INSERT | uid | クライアントID（自動生成） | |
| INSERT | secret | クライアントシークレット（暗号化） | |
| INSERT | redirect_uri | リダイレクトURI | |
| INSERT | scopes | 許可スコープ | |
| INSERT | owner_id | 作成者ID | |
| INSERT | organization_id | 組織ID | |
| UPDATE | secret | 新しいシークレット | renew時 |

#### oauth_access_tokens

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | token | アクセストークン（ハッシュ化） | |
| INSERT | refresh_token | リフレッシュトークン | |
| INSERT | resource_owner_id | ユーザーID | |
| INSERT | application_id | アプリケーションID | |
| INSERT | expires_in | 有効期限（秒） | 必須 |
| INSERT | scopes | 付与スコープ | |
| INSERT | organization_id | 組織ID | |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| unconfirmed_email | 認証エラー | メール未確認 | メール確認を促す |
| pkce_required_for_dynamic_applications | 認可エラー | ダイナミックアプリでPKCEなし | PKCEパラメータ追加 |
| invalid_code_challenge_method | 認可エラー | S256以外のPKCEメソッド | S256を使用 |
| - | 権限エラー | 管理者向けスコープ制限 | forbidden画面表示 |

### リトライ仕様

認可フローはリトライ不要。トークン交換は同一認可コードでの再試行は不可。

## トランザクション仕様

トークン発行は単一レコード操作のため、明示的なトランザクション不要。

## パフォーマンス要件

- 認可処理：1秒以内
- トークン発行：500ms以内
- トークン保持期間：RETENTION_PERIOD = 1ヶ月

## セキュリティ考慮事項

- クライアントシークレットは暗号化保存
- アクセストークンはハッシュ化保存
- PKCEによるコード横取り攻撃防止
- CSRFトークンによる認可リクエスト保護
- 管理者向けスコープは設定で制限可能
- 監査ログによるOAuth認可の追跡

## 備考

- Doorkeeper gemを基盤としたOAuth2サーバー実装
- MCP（Model Context Protocol）サポート
- 組織（Organization）対応

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | oauth_application.rb | `app/models/authn/oauth_application.rb` | OAuthアプリケーションモデル |
| 1-2 | oauth_access_token.rb | `app/models/oauth_access_token.rb` | アクセストークンモデル |

**主要処理フロー（OauthApplication）**:
- **4行目**: Doorkeeper::Applicationを継承
- **7行目**: organizationとの関連
- **17-31行目**: secret_matches?メソッド（フォールバック対応）

**主要処理フロー（OauthAccessToken）**:
- **3行目**: Doorkeeper::AccessTokenを継承
- **10行目**: expires_in必須バリデーション
- **15行目**: latest_per_applicationスコープ
- **18行目**: RETENTION_PERIOD = 1.month
- **20-26行目**: scopes=メソッド（配列対応）
- **28-34行目**: scope_userメソッド

#### Step 2: コントローラー層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | authorizations_controller.rb | `app/controllers/oauth/authorizations_controller.rb` | 認可フロー |
| 2-2 | applications_controller.rb | `app/controllers/oauth/applications_controller.rb` | アプリ管理 |

**主要処理フロー（AuthorizationsController）**:
- **3行目**: Doorkeeper::AuthorizationsControllerを継承
- **14行目**: verify_confirmed_email!, verify_admin_allowed!
- **16行目**: validate_pkce_for_dynamic_applications
- **17行目**: audit_oauth_authorization
- **24-40行目**: newアクション（認可画面）
- **44-62行目**: audit_oauth_authorization（監査ログ）
- **103-110行目**: downgrade_scopes!（スコープダウングレード）
- **143-148行目**: verify_confirmed_email!
- **154-157行目**: disallow_connect?（管理者制限）
- **159-165行目**: dangerous_scopes?判定
- **167-180行目**: validate_pkce_for_dynamic_applications

**主要処理フロー（ApplicationsController）**:
- **3行目**: Doorkeeper::ApplicationsControllerを継承
- **30-42行目**: createアクション
- **44-54行目**: renewアクション（シークレット更新）
- **64-73行目**: set_index_vars

#### Step 3: サービス層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | create_service.rb | `app/services/applications/create_service.rb` | アプリ作成サービス |

**主要処理フロー**:
- **18-22行目**: initialize
- **24-36行目**: executeメソッド
- **27-31行目**: スコープ必須チェック
- **33行目**: ropc_enabled設定（EE）

### プログラム呼び出し階層図

```
Oauth::ApplicationsController#create
    │
    ├─ Applications::CreateService.new(current_user, request, params)
    │      └─ #execute
    │             ├─ Authn::OauthApplication.new(params)
    │             ├─ スコープバリデーション
    │             └─ save
    │
    └─ レスポンス

Oauth::AuthorizationsController#new
    │
    ├─ verify_confirmed_email! - メール確認チェック
    ├─ verify_admin_allowed! - 管理者制限チェック
    ├─ validate_pkce_for_dynamic_applications - PKCE検証
    │
    ├─ pre_auth_params
    │      ├─ downgrade_scopes! (gl_auth_type=login時)
    │      └─ organization_id設定
    │
    ├─ pre_auth.authorizable?
    │      │
    │      └─ Doorkeeper認可チェック
    │
    ├─ skip_authorization? || matching_token?
    │      └─ 自動承認判定
    │
    └─ authorization.authorize
           │
           ├─ 認可コード生成
           └─ audit_oauth_authorization (after_action)
                  └─ Gitlab::Audit::Auditor.audit
```

### データフロー図

```
[入力]                    [処理]                         [出力]

クライアント          AuthorizationsController          認可コード
(client_id,      ──▶ ├─メール確認              ──▶  redirect_uri?code=xxx
 redirect_uri,       ├─管理者制限チェック
 scope,              ├─PKCE検証
 code_challenge)     ├─認可可能性チェック
                     └─認可画面/自動承認
                                                      ▼
                                                   Doorkeeper
                                                   TokensController
                                                      ▼
                                                   access_token
                                                   refresh_token
                                                   expires_in
                                                      ▼
                                                   監査ログ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| oauth_application.rb | `app/models/authn/oauth_application.rb` | モデル | アプリケーション |
| oauth_access_token.rb | `app/models/oauth_access_token.rb` | モデル | アクセストークン |
| authorizations_controller.rb | `app/controllers/oauth/authorizations_controller.rb` | コントローラー | 認可フロー |
| applications_controller.rb | `app/controllers/oauth/applications_controller.rb` | コントローラー | アプリ管理 |
| create_service.rb | `app/services/applications/create_service.rb` | サービス | アプリ作成 |
| doorkeeper.rb | `config/initializers/doorkeeper.rb` | 設定 | Doorkeeper設定 |
