# 機能設計書 91-JWT認証

## 概要

本ドキュメントは、GitLabにおけるJWT（JSON Web Token）認証機能の設計仕様を記載する。本機能は、コンテナレジストリおよびDependency Proxyへのアクセス時にJWTトークンを発行・検証する認証エンドポイントを提供する。

### 本機能の処理概要

JWT認証機能は、GitLabの内部サービス（Container Registry、Dependency Proxy）に対するトークンベースの認証を実現する。Docker CLIや他のクライアントからのリクエストに対し、HTTP Basic認証を通じてユーザーを認証し、適切な権限スコープを持つJWTトークンを発行する。

**業務上の目的・背景**：コンテナレジストリやDependency Proxyは、Dockerイメージのプッシュ/プル操作を行うために独立したトークンベースの認証が必要である。GitLabの通常のセッション認証とは異なり、CLIツールや自動化されたCI/CDパイプラインからのアクセスに対応するため、標準的なJWTプロトコルを採用している。これにより、各サービスが独自に認証状態を管理せず、GitLabが一元的に認証・認可を行える。

**機能の利用シーン**：本機能は以下のシーンで利用される。
- Docker CLIを使用したコンテナイメージのプッシュ/プル時
- CI/CDパイプラインでのコンテナレジストリアクセス時
- Dependency Proxyを介したDockerイメージのキャッシュアクセス時
- デプロイトークンを使用した自動デプロイ時

**主要な処理内容**：
1. HTTP Basic認証によるユーザー/トークンの検証
2. 要求されたサービス（container_registry/dependency_proxy）の特定
3. スコープに基づく権限チェック
4. RSA署名付きJWTトークンの生成と返却
5. 認証失敗時のエラーレスポンス生成

**関連システム・外部連携**：Container Registry（外部サービス）、Dependency Proxy（外部プロキシ）、LDAP（オプション認証）、OAuth2プロバイダ（トークン認証）

**権限による制御**：リポジトリへのpull/push/delete操作は、ユーザーのプロジェクトメンバーシップと権限レベルに基づいて制御される。管理者はすべての操作が可能。デプロイトークンは設定されたスコープに応じて制限される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | CLI（Docker/Podman等） | 主機能 | docker loginによるJWTトークン取得 |
| 96 | コンテナレジストリ | 参照画面 | トークン使用によるイメージ操作 |

## 機能種別

認証・認可処理 / トークン発行

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| service | String | Yes | 認証対象サービス名（container_registry, dependency_proxy） | SERVICES定数に含まれること |
| scope | String | No | 要求するアクセススコープ（複数可） | type:name:actions形式 |
| account | String | No | アカウント名 | - |
| client_id | String | No | クライアント識別子 | - |
| Authorization | Header | No | HTTP Basic認証ヘッダー | Base64エンコードされたlogin:password |

### 入力データソース

- HTTPリクエストヘッダー（Authorization: Basic）
- クエリパラメータ（service, scope, account, client_id）
- GitLabデータベース（ユーザー情報、デプロイトークン、個人アクセストークン）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| token | String | 署名付きJWTトークン（成功時） |
| errors | Array | エラーコードとメッセージの配列（失敗時） |
| errors[].code | String | エラーコード（UNAUTHORIZED, DENIED等） |
| errors[].message | String | エラーメッセージ |

### 出力先

- HTTPレスポンスボディ（JSON形式）
- 認証ログ（Gitlab::AuthLogger）

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信
   └─ JwtController#authアクションが呼び出される

2. 認証前処理
   └─ セッションストレージスキップ、CSRF検証スキップ

3. ユーザー/プロジェクト認証（authenticate_project_or_user）
   └─ HTTP Basic認証ヘッダーからlogin/passwordを取得
   └─ Gitlab::Auth.find_for_git_clientで認証実行
   └─ 認証失敗時はrender_access_deniedでエラー返却

4. サービス特定
   └─ params[:service]に基づきサービスクラスを決定
   └─ 未知のサービスの場合は404を返却

5. 認証サービス実行
   └─ ContainerRegistryAuthenticationService または
      DependencyProxyAuthenticationServiceを実行
   └─ スコープに基づく権限チェック
   └─ JWTトークン生成

6. レスポンス返却
   └─ 成功時：{token: "..."}をJSON返却
   └─ 失敗時：{errors: [...]}をJSON返却
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{Basic認証ヘッダー存在?}
    B -->|No| C[匿名アクセスとして処理]
    B -->|Yes| D[認証情報抽出]
    D --> E[Gitlab::Auth.find_for_git_client]
    E --> F{認証成功?}
    F -->|No| G[認証失敗ログ出力]
    G --> H[401 Unauthorized返却]
    F -->|Yes| I{サービス存在?}
    C --> I
    I -->|No| J[404 Not Found返却]
    I -->|Yes| K[認証サービス実行]
    K --> L{権限チェック成功?}
    L -->|No| M[403 Forbidden返却]
    L -->|Yes| N[JWTトークン生成]
    N --> O[トークン返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-91-01 | サービス制限 | container_registryとdependency_proxyのみ許可 | 全リクエスト |
| BR-91-02 | レジストリ有効性 | レジストリが無効な場合はアクセス拒否 | container_registryサービス |
| BR-91-03 | 権限継承 | ビルドジョブは起動元プロジェクトからの権限を継承 | CI/CDビルド認証時 |
| BR-91-04 | トークン有効期限 | container_registry_token_expire_delay設定に従う | トークン生成時 |
| BR-91-05 | 保護リポジトリ | 保護ルールに該当するリポジトリへのプッシュは制限 | push/delete操作時 |

### 計算ロジック

**トークン有効期限の算出**：
```
expire_time = Time.current + Gitlab::CurrentSettings.container_registry_token_expire_delay.minutes
```

**HMAC秘密鍵の生成**：
```
secret = OpenSSL::HMAC.hexdigest('sha256', db_key_base, 'gitlab-dependency-proxy')
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ユーザー認証 | users | SELECT | ユーザー情報の取得 |
| トークン検証 | personal_access_tokens | SELECT | 個人アクセストークンの検証 |
| デプロイトークン検証 | deploy_tokens | SELECT | デプロイトークンの検証 |
| コンテナリポジトリ作成 | container_repositories | INSERT | プッシュ時のリポジトリ自動作成 |

### テーブル別操作詳細

#### users

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | id, username, encrypted_password等 | login条件で検索 | 認証時 |

#### container_repositories

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | name, project_id | スコープで指定されたパス | push権限時に自動作成 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| UNAUTHORIZED | 認証エラー | 認証情報が不正または期限切れ | 正しい認証情報で再試行 |
| DENIED | 認可エラー | 要求されたスコープへの権限なし | 必要な権限を持つアカウントで再試行 |
| UNAVAILABLE | サービスエラー | レジストリが無効 | 管理者によるレジストリ有効化 |
| Not Found | パラメータエラー | 不明なservice指定 | 正しいservice名を指定 |

### リトライ仕様

JWTトークン取得はステートレスな操作のため、クライアント側でリトライ可能。認証失敗が連続した場合はIP単位でレートリミットが適用される。

## トランザクション仕様

コンテナリポジトリの自動作成時のみトランザクションを使用。`ContainerRepository.find_or_create_from_path!`メソッド内でトランザクションが管理される。

## パフォーマンス要件

- レスポンス時間：200ms以内（認証処理含む）
- スループット：1000 req/sec（レジストリアクセス頻度に依存）
- IP単位のレートリミット：Rack::Attackによる制御

## セキュリティ考慮事項

- **トークン署名**：RSA秘密鍵によるJWT署名（Container Registry用）
- **HMAC認証**：SHA256 HMACによるトークン生成（Dependency Proxy用）
- **認証ログ**：認証失敗時はGitlab::AuthLoggerに詳細を記録
- **秘密鍵管理**：秘密鍵は32バイト、ファイルパーミッション0600で保護
- **CSRF保護**：APIエンドポイントのためCSRF検証はスキップ
- **セッション不使用**：ステートレス認証のためセッションストレージをスキップ

## 備考

- Docker Registry API v2のトークン認証仕様に準拠
- RFC6749（OAuth2）のスコープパラメータ形式をサポート（スペース区切り）
- 管理者モードのバイパス機能あり（bypass_admin_mode!）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

JWTトークンの構造と認証結果のデータ構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | jwt_token.rb | `lib/gitlab/jwt_token.rb` | JWTトークンのエンコード/デコード処理、HMAC署名の実装 |
| 1-2 | jwt_authenticatable.rb | `lib/gitlab/jwt_authenticatable.rb` | JWT検証のベースモジュール、秘密鍵管理 |

**読解のコツ**: `JSONWebToken::HMACToken`と`JSONWebToken::RSAToken`の2種類のトークン形式が使用される。HMACはDependency Proxy用、RSAはContainer Registry用。

#### Step 2: エントリーポイントを理解する

JWTコントローラーが認証リクエストのエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | jwt_controller.rb | `app/controllers/jwt_controller.rb` | authアクションの処理フロー、サービスディスパッチ |

**主要処理フロー**:
1. **9行目**: `prepend_before_action :auth_user, :authenticate_project_or_user` - 認証の前処理
2. **16-19行目**: `SERVICES`定数 - サポートされるサービスの定義
3. **24-32行目**: `auth`アクション - メイン処理
4. **36-56行目**: `authenticate_project_or_user` - HTTP Basic認証の実行
5. **71-89行目**: `render_access_denied` - 認証失敗時のエラーレスポンス

#### Step 3: 認証ロジックを理解する

Gitlab::Authモジュールが実際の認証処理を担当。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | auth.rb | `lib/gitlab/auth.rb` | find_for_git_clientメソッド、各種認証方式のチェーン |

**主要処理フロー**:
- **119-147行目**: `find_for_git_client` - 認証メソッドのチェーン実行
- **130-136行目**: 認証チェックの順序（サービスリクエスト→ビルドトークン→LFS→OAuth→PAT→デプロイトークン→パスワード）
- **430-438行目**: `build_authentication_abilities` - ビルドジョブの権限定義

#### Step 4: Container Registry認証サービスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | container_registry_authentication_service.rb | `app/services/auth/container_registry_authentication_service.rb` | スコープ処理、トークン生成、権限チェック |

**主要処理フロー**:
- **20-36行目**: `execute` - メイン実行ロジック
- **184-193行目**: `authorized_token` - RSAトークン生成
- **217-223行目**: `scopes` - スコープパラメータのパース
- **246-271行目**: `process_repository_access` - リポジトリアクセス権限の判定
- **291-306行目**: `can_access?` - 操作別権限チェック

#### Step 5: Dependency Proxy認証サービスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | dependency_proxy_authentication_service.rb | `app/services/auth/dependency_proxy_authentication_service.rb` | HMACトークン生成、権限チェック |

**主要処理フロー**:
- **14-21行目**: `execute` - メイン実行ロジック
- **59-66行目**: `authorized_token` - HMACトークン生成

### プログラム呼び出し階層図

```
JwtController#auth
    |
    +-- authenticate_project_or_user (before_action)
    |       |
    |       +-- Gitlab::Auth.find_for_git_client
    |               |
    |               +-- service_request_check
    |               +-- build_access_token_check
    |               +-- lfs_token_check
    |               +-- oauth_access_token_check
    |               +-- personal_access_token_check
    |               +-- deploy_token_check
    |               +-- user_with_password_for_git
    |
    +-- Auth::ContainerRegistryAuthenticationService#execute
    |       |
    |       +-- authorized_token
    |       +-- process_scope
    |       +-- can_access?
    |
    +-- Auth::DependencyProxyAuthenticationService#execute
            |
            +-- authorized_token
            +-- valid_user_actor?
```

### データフロー図

```
[入力]                      [処理]                           [出力]

HTTP Basic Auth     ---->   JwtController                ---->  JSON Response
(login:password)            #authenticate_project_or_user       {token: "..."}
                                    |
Query Params        ---->           v
(service, scope)            Gitlab::Auth.find_for_git_client
                                    |
                                    v
                            Auth::*AuthenticationService
                                    |
                                    v
                            JSONWebToken::RSAToken or
                            JSONWebToken::HMACToken
                                    |
                                    v
                            Signed JWT Token
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| jwt_controller.rb | `app/controllers/jwt_controller.rb` | ソース | JWTエンドポイントのコントローラー |
| container_registry_authentication_service.rb | `app/services/auth/container_registry_authentication_service.rb` | ソース | Container Registry用認証サービス |
| dependency_proxy_authentication_service.rb | `app/services/auth/dependency_proxy_authentication_service.rb` | ソース | Dependency Proxy用認証サービス |
| auth.rb | `lib/gitlab/auth.rb` | ソース | 認証処理の共通モジュール |
| jwt_token.rb | `lib/gitlab/jwt_token.rb` | ソース | JWTトークンクラス |
| jwt_authenticatable.rb | `lib/gitlab/jwt_authenticatable.rb` | ソース | JWT認証のベースモジュール |
| routes.rb | `config/routes.rb` | 設定 | ルーティング定義（/jwt/auth） |
