# 画面設計書 292-Bitbucket Serverインポート状態

## 概要

本ドキュメントは、GitLabにおけるBitbucket Serverからのインポート状態・リポジトリ選択画面の設計を記載したものである。

### 本画面の処理概要

Bitbucket Serverからインポート可能なリポジトリの一覧を表示し、ユーザーがインポートするリポジトリを選択・実行できる画面である。認証済みのBitbucket Server接続情報を使用してリポジトリ一覧を取得し、各リポジトリのインポート状態を管理する。

**業務上の目的・背景**：Bitbucket Serverインポート画面（No.291）で認証した後、実際にインポートするリポジトリを選択するための画面である。ユーザーは利用可能なリポジトリを一覧で確認し、必要なものを選択してGitLabにインポートできる。インポート中・完了・失敗の状態をリアルタイムで確認できる。

**画面へのアクセス方法**：
1. Bitbucket Serverインポート画面（No.291）で認証情報を入力
2. 「List your Bitbucket Server repositories」ボタンを押下
3. 認証成功後、自動的に本画面へリダイレクト

**主要な操作・処理内容**：
1. Bitbucket Serverのリポジトリ一覧を表示
2. リポジトリのフィルタリング・検索
3. インポート先ネームスペースの選択
4. インポート先プロジェクト名のカスタマイズ
5. 「Import」ボタンでインポート実行
6. インポート状態のリアルタイム監視

**画面遷移**：
- 遷移元：Bitbucket Serverインポート画面（No.291）
- 遷移先：インポートされたプロジェクト詳細画面

**権限による表示制御**：
- プロジェクト作成権限を持つユーザーのみがインポートを実行可能
- namespace_idが指定されている場合、そのネームスペースへのインポート権限が必要
- インポート先ネームスペースの選択は、ユーザーがアクセス可能なネームスペースのみ表示

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 107 | Bitbucketインポート | 主機能 | Bitbucket Serverからのインポート処理・状態管理 |

## 画面種別

一覧・操作

## URL/ルーティング

```
GET /import/bitbucket_server/status
GET /import/bitbucket_server/status.json
POST /import/bitbucket_server
GET /import/bitbucket_server/realtime_changes.json
```

ルーティング定義: `config/routes/import.rb`
```ruby
resource :bitbucket_server, only: [:create, :new], controller: :bitbucket_server do
  post :configure
  get :status
  get :callback
  get :realtime_changes
end
```

## 入出力項目

### 入力項目（リポジトリ選択時）

| 項目名 | 項目ID | 型 | 必須 | 説明 |
|--------|--------|-----|------|------|
| リポジトリID | repo_id | 文字列 | 必須 | インポート対象のリポジトリ識別子（project_key/repo_slug形式） |
| 新しい名前 | new_name | 文字列 | 任意 | GitLabでのプロジェクト名（デフォルトは元の名前） |
| 新しいネームスペース | new_namespace | 文字列 | 任意 | インポート先ネームスペース |
| フィルター | filter | 文字列 | 任意 | リポジトリ名での絞り込み |

### 出力項目（JSON API）

| 項目名 | 説明 |
|--------|------|
| imported_projects | インポート済み/進行中のプロジェクト一覧 |
| provider_repos | インポート可能なリポジトリ一覧 |
| incompatible_repos | 互換性のないリポジトリ一覧 |
| page_info | ページネーション情報 |

## 表示項目

| 項目名 | 説明 |
|--------|------|
| ページタイトル | 「Bitbucket Server import」 |
| Bitbucketアイコン | Bitbucketロゴアイコン（48px） |
| リポジトリ一覧 | インポート可能なリポジトリのテーブル |
| インポート状態 | 各リポジトリのインポート進捗状況 |
| フィルター入力 | リポジトリ名での絞り込み |
| ネームスペース選択 | インポート先の選択 |
| Reconfigureリンク | 接続設定を変更するためのリンク |

## イベント仕様

### 1-インポート実行

「Import」ボタンを押下すると以下の処理が実行される：

1. POSTリクエストを`/import/bitbucket_server`に送信
2. コントローラーでcreateアクションが実行
3. `Import::BitbucketServerService`でインポート処理を実行
4. 成功時：プロジェクト情報をJSONで返却
5. 失敗時：エラーメッセージをJSONで返却

```ruby
# app/controllers/import/bitbucket_server_controller.rb (行32-54)
def create
  repo = client.repo(@project_key, @repo_slug)

  unless repo
    return render json: {
      errors: safe_format(
        s_("Project %{project_repo} could not be found"),
        project_repo: "#{@project_key}/#{@repo_slug}"
      )
    }, status: :unprocessable_entity
  end

  result = Import::BitbucketServerService.new(
    client,
    current_user,
    params.merge({ organization_id: Current.organization.id })
  ).execute(credentials)

  if result[:status] == :success
    render json: ProjectSerializer.new.represent(result[:project], serializer: :import)
  else
    render json: { errors: result[:message] }, status: result[:http_status]
  end
end
```

### 2-リアルタイム状態更新

ポーリングによりインポート状態をリアルタイム更新：

1. `realtime_changes`エンドポイントへ3秒間隔でリクエスト
2. インポート中のプロジェクトの状態（import_status）を取得
3. UIを更新して進捗を反映

### 3-フィルタリング

フィルター入力時：
1. 入力値をサニタイズ
2. リポジトリ名で部分一致検索
3. 一覧を再描画

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| Importボタン押下 | projects | INSERT | 新規プロジェクトを作成 |
| Importボタン押下 | project_import_data | INSERT | インポートデータを保存 |
| Importボタン押下 | import_states | INSERT | インポート状態を記録 |
| Importボタン押下 | namespaces | SELECT | インポート先ネームスペースを取得 |

### テーブル別更新項目詳細

#### projects

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | name | new_name または リポジトリ名 | プロジェクト名 |
| INSERT | path | プロジェクト名から生成 | URLパス |
| INSERT | namespace_id | new_namespace から取得 | インポート先ネームスペース |
| INSERT | import_type | 'bitbucket_server' | インポート種別 |
| INSERT | import_url | Bitbucket Server リポジトリURL | インポート元URL |
| INSERT | creator_id | current_user.id | 作成者 |

## メッセージ仕様

| メッセージID | メッセージ種別 | メッセージ内容 | 表示条件 |
|-------------|---------------|---------------|---------|
| MSG001 | エラー | Project {project_repo} could not be found | リポジトリが見つからない場合 |
| MSG002 | エラー | You don't have permissions to import this project | 権限不足の場合 |
| MSG003 | エラー | Import failed due to a BitBucket Server error: {error} | 接続エラーの場合 |
| MSG004 | エラー | Invalid URL: {url} | URLがブロックされている場合 |
| MSG005 | エラー | Invalid project key | プロジェクトキーが不正な場合 |
| MSG006 | エラー | Invalid repository slug | リポジトリスラグが不正な場合 |

## 例外処理

| 例外パターン | 処理内容 |
|-------------|---------|
| 認証情報がセッションにない | Bitbucket Serverインポート画面（No.291）へリダイレクト |
| Bitbucket Server接続エラー | エラーメッセージを表示し、セッションをクリアして再認証画面へ |
| リポジトリが見つからない | 422エラーとエラーメッセージを返却 |
| 権限不足 | 401エラーとエラーメッセージを返却 |
| URLブロック | 400エラーとエラーメッセージを返却 |

## 備考

- ページネーション対応（`paginatable: true`）
- Bitbucket Serverの仕様上、プロジェクトキーは英数字とアンダースコアのみ許可
- パーソナルプロジェクトは`~`で始まるプロジェクトキーを使用
- インポート処理はバックグラウンドジョブで実行される

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Bitbucket Serverからのリポジトリ情報とインポート状態の構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | bitbucket_server_controller.rb | `app/controllers/import/bitbucket_server_controller.rb` | importable_repos（行74-77）、incompatible_repos（行79-82） |
| 1-2 | bitbucket_server_service.rb | `app/services/import/bitbucket_server_service.rb` | サービスの戻り値構造 |

**読解のコツ**: リポジトリ一覧は`valid?`メソッドで互換性チェックが行われ、互換性のあるものとないものに分類される。

#### Step 2: エントリーポイントを理解する

ステータス画面の表示とインポート実行のエントリーポイントを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | bitbucket_server_controller.rb | `app/controllers/import/bitbucket_server_controller.rb` | statusアクション（行66-70）、createアクション（行32-54） |
| 2-2 | base_controller.rb | `app/controllers/import/base_controller.rb` | 親クラスのstatusメソッド（行17-32） |

**主要処理フロー**:
1. **行66-70**: `status`アクション - 親クラスのstatusを呼び出し
2. **行32-54**: `create`アクション - インポート実行

#### Step 3: ビューレイヤーを理解する

画面表示と共通コンポーネントの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | status.html.haml | `app/views/import/bitbucket_server/status.html.haml` | パーシャルの呼び出しとパラメータ |
| 3-2 | _githubish_status.html.haml | `app/views/import/_githubish_status.html.haml` | 共通インポート状態表示コンポーネント |

**主要処理フロー**:
- **行8-12**: `_githubish_status`パーシャルに`provider: 'bitbucket_server'`、`paginatable: true`等を渡す
- パーシャル内でVueコンポーネントをマウント

#### Step 4: サービスレイヤーを理解する

インポート処理の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | bitbucket_server_service.rb | `app/services/import/bitbucket_server_service.rb` | executeメソッド（行7-33）、create_project（行37-48） |

### プログラム呼び出し階層図

```
GET /import/bitbucket_server/status
    │
    └─ Import::BitbucketServerController#status
           │
           └─ Import::BaseController#status
                  │
                  ├─ serialized_imported_projects
                  │
                  ├─ serialized_provider_repos
                  │      └─ importable_repos (bitbucket_repos.filter(&:valid?))
                  │
                  └─ serialized_incompatible_repos

POST /import/bitbucket_server
    │
    └─ Import::BitbucketServerController#create
           │
           ├─ normalize_import_params
           │
           ├─ validate_import_params
           │
           ├─ client.repo(@project_key, @repo_slug)
           │
           └─ Import::BitbucketServerService#execute
                  │
                  └─ Gitlab::BitbucketServerImport::ProjectCreator#execute
```

### データフロー図

```
[入力]                    [処理]                              [出力]

Bitbucket Server   ───▶  BitbucketServer::Client       ───▶  リポジトリ一覧
認証情報                  #repos                              (JSON)
                                                              │
                                                              ▼
ユーザー選択       ───▶  BitbucketServerService       ───▶  Project (作成)
(repo_id,                 #execute                            │
new_name,                                                     ▼
new_namespace)                                           インポートジョブ
                                                         (バックグラウンド)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| bitbucket_server_controller.rb | `app/controllers/import/bitbucket_server_controller.rb` | コントローラー | リクエスト処理 |
| status.html.haml | `app/views/import/bitbucket_server/status.html.haml` | テンプレート | ステータス画面表示 |
| _githubish_status.html.haml | `app/views/import/_githubish_status.html.haml` | パーシャル | インポート状態共通UI |
| bitbucket_server_service.rb | `app/services/import/bitbucket_server_service.rb` | サービス | インポート処理 |
| base_controller.rb | `app/controllers/import/base_controller.rb` | コントローラー | 基底クラス |
| project_creator.rb | `lib/gitlab/bitbucket_server_import/project_creator.rb` | ライブラリ | プロジェクト作成 |
| client.rb | `lib/bitbucket_server/client.rb` | ライブラリ | Bitbucket Server API クライアント |
