# 画面設計書 295-Manifestインポート状態

## 概要

本ドキュメントは、GitLabにおけるManifestファイルからのインポート状態・リポジトリ選択画面の設計を記載したものである。

### 本画面の処理概要

Manifestインポート画面（No.294）でアップロードされたManifestファイルから解析されたリポジトリ一覧を表示し、ユーザーがインポートするリポジトリを選択・実行できる画面である。各リポジトリのインポート状態をリアルタイムで監視できる。

**業務上の目的・背景**：Manifestファイルから抽出された複数のリポジトリ情報を一覧表示し、ユーザーが必要なリポジトリを選択してインポートできるようにする。大規模プロジェクトでは数百のリポジトリが含まれることがあるため、選択的なインポートが重要となる。

**画面へのアクセス方法**：
1. Manifestインポート画面（No.294）でManifestファイルをアップロード
2. 解析成功後、自動的に本画面へリダイレクト

**主要な操作・処理内容**：
1. Manifestから解析されたリポジトリ一覧を表示
2. インポート先ネームスペースの確認（グループ固定）
3. 「Import」ボタンで各リポジトリのインポート実行
4. インポート状態のリアルタイム監視

**画面遷移**：
- 遷移元：Manifestインポート画面（No.294）
- 遷移先：インポートされたプロジェクト詳細画面

**権限による表示制御**：
- インポート先グループに対するインポート権限が必要
- グループが削除されている場合はManifestインポート画面へリダイレクト

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 110 | バルクインポート | 主機能 | Manifestファイルからのインポート処理・状態管理 |

## 画面種別

一覧・操作

## URL/ルーティング

```
GET /import/manifest/status
GET /import/manifest/status.json
POST /import/manifest
GET /import/manifest/realtime_changes.json
```

ルーティング定義: `config/routes/import.rb`
```ruby
resource :manifest, only: [:create, :new], controller: :manifest do
  get :status
  get :realtime_changes
  post :upload
end
```

## 入出力項目

### 入力項目（リポジトリ選択時）

| 項目名 | 項目ID | 型 | 必須 | 説明 |
|--------|--------|-----|------|------|
| リポジトリID | repo_id | 整数 | 必須 | インポート対象のリポジトリインデックス |
| 新しい名前 | new_name | 文字列 | 任意 | GitLabでのプロジェクト名 |

### 出力項目（JSON API）

| 項目名 | 説明 |
|--------|------|
| imported_projects | インポート済み/進行中のプロジェクト一覧 |
| provider_repos | インポート可能なリポジトリ一覧 |
| incompatible_repos | 常に空配列（Manifestでは非互換リポジトリなし） |

## 表示項目

| 項目名 | 説明 |
|--------|------|
| ページタイトル | 「Manifest file import」 |
| リポジトリ一覧 | Manifestから解析されたリポジトリのテーブル |
| インポート状態 | 各リポジトリのインポート進捗状況 |
| グループパス | インポート先グループのフルパス |

## イベント仕様

### 1-インポート実行

「Import」ボタンを押下すると以下の処理が実行される：

```ruby
# app/controllers/import/manifest_controller.rb (行62-74)
def create
  repository = importable_repos.find do |project|
    project[:id] == params[:repo_id].to_i
  end

  project = Gitlab::ManifestImport::ProjectCreator.new(repository, group, current_user).execute

  if project.persisted?
    render json: ProjectSerializer.new.represent(project, serializer: :import)
  else
    render json: { errors: project_save_error(project) }, status: :unprocessable_entity
  end
end
```

### 2-リアルタイム状態更新

基底クラス`Import::BaseController`の`realtime_changes`アクションにより、3秒間隔でインポート状態を更新。

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| Importボタン押下 | projects | INSERT | 新規プロジェクトを作成 |
| Importボタン押下 | project_import_data | INSERT | インポートデータを保存 |
| Importボタン押下 | import_states | INSERT | インポート状態を記録 |

### テーブル別更新項目詳細

#### projects

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | name | new_name または リポジトリパス | プロジェクト名 |
| INSERT | path | リポジトリパスから生成 | URLパス |
| INSERT | namespace_id | group.id | インポート先グループ |
| INSERT | import_type | 'manifest' | インポート種別 |
| INSERT | import_url | リポジトリURL | インポート元URL |
| INSERT | creator_id | current_user.id | 作成者 |

## メッセージ仕様

| メッセージID | メッセージ種別 | メッセージ内容 | 表示条件 |
|-------------|---------------|---------------|---------|
| MSG001 | エラー | {project_save_error} | プロジェクト保存エラー時 |

## 例外処理

| 例外パターン | 処理内容 |
|-------------|---------|
| メタデータがセッションにない | Manifestインポート画面（No.294）へリダイレクト |
| グループが存在しない | Manifestインポート画面へリダイレクト |
| インポート可能なリポジトリがない | Manifestインポート画面へリダイレクト |

## 備考

- Manifestインポートでは非互換リポジトリは存在しない（incompatible_reposは常に空）
- インポート先グループはアップロード時に指定されたグループに固定
- グループのフルパスがextra_representation_optsとして渡される

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Manifestメタデータの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | manifest_controller.rb | `app/controllers/import/manifest_controller.rb` | importable_repos（行78-81）、group（行110-113） |
| 1-2 | metadata.rb | `lib/gitlab/manifest_import/metadata.rb` | repositoriesメソッド、group_id |

**読解のコツ**: メタデータはセッションまたはfallbackストレージに保存され、repositoriesメソッドで取得。

#### Step 2: エントリーポイントを理解する

statusアクションとcreateアクションを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | manifest_controller.rb | `app/controllers/import/manifest_controller.rb` | status（行15-20）、create（行62-74） |
| 2-2 | base_controller.rb | `app/controllers/import/base_controller.rb` | 親クラスのstatusメソッド |

**主要処理フロー**:
1. **行15-20**: `status`アクション - 親クラスのstatusを呼び出し
2. **行62-74**: `create`アクション - インポート実行

#### Step 3: ビューレイヤーを理解する

画面表示の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | status.html.haml | `app/views/import/manifest/status.html.haml` | _githubish_statusパーシャルの呼び出し |
| 3-2 | _githubish_status.html.haml | `app/views/import/_githubish_status.html.haml` | 共通インポート状態表示コンポーネント |

#### Step 4: インポート処理を理解する

プロジェクト作成の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | project_creator.rb | `lib/gitlab/manifest_import/project_creator.rb` | executeメソッド |

### プログラム呼び出し階層図

```
GET /import/manifest/status
    │
    └─ Import::ManifestController#status
           │
           ├─ ensure_import_vars (before_action)
           │      └─ group && importable_repos.present? チェック
           │
           └─ Import::BaseController#status
                  │
                  ├─ serialized_imported_projects
                  │
                  └─ serialized_provider_repos
                         └─ importable_repos (manifest_import_metadata.repositories)

POST /import/manifest
    │
    └─ Import::ManifestController#create
           │
           ├─ importable_repos.find (repo_id でマッチング)
           │
           └─ Gitlab::ManifestImport::ProjectCreator#execute
```

### データフロー図

```
[入力]                    [処理]                              [出力]

セッション/Redis   ───▶  manifest_import_metadata     ───▶  リポジトリ一覧
(メタデータ)              .repositories                      (JSON)
                                                              │
                                                              ▼
ユーザー選択       ───▶  ManifestImport::             ───▶  Project (作成)
(repo_id)                 ProjectCreator#execute
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| manifest_controller.rb | `app/controllers/import/manifest_controller.rb` | コントローラー | リクエスト処理 |
| status.html.haml | `app/views/import/manifest/status.html.haml` | テンプレート | ステータス画面表示 |
| _githubish_status.html.haml | `app/views/import/_githubish_status.html.haml` | パーシャル | インポート状態共通UI |
| base_controller.rb | `app/controllers/import/base_controller.rb` | コントローラー | 基底クラス |
| project_creator.rb | `lib/gitlab/manifest_import/project_creator.rb` | ライブラリ | プロジェクト作成 |
| metadata.rb | `lib/gitlab/manifest_import/metadata.rb` | ライブラリ | メタデータ管理 |
