# 画面設計書 299-URLインポート

## 概要

本ドキュメントは、GitLabにおけるリポジトリURLを直接指定してプロジェクトをインポートする画面の設計を記載したものである。

### 本画面の処理概要

外部のGitリポジトリURLを入力し、そのリポジトリをGitLabに直接インポートする画面である。GitHub、Bitbucket、GitLabなど特定のプロバイダに依存せず、HTTP(S)やGitプロトコルでアクセス可能な任意のGitリポジトリをインポートできる。

**業務上の目的・背景**：プライベートなGitサーバーや、特定のインポート機能が提供されていないGitホスティングサービスからリポジトリをインポートする際に使用する。認証情報（ユーザー名・パスワード）を指定することで、プライベートリポジトリのインポートも可能。また、ミラーリング機能を使用して定期的に同期することも可能。

**画面へのアクセス方法**：
1. サイドバーまたはヘッダーから「プロジェクト新規作成」を選択
2. 「Import project」タブを選択
3. 「Repository by URL」を選択して本画面にアクセス

**主要な操作・処理内容**：
1. Git リポジトリ URL の入力
2. 認証情報（ユーザー名・パスワード）の入力（オプション）
3. 接続確認ボタンで接続テスト
4. プロジェクト名・スラグの入力
5. インポート先ネームスペースの選択
6. ミラーリングオプションの設定（オプション、Premium機能）
7. 「Create project」ボタンでインポート実行

**画面遷移**：
- 遷移元：プロジェクト新規作成画面（インポート選択）
- 遷移先：インポートされたプロジェクト詳細画面

**権限による表示制御**：
- プロジェクト作成権限を持つユーザーのみがアクセス可能
- ミラーリング機能はPremium以上のライセンスで利用可能

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 108 | プロジェクトエクスポート | 主機能 | リポジトリURLからのインポート処理 |

## 画面種別

登録

## URL/ルーティング

```
GET /import/url/new
POST /import/url/validate
POST /projects (プロジェクト作成)
```

ルーティング定義: `config/routes/import.rb`
```ruby
resource :url, only: [:new], controller: :url do
  post :validate
end
```

## 入出力項目

| 項目名 | 項目ID | 型 | 必須 | 説明 |
|--------|--------|-----|------|------|
| Git リポジトリ URL | project[import_url] | URL | 必須 | インポート元のGitリポジトリURL |
| ユーザー名 | project[import_url_user] | 文字列 | 任意 | 認証用ユーザー名 |
| パスワード | project[import_url_password] | パスワード | 任意 | 認証用パスワード |
| プロジェクト名 | name | 文字列 | 必須 | GitLabでのプロジェクト名 |
| プロジェクトスラグ | path | 文字列 | 必須 | GitLabでのプロジェクトパス |
| ネームスペースID | namespace_id | 整数 | 必須 | インポート先ネームスペース |
| ミラーリング | project[mirror] | ブーリアン | 任意 | リポジトリミラーリングを有効にするか |

## 表示項目

| 項目名 | 説明 |
|--------|------|
| ページタイトル | 「Import repository by URL」 |
| フォームタイトル | 「Import repository by URL」 |
| URL説明 | 「The repository must be accessible over http://, https:// or git://」 |
| 接続確認ボタン | 「Check connection」で接続テスト |
| ミラーリング説明 | プルミラーリングの説明とヘルプリンク |

## イベント仕様

### 1-接続確認

「Check connection」ボタンを押下すると以下の処理が実行される：

```javascript
// app/assets/javascripts/projects/new_v2/components/import_by_url_form.vue (行74-98)
async checkConnection() {
  this.isRepositoryUrlValid = isReasonableGitUrl(this.repositoryUrl);
  if (!this.isRepositoryUrlValid) return;

  this.isCheckingConnection = true;
  try {
    const { data } = await axios.post(this.importByUrlValidatePath, {
      url: this.repositoryUrl,
      user: this.repositoryUsername,
      password: this.repositoryPassword,
    });

    if (data.success) {
      this.$toast.show(s__('ProjectImportByURL|Connection successful.'));
    } else {
      this.$toast.show(
        sprintf(s__('ProjectImportByURL|Connection failed: %{error}'), { error: data.message }),
      );
    }
  } catch (error) {
    this.$toast.show(sprintf(s__('ProjectImportByURL|Connection failed: %{error}'), { error }));
  } finally {
    this.isCheckingConnection = false;
  }
}
```

### 2-プロジェクト作成

「Create project」ボタンを押下すると以下の処理が実行される：

1. フォームをPOSTで`/projects`に送信
2. `Projects::CreateService`でプロジェクトを作成
3. import_urlが設定されている場合、インポート処理を開始
4. 成功時、プロジェクト詳細画面へリダイレクト

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| Create project | projects | INSERT | 新規プロジェクトを作成 |
| Create project | project_import_data | INSERT | インポートデータ（URL、認証情報）を保存 |
| Create project | import_states | INSERT | インポート状態を記録 |
| Create project | remote_mirrors | INSERT | ミラーリング有効時、ミラー設定を保存 |

### テーブル別更新項目詳細

#### projects

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | name | 入力されたプロジェクト名 | プロジェクト名 |
| INSERT | path | 入力されたスラグ | URLパス |
| INSERT | namespace_id | 選択されたネームスペースID | インポート先 |
| INSERT | import_type | 'none'（通常インポート） | インポート種別 |
| INSERT | import_url | 入力されたURL | インポート元URL |
| INSERT | creator_id | current_user.id | 作成者 |
| INSERT | mirror | true/false | ミラーリング有効/無効 |

#### project_import_data

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | project_id | 作成されたプロジェクトID | 関連プロジェクト |
| INSERT | credentials | 暗号化された認証情報 | ユーザー名・パスワード（暗号化） |

## メッセージ仕様

| メッセージID | メッセージ種別 | メッセージ内容 | 表示条件 |
|-------------|---------------|---------------|---------|
| MSG001 | 成功 | Connection successful. | 接続確認成功時 |
| MSG002 | エラー | Connection failed: {error} | 接続確認失敗時 |
| MSG003 | バリデーション | Enter a valid URL | URL形式不正時 |
| MSG004 | エラー | {project_errors} | プロジェクト作成エラー時 |

## 例外処理

| 例外パターン | 処理内容 |
|-------------|---------|
| URL形式不正 | バリデーションエラーを表示 |
| 接続失敗 | トーストでエラーメッセージを表示 |
| プロジェクト作成エラー | エラーメッセージを表示して再入力を促す |
| 権限不足 | プロジェクト作成画面でエラーを表示 |

## 備考

- URL形式のバリデーションはフロントエンドで実施（`isReasonableGitUrl`）
- 許可されるプロトコル: http://, https://, git://
- ミラーリング機能はPremium/Ultimate版で利用可能
- 認証情報は暗号化されてデータベースに保存される

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

プロジェクト作成パラメータとインポートデータの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | import_by_url_form.vue | `app/assets/javascripts/projects/new_v2/components/import_by_url_form.vue` | フォームデータ構造（行55-63） |

**読解のコツ**: フォームは標準のプロジェクト作成フォームにインポートURL関連のフィールドを追加した形式。

#### Step 2: エントリーポイントを理解する

ビューの初期化とVueコンポーネントのマウントを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | new.html.haml | `app/views/import/url/new.html.haml` | データ属性の設定 |
| 2-2 | index.js | `app/assets/javascripts/import/url/index.js` | Vueコンポーネントの初期化 |

**主要処理フロー**:
1. HAMLでdata属性を設定
2. JavaScript初期化でVueコンポーネントをマウント

#### Step 3: フォームコンポーネントを理解する

Vueコンポーネントの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | import_by_url_form.vue | `app/assets/javascripts/projects/new_v2/components/import_by_url_form.vue` | 全体構造、checkConnectionメソッド |

**主要処理フロー**:
- **行74-98**: `checkConnection`メソッド - 接続確認処理
- **行99-108**: `onSelectNamespace`, `onBack`メソッド

#### Step 4: バリデーション処理を理解する

接続確認のバリデーションを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | url_utility.js | `app/assets/javascripts/lib/utils/url_utility.js` | isReasonableGitUrl関数 |

### プログラム呼び出し階層図

```
GET /import/url/new
    │
    └─ Import::UrlController#new（暗黙的）
           └─ app/views/import/url/new.html.haml
                  │
                  └─ .js-vue-import-by-url-app
                         └─ initImportByUrl (JavaScript)
                                └─ import_by_url_form.vue

POST /import/url/validate
    │
    └─ Import::UrlController#validate
           │
           └─ 接続確認処理
                  └─ JSON レスポンス

POST /projects (フォーム送信)
    │
    └─ ProjectsController#create
           │
           ├─ Projects::CreateService
           │
           └─ インポート処理（バックグラウンドジョブ）
```

### データフロー図

```
[入力]                    [処理]                              [出力]

Git URL, User,     ───▶  Import::UrlController        ───▶  接続確認結果
Password                  #validate                          (JSON)
                                                              │
                                                              ▼
                                                         トースト表示

フォーム全体       ───▶  ProjectsController           ───▶  Project (作成)
(name, path,             #create                             │
namespace_id,                                                ▼
import_url等)                                           プロジェクト画面へ
                                                         リダイレクト
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| new.html.haml | `app/views/import/url/new.html.haml` | テンプレート | 初期画面表示 |
| index.js | `app/assets/javascripts/import/url/index.js` | JavaScript | Vue初期化 |
| import_by_url_form.vue | `app/assets/javascripts/projects/new_v2/components/import_by_url_form.vue` | Vueコンポーネント | フォームUI |
| import.rb | `config/routes/import.rb` | ルーティング | URLルーティング定義 |
| url_utility.js | `app/assets/javascripts/lib/utils/url_utility.js` | ユーティリティ | URLバリデーション |
