# API設計書

## 概要

本ドキュメントは、Godotゲームエンジンが提供するプログラミングAPI（Application Programming Interface）の設計仕様を記載する。Godotは従来のREST APIサーバーではなく、ゲームエンジンであるため、以下のカテゴリのAPIを提供する：

1. **HTTP Client API** - HTTPリクエスト送信用クラス
2. **HTTPRequest API** - 高レベルHTTPリクエストノード
3. **WebSocket API** - WebSocket通信用クラス
4. **WebSocketMultiplayerPeer API** - マルチプレイヤーWebSocket通信
5. **JSON-RPC API** - JSON-RPCプロトコル処理用クラス
6. **JSON API** - JSONシリアライゼーション用クラス
7. **GDExtension API** - C/C++向けネイティブ拡張インターフェース
8. **GDExtensionManager API** - GDExtension管理シングルトン

## 共通仕様

### API提供形式

Godotはゲームエンジンであり、APIはC++クラスおよびGDScript/C#向けのバインディングとして提供される。

```
- C++ヘッダーファイル: core/, modules/, scene/ ディレクトリ
- GDScriptバインディング: ClassDB経由で自動登録
- C#バインディング: Mono/.NETバインディング
```

### 共通エラーコード（Error enum）

| エラーコード | 値 | 説明 |
| --- | --- | --- |
| OK | 0 | 正常終了 |
| FAILED | 1 | 一般的な失敗 |
| ERR_UNAVAILABLE | 2 | 機能が利用不可 |
| ERR_UNCONFIGURED | 3 | 未設定 |
| ERR_UNAUTHORIZED | 4 | 認証エラー |
| ERR_PARAMETER_RANGE_ERROR | 5 | パラメータ範囲外 |
| ERR_OUT_OF_MEMORY | 6 | メモリ不足 |
| ERR_FILE_NOT_FOUND | 7 | ファイル未検出 |
| ERR_FILE_BAD_DRIVE | 8 | 不正なドライブ |
| ERR_FILE_BAD_PATH | 9 | 不正なパス |
| ERR_TIMEOUT | 24 | タイムアウト |
| ERR_CANT_CONNECT | 25 | 接続失敗 |
| ERR_CANT_RESOLVE | 26 | 名前解決失敗 |
| ERR_CONNECTION_ERROR | 27 | 接続エラー |

## API一覧

| カテゴリ | クラス名 | ソースファイル | 主要メソッド | 説明 |
| --- | --- | --- | --- | --- |
| HTTP | HTTPClient | core/io/http_client.h | connect_to_host(), request() | 低レベルHTTPクライアント |
| HTTP | HTTPRequest | scene/main/http_request.h | request(), cancel_request() | 高レベルHTTPリクエストノード |
| WebSocket | WebSocketPeer | modules/websocket/websocket_peer.h | connect_to_url(), send() | WebSocket通信ピア |
| WebSocket | WebSocketMultiplayerPeer | modules/websocket/websocket_multiplayer_peer.h | create_client(), create_server() | マルチプレイヤーWebSocket |
| JSON-RPC | JSONRPC | modules/jsonrpc/jsonrpc.h | process_action(), make_request() | JSON-RPCプロトコル処理 |
| JSON | JSON | core/io/json.h | parse(), stringify() | JSONシリアライゼーション |
| Extension | GDExtension | core/extension/gdextension.h | open_library(), initialize_library() | GDExtensionライブラリ管理 |
| Extension | GDExtensionManager | core/extension/gdextension_manager.h | load_extension(), unload_extension() | GDExtension管理シングルトン |

## 各APIエンドポイント定義

### HTTP Client API

#### 1. HTTPClient.connect_to_host

HTTPホストへの接続を確立する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPClient` (extends RefCounted) |
| ソースファイル | `core/io/http_client.h:171` |
| メソッド | `connect_to_host(host: String, port: int, tls_options: TLSOptions)` |
| 戻り値 | `Error` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| host | String | ○ | - | 接続先ホスト名またはIPアドレス |
| port | int | - | -1 | ポート番号（-1の場合は自動判定：HTTP=80, HTTPS=443） |
| tls_options | TLSOptions | - | null | TLS設定オプション |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Error | OK=成功、その他=エラーコード |

**使用例（GDScript）**

```gdscript
var http = HTTPClient.new()
var err = http.connect_to_host("example.com", 443, TLSOptions.client())
if err == OK:
    print("接続開始")
```

#### 2. HTTPClient.request

HTTPリクエストを送信する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPClient` |
| ソースファイル | `core/io/http_client.h:170` |
| メソッド | `request(method: Method, url: String, headers: PackedStringArray, body: String)` |
| 戻り値 | `Error` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| method | Method | ○ | - | HTTPメソッド（GET, POST, PUT, DELETE等） |
| url | String | ○ | - | リクエストURL（パス部分） |
| headers | PackedStringArray | ○ | - | リクエストヘッダー配列 |
| body | String | - | "" | リクエストボディ |

**HTTPメソッド（Method enum）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| METHOD_GET | 0 | GET |
| METHOD_HEAD | 1 | HEAD |
| METHOD_POST | 2 | POST |
| METHOD_PUT | 3 | PUT |
| METHOD_DELETE | 4 | DELETE |
| METHOD_OPTIONS | 5 | OPTIONS |
| METHOD_TRACE | 6 | TRACE |
| METHOD_CONNECT | 7 | CONNECT |
| METHOD_PATCH | 8 | PATCH |
| METHOD_MAX | 9 | メソッド数（内部用） |

**レスポンスコード（ResponseCode enum）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| RESPONSE_CONTINUE | 100 | 続行 |
| RESPONSE_SWITCHING_PROTOCOLS | 101 | プロトコル切替 |
| RESPONSE_PROCESSING | 102 | 処理中 |
| RESPONSE_OK | 200 | 成功 |
| RESPONSE_CREATED | 201 | 作成成功 |
| RESPONSE_ACCEPTED | 202 | 受理 |
| RESPONSE_NON_AUTHORITATIVE_INFORMATION | 203 | 非権威情報 |
| RESPONSE_NO_CONTENT | 204 | コンテンツなし |
| RESPONSE_RESET_CONTENT | 205 | コンテンツリセット |
| RESPONSE_PARTIAL_CONTENT | 206 | 部分コンテンツ |
| RESPONSE_MULTI_STATUS | 207 | マルチステータス |
| RESPONSE_ALREADY_REPORTED | 208 | 既報告 |
| RESPONSE_IM_USED | 226 | IM使用 |
| RESPONSE_MULTIPLE_CHOICES | 300 | 複数選択 |
| RESPONSE_MOVED_PERMANENTLY | 301 | 恒久的移動 |
| RESPONSE_FOUND | 302 | 発見 |
| RESPONSE_SEE_OTHER | 303 | 他参照 |
| RESPONSE_NOT_MODIFIED | 304 | 未更新 |
| RESPONSE_USE_PROXY | 305 | プロキシ使用 |
| RESPONSE_SWITCH_PROXY | 306 | プロキシ切替 |
| RESPONSE_TEMPORARY_REDIRECT | 307 | 一時的リダイレクト |
| RESPONSE_PERMANENT_REDIRECT | 308 | 恒久的リダイレクト |
| RESPONSE_BAD_REQUEST | 400 | 不正なリクエスト |
| RESPONSE_UNAUTHORIZED | 401 | 認証エラー |
| RESPONSE_PAYMENT_REQUIRED | 402 | 支払い必要 |
| RESPONSE_FORBIDDEN | 403 | 権限エラー |
| RESPONSE_NOT_FOUND | 404 | リソース未検出 |
| RESPONSE_METHOD_NOT_ALLOWED | 405 | メソッド不許可 |
| RESPONSE_NOT_ACCEPTABLE | 406 | 受入不可 |
| RESPONSE_PROXY_AUTHENTICATION_REQUIRED | 407 | プロキシ認証必要 |
| RESPONSE_REQUEST_TIMEOUT | 408 | リクエストタイムアウト |
| RESPONSE_CONFLICT | 409 | 競合 |
| RESPONSE_GONE | 410 | 消滅 |
| RESPONSE_LENGTH_REQUIRED | 411 | 長さ必要 |
| RESPONSE_PRECONDITION_FAILED | 412 | 前提条件失敗 |
| RESPONSE_REQUEST_ENTITY_TOO_LARGE | 413 | エンティティ過大 |
| RESPONSE_REQUEST_URI_TOO_LONG | 414 | URI過長 |
| RESPONSE_UNSUPPORTED_MEDIA_TYPE | 415 | 非対応メディア型 |
| RESPONSE_REQUESTED_RANGE_NOT_SATISFIABLE | 416 | 範囲不満足 |
| RESPONSE_EXPECTATION_FAILED | 417 | 期待失敗 |
| RESPONSE_IM_A_TEAPOT | 418 | ティーポット |
| RESPONSE_MISDIRECTED_REQUEST | 421 | 誤方向リクエスト |
| RESPONSE_UNPROCESSABLE_ENTITY | 422 | バリデーションエラー |
| RESPONSE_LOCKED | 423 | ロック中 |
| RESPONSE_FAILED_DEPENDENCY | 424 | 依存関係失敗 |
| RESPONSE_UPGRADE_REQUIRED | 426 | アップグレード必要 |
| RESPONSE_PRECONDITION_REQUIRED | 428 | 前提条件必要 |
| RESPONSE_TOO_MANY_REQUESTS | 429 | リクエスト過多 |
| RESPONSE_REQUEST_HEADER_FIELDS_TOO_LARGE | 431 | ヘッダー過大 |
| RESPONSE_UNAVAILABLE_FOR_LEGAL_REASONS | 451 | 法的理由で利用不可 |
| RESPONSE_INTERNAL_SERVER_ERROR | 500 | サーバーエラー |
| RESPONSE_NOT_IMPLEMENTED | 501 | 未実装 |
| RESPONSE_BAD_GATEWAY | 502 | 不正ゲートウェイ |
| RESPONSE_SERVICE_UNAVAILABLE | 503 | サービス利用不可 |
| RESPONSE_GATEWAY_TIMEOUT | 504 | ゲートウェイタイムアウト |
| RESPONSE_HTTP_VERSION_NOT_SUPPORTED | 505 | HTTPバージョン非対応 |
| RESPONSE_VARIANT_ALSO_NEGOTIATES | 506 | バリアント交渉 |
| RESPONSE_INSUFFICIENT_STORAGE | 507 | ストレージ不足 |
| RESPONSE_LOOP_DETECTED | 508 | ループ検出 |
| RESPONSE_NOT_EXTENDED | 510 | 拡張なし |
| RESPONSE_NETWORK_AUTH_REQUIRED | 511 | ネットワーク認証必要 |

#### 3. HTTPClient.get_status

現在の接続ステータスを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPClient` |
| ソースファイル | `core/io/http_client.h:178` |
| メソッド | `get_status()` |
| 戻り値 | `Status` |

**ステータス（Status enum）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| STATUS_DISCONNECTED | 0 | 切断状態 |
| STATUS_RESOLVING | 1 | ホスト名解決中 |
| STATUS_CANT_RESOLVE | 2 | 名前解決失敗 |
| STATUS_CONNECTING | 3 | 接続中 |
| STATUS_CANT_CONNECT | 4 | 接続失敗 |
| STATUS_CONNECTED | 5 | 接続完了 |
| STATUS_REQUESTING | 6 | リクエスト送信中 |
| STATUS_BODY | 7 | ボディ受信中 |
| STATUS_CONNECTION_ERROR | 8 | 接続エラー |
| STATUS_TLS_HANDSHAKE_ERROR | 9 | TLSハンドシェイクエラー |

#### 4. HTTPClient.poll

接続をポーリングして状態を更新する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPClient` |
| ソースファイル | `core/io/http_client.h:194` |
| メソッド | `poll()` |
| 戻り値 | `Error` |

#### 5. HTTPClient.close

接続を閉じる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPClient` |
| ソースファイル | `core/io/http_client.h:176` |
| メソッド | `close()` |
| 戻り値 | `void` |

#### 6. HTTPClient.query_string_from_dict

DictionaryからURLクエリ文字列を生成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPClient` |
| ソースファイル | `core/io/http_client.h:167` |
| メソッド | `query_string_from_dict(dict: Dictionary)` |
| 戻り値 | `String` |

#### 7. HTTPClient.set_http_proxy / set_https_proxy

HTTPプロキシを設定する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPClient` |
| ソースファイル | `core/io/http_client.h:197-198` |
| メソッド | `set_http_proxy(host: String, port: int)` / `set_https_proxy(host: String, port: int)` |
| 戻り値 | `void` |

---

### HTTPRequest API

#### 8. HTTPRequest.request

HTTPリクエストを送信する高レベルAPI（ノードベース）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPRequest` (extends Node) |
| ソースファイル | `scene/main/http_request.h:131` |
| メソッド | `request(url: String, custom_headers: PackedStringArray, method: HTTPClient.Method, request_data: String)` |
| 戻り値 | `Error` |
| シグナル | `request_completed(result, response_code, headers, body)` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| url | String | ○ | - | 完全なURL（http://またはhttps://） |
| custom_headers | PackedStringArray | - | [] | カスタムヘッダー |
| method | HTTPClient.Method | - | METHOD_GET | HTTPメソッド |
| request_data | String | - | "" | リクエストボディ |

**結果コード（Result enum）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| RESULT_SUCCESS | 0 | 成功 |
| RESULT_CHUNKED_BODY_SIZE_MISMATCH | 1 | チャンクサイズ不一致 |
| RESULT_CANT_CONNECT | 2 | 接続失敗 |
| RESULT_CANT_RESOLVE | 3 | 名前解決失敗 |
| RESULT_CONNECTION_ERROR | 4 | 接続エラー |
| RESULT_TLS_HANDSHAKE_ERROR | 5 | TLSエラー |
| RESULT_NO_RESPONSE | 6 | レスポンスなし |
| RESULT_BODY_SIZE_LIMIT_EXCEEDED | 7 | ボディサイズ超過 |
| RESULT_BODY_DECOMPRESS_FAILED | 8 | 解凍失敗 |
| RESULT_REQUEST_FAILED | 9 | リクエスト失敗 |
| RESULT_DOWNLOAD_FILE_CANT_OPEN | 10 | ファイルオープン失敗 |
| RESULT_DOWNLOAD_FILE_WRITE_ERROR | 11 | ファイル書き込みエラー |
| RESULT_REDIRECT_LIMIT_REACHED | 12 | リダイレクト上限 |
| RESULT_TIMEOUT | 13 | タイムアウト |

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| download_file | String | "" | ダウンロード先ファイルパス |
| download_chunk_size | int | 65536 | ダウンロードチャンクサイズ |
| use_threads | bool | false | スレッド使用フラグ |
| accept_gzip | bool | true | gzip圧縮受け入れ |
| body_size_limit | int | -1 | ボディサイズ上限（-1=無制限） |
| max_redirects | int | 8 | 最大リダイレクト回数 |
| timeout | float | 0.0 | タイムアウト秒数（0=無制限） |

**使用例（GDScript）**

```gdscript
func _ready():
    var http_request = HTTPRequest.new()
    add_child(http_request)
    http_request.request_completed.connect(_on_request_completed)

    var error = http_request.request("https://api.example.com/data")
    if error != OK:
        push_error("リクエスト送信失敗")

func _on_request_completed(result, response_code, headers, body):
    if result == HTTPRequest.RESULT_SUCCESS:
        var json = JSON.parse_string(body.get_string_from_utf8())
        print(json)
```

#### 9. HTTPRequest.request_raw

バイナリデータを含むHTTPリクエストを送信する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPRequest` |
| ソースファイル | `scene/main/http_request.h:132` |
| メソッド | `request_raw(url: String, custom_headers: PackedStringArray, method: HTTPClient.Method, request_data_raw: PackedByteArray)` |
| 戻り値 | `Error` |

#### 10. HTTPRequest.cancel_request

進行中のリクエストをキャンセルする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPRequest` |
| ソースファイル | `scene/main/http_request.h:133` |
| メソッド | `cancel_request()` |
| 戻り値 | `void` |

#### 11. HTTPRequest.get_http_client_status

HTTPクライアントの状態を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPRequest` |
| ソースファイル | `scene/main/http_request.h:134` |
| メソッド | `get_http_client_status()` |
| 戻り値 | `HTTPClient.Status` |

#### 12. HTTPRequest.get_downloaded_bytes

ダウンロード済みバイト数を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPRequest` |
| ソースファイル | `scene/main/http_request.h:161` |
| メソッド | `get_downloaded_bytes()` |
| 戻り値 | `int` |

#### 13. HTTPRequest.get_body_size

レスポンスボディの総サイズを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `HTTPRequest` |
| ソースファイル | `scene/main/http_request.h:162` |
| メソッド | `get_body_size()` |
| 戻り値 | `int` |

---

### WebSocket API

#### 14. WebSocketPeer.connect_to_url

WebSocketサーバーに接続する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketPeer` (extends PacketPeer) |
| ソースファイル | `modules/websocket/websocket_peer.h:83` |
| メソッド | `connect_to_url(url: String, tls_options: TLSOptions)` |
| 戻り値 | `Error` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| url | String | ○ | - | WebSocket URL（ws://またはwss://） |
| tls_options | TLSOptions | - | null | TLS設定オプション |

**状態（State enum）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| STATE_CONNECTING | 0 | 接続中 |
| STATE_OPEN | 1 | 接続完了 |
| STATE_CLOSING | 2 | 切断中 |
| STATE_CLOSED | 3 | 切断完了 |

#### 15. WebSocketPeer.accept_stream

既存のStreamPeerをWebSocketとして受け入れる（サーバー側）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketPeer` |
| ソースファイル | `modules/websocket/websocket_peer.h:84` |
| メソッド | `accept_stream(stream: StreamPeer)` |
| 戻り値 | `Error` |

#### 16. WebSocketPeer.send

データを送信する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketPeer` |
| ソースファイル | `modules/websocket/websocket_peer.h:86` |
| メソッド | `send(buffer: PackedByteArray, write_mode: WriteMode)` |
| 戻り値 | `Error` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| buffer | PackedByteArray | ○ | - | 送信データ |
| write_mode | WriteMode | - | WRITE_MODE_BINARY | 書き込みモード |

**書き込みモード（WriteMode enum）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| WRITE_MODE_TEXT | 0 | テキストモード |
| WRITE_MODE_BINARY | 1 | バイナリモード |

#### 17. WebSocketPeer.send_text

テキストデータを送信する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketPeer` |
| ソースファイル | `modules/websocket/websocket_peer.h:102` |
| メソッド | `send_text(text: String)` |
| 戻り値 | `Error` |

#### 18. WebSocketPeer.close

接続を閉じる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketPeer` |
| ソースファイル | `modules/websocket/websocket_peer.h:87` |
| メソッド | `close(code: int, reason: String)` |
| 戻り値 | `void` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| code | int | - | 1000 | 切断コード |
| reason | String | - | "" | 切断理由 |

#### 19. WebSocketPeer.poll

接続をポーリングして状態を更新する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketPeer` |
| ソースファイル | `modules/websocket/websocket_peer.h:97` |
| メソッド | `poll()` |
| 戻り値 | `void` |

#### 20. WebSocketPeer.get_ready_state

現在の接続状態を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketPeer` |
| ソースファイル | `modules/websocket/websocket_peer.h:98` |
| メソッド | `get_ready_state()` |
| 戻り値 | `State` |

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| supported_protocols | PackedStringArray | [] | サポートするサブプロトコル |
| handshake_headers | PackedStringArray | [] | ハンドシェイク時のカスタムヘッダー |
| inbound_buffer_size | int | 65535 | 受信バッファサイズ |
| outbound_buffer_size | int | 65535 | 送信バッファサイズ |
| max_queued_packets | int | 4096 | 最大キューパケット数 |
| heartbeat_interval | float | 0.0 | ハートビート間隔（秒） |

---

### WebSocketMultiplayerPeer API

#### 21. WebSocketMultiplayerPeer.create_server

WebSocketサーバーを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketMultiplayerPeer` (extends MultiplayerPeer) |
| ソースファイル | `modules/websocket/websocket_multiplayer_peer.h:115` |
| メソッド | `create_server(port: int, bind_ip: IPAddress, tls_options: TLSOptions)` |
| 戻り値 | `Error` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| port | int | ○ | - | リッスンポート |
| bind_ip | IPAddress | - | "*" | バインドIPアドレス |
| tls_options | TLSOptions | - | null | TLS設定オプション |

#### 22. WebSocketMultiplayerPeer.create_client

WebSocketクライアントを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketMultiplayerPeer` |
| ソースファイル | `modules/websocket/websocket_multiplayer_peer.h:114` |
| メソッド | `create_client(url: String, tls_options: TLSOptions)` |
| 戻り値 | `Error` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| url | String | ○ | - | WebSocket URL |
| tls_options | TLSOptions | - | null | TLS設定オプション |

#### 23. WebSocketMultiplayerPeer.get_peer

指定したピアIDのWebSocketPeerを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketMultiplayerPeer` |
| ソースファイル | `modules/websocket/websocket_multiplayer_peer.h:112` |
| メソッド | `get_peer(peer_id: int)` |
| 戻り値 | `WebSocketPeer` |

#### 24. WebSocketMultiplayerPeer.get_peer_address / get_peer_port

指定したピアのアドレス/ポートを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `WebSocketMultiplayerPeer` |
| ソースファイル | `modules/websocket/websocket_multiplayer_peer.h:132-133` |
| メソッド | `get_peer_address(peer_id: int)` / `get_peer_port(peer_id: int)` |
| 戻り値 | `IPAddress` / `int` |

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| supported_protocols | PackedStringArray | [] | サポートするサブプロトコル |
| handshake_headers | PackedStringArray | [] | ハンドシェイク時のカスタムヘッダー |
| inbound_buffer_size | int | 65535 | 受信バッファサイズ |
| outbound_buffer_size | int | 65535 | 送信バッファサイズ |
| handshake_timeout | float | 3.0 | ハンドシェイクタイムアウト（秒） |
| max_queued_packets | int | 4096 | 最大キューパケット数 |

---

### JSON-RPC API

#### 25. JSONRPC.set_method

JSON-RPCメソッドを登録する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSONRPC` (extends Object) |
| ソースファイル | `modules/jsonrpc/jsonrpc.h:69` |
| メソッド | `set_method(name: String, callback: Callable)` |
| 戻り値 | `void` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | String | ○ | メソッド名 |
| callback | Callable | ○ | コールバック関数 |

#### 26. JSONRPC.process_action

JSON-RPCアクションを処理する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSONRPC` |
| ソースファイル | `modules/jsonrpc/jsonrpc.h:66` |
| メソッド | `process_action(action: Variant, recurse: bool)` |
| 戻り値 | `Variant` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| action | Variant | ○ | - | JSON-RPCリクエスト（DictionaryまたはArray） |
| recurse | bool | - | false | 配列要素を再帰処理するか |

**リクエスト形式**

```json
{
  "jsonrpc": "2.0",
  "method": "method_name",
  "params": ["param1", "param2"],
  "id": 1
}
```

**レスポンス形式（成功時）**

```json
{
  "jsonrpc": "2.0",
  "id": 1,
  "result": "return_value"
}
```

**レスポンス形式（エラー時）**

```json
{
  "jsonrpc": "2.0",
  "id": 1,
  "error": {
    "code": -32601,
    "message": "Method not found"
  }
}
```

**エラーコード（ErrorCode enum）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| PARSE_ERROR | -32700 | JSONパースエラー |
| INVALID_REQUEST | -32600 | 不正なリクエスト |
| METHOD_NOT_FOUND | -32601 | メソッド未検出 |
| INVALID_PARAMS | -32602 | 不正なパラメータ |
| INTERNAL_ERROR | -32603 | 内部エラー |

#### 27. JSONRPC.process_string

JSON文字列を処理する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSONRPC` |
| ソースファイル | `modules/jsonrpc/jsonrpc.h:67` |
| メソッド | `process_string(input: String)` |
| 戻り値 | `String` |

#### 28. JSONRPC.make_request

JSON-RPCリクエストを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSONRPC` |
| ソースファイル | `modules/jsonrpc/jsonrpc.h:64` |
| メソッド | `make_request(method: String, params: Variant, id: Variant)` |
| 戻り値 | `Dictionary` |

#### 29. JSONRPC.make_response

JSON-RPCレスポンスを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSONRPC` |
| ソースファイル | `modules/jsonrpc/jsonrpc.h:62` |
| メソッド | `make_response(result: Variant, id: Variant)` |
| 戻り値 | `Dictionary` |

#### 30. JSONRPC.make_notification

JSON-RPC通知を作成する（idなしのリクエスト）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSONRPC` |
| ソースファイル | `modules/jsonrpc/jsonrpc.h:63` |
| メソッド | `make_notification(method: String, params: Variant)` |
| 戻り値 | `Dictionary` |

#### 31. JSONRPC.make_response_error

JSON-RPCエラーレスポンスを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSONRPC` |
| ソースファイル | `modules/jsonrpc/jsonrpc.h:61` |
| メソッド | `make_response_error(code: int, message: String, id: Variant)` |
| 戻り値 | `Dictionary` |

**使用例（GDScript）**

```gdscript
var jsonrpc = JSONRPC.new()

# メソッドを登録
jsonrpc.set_method("add", func(a, b): return a + b)

# リクエストを処理
var request = jsonrpc.make_request("add", [1, 2], 1)
var response = jsonrpc.process_action(request)
print(response)  # {"jsonrpc": "2.0", "id": 1, "result": 3}

# 文字列として処理
var json_str = '{"jsonrpc": "2.0", "method": "add", "params": [3, 4], "id": 2}'
var result = jsonrpc.process_string(json_str)
print(result)  # {"jsonrpc": "2.0", "id": 2, "result": 7}
```

---

### JSON API

#### 32. JSON.parse

JSON文字列をパースする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSON` (extends Resource) |
| ソースファイル | `core/io/json.h:89` |
| メソッド | `parse(json_string: String, keep_text: bool)` |
| 戻り値 | `Error` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| json_string | String | ○ | - | JSON文字列 |
| keep_text | bool | - | false | 元のテキストを保持するか |

#### 33. JSON.stringify（静的メソッド）

VariantをJSON文字列に変換する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSON` |
| ソースファイル | `core/io/json.h:92` |
| メソッド | `stringify(data: Variant, indent: String, sort_keys: bool, full_precision: bool)` |
| 戻り値 | `String` |
| 静的 | ○ |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| data | Variant | ○ | - | シリアライズするデータ |
| indent | String | - | "" | インデント文字列 |
| sort_keys | bool | - | true | キーをソートするか |
| full_precision | bool | - | false | 浮動小数点の完全精度 |

#### 34. JSON.parse_string（静的メソッド）

JSON文字列をパースしてVariantを返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSON` |
| ソースファイル | `core/io/json.h:93` |
| メソッド | `parse_string(json_string: String)` |
| 戻り値 | `Variant` |
| 静的 | ○ |

#### 35. JSON.from_native（静的メソッド）

GodotネイティブオブジェクトをJSON互換形式に変換する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSON` |
| ソースファイル | `core/io/json.h:95-97` |
| メソッド | `from_native(variant: Variant, full_objects: bool)` |
| 戻り値 | `Variant` |
| 静的 | ○ |

#### 36. JSON.to_native（静的メソッド）

JSON互換形式をGodotネイティブオブジェクトに変換する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `JSON` |
| ソースファイル | `core/io/json.h:98-100` |
| メソッド | `to_native(json: Variant, allow_objects: bool)` |
| 戻り値 | `Variant` |
| 静的 | ○ |

**プロパティ/メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| get_data() | Variant | パース結果データを取得 |
| set_data(data) | void | データを設定 |
| get_error_line() | int | エラー行番号を取得 |
| get_error_message() | String | エラーメッセージを取得 |
| get_parsed_text() | String | パース済みテキストを取得 |

**使用例（GDScript）**

```gdscript
# パース
var json_str = '{"name": "Godot", "version": 4}'
var data = JSON.parse_string(json_str)
print(data.name)  # "Godot"

# シリアライズ
var dict = {"a": 1, "b": [2, 3]}
var result = JSON.stringify(dict, "\t")
print(result)

# インスタンスを使用
var json = JSON.new()
var err = json.parse('{"key": "value"}')
if err == OK:
    print(json.get_data())
else:
    print("Error at line ", json.get_error_line(), ": ", json.get_error_message())
```

---

### GDExtension API

#### 37. GDExtension.open_library

GDExtensionライブラリを開く。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtension` (extends Resource) |
| ソースファイル | `core/extension/gdextension.h:139` |
| メソッド | `open_library(path: String, loader: GDExtensionLoader)` |
| 戻り値 | `Error` |

**初期化レベル（InitializationLevel enum）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| INITIALIZATION_LEVEL_CORE | 0 | コアレベル |
| INITIALIZATION_LEVEL_SERVERS | 1 | サーバーレベル |
| INITIALIZATION_LEVEL_SCENE | 2 | シーンレベル |
| INITIALIZATION_LEVEL_EDITOR | 3 | エディタレベル |

#### 38. GDExtension.close_library

GDExtensionライブラリを閉じる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtension` |
| ソースファイル | `core/extension/gdextension.h:140` |
| メソッド | `close_library()` |
| 戻り値 | `void` |

#### 39. GDExtension.is_library_open

ライブラリが開いているかを確認する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtension` |
| ソースファイル | `core/extension/gdextension.h:141` |
| メソッド | `is_library_open()` |
| 戻り値 | `bool` |

#### 40. GDExtension.initialize_library

指定レベルでライブラリを初期化する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtension` |
| ソースファイル | `core/extension/gdextension.h:172` |
| メソッド | `initialize_library(level: InitializationLevel)` |
| 戻り値 | `void` |

#### 41. GDExtension.deinitialize_library

指定レベルでライブラリを非初期化する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtension` |
| ソースファイル | `core/extension/gdextension.h:173` |
| メソッド | `deinitialize_library(level: InitializationLevel)` |
| 戻り値 | `void` |

#### 42. GDExtension.get_minimum_library_initialization_level

ライブラリの最小初期化レベルを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtension` |
| ソースファイル | `core/extension/gdextension.h:171` |
| メソッド | `get_minimum_library_initialization_level()` |
| 戻り値 | `InitializationLevel` |

---

### GDExtensionManager API

#### 43. GDExtensionManager.load_extension

GDExtensionを読み込む。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtensionManager` (extends Object) |
| ソースファイル | `core/extension/gdextension_manager.h:71` |
| メソッド | `load_extension(path: String)` |
| 戻り値 | `LoadStatus` |
| シングルトン | ○ |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| path | String | ○ | .gdextensionファイルのパス |

**読み込みステータス（LoadStatus enum）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| LOAD_STATUS_OK | 0 | 読み込み成功 |
| LOAD_STATUS_FAILED | 1 | 読み込み失敗 |
| LOAD_STATUS_ALREADY_LOADED | 2 | 既に読み込み済み |
| LOAD_STATUS_NOT_LOADED | 3 | 未読み込み |
| LOAD_STATUS_NEEDS_RESTART | 4 | 再起動が必要 |

#### 44. GDExtensionManager.load_extension_with_loader

カスタムローダーでGDExtensionを読み込む。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtensionManager` |
| ソースファイル | `core/extension/gdextension_manager.h:73` |
| メソッド | `load_extension_with_loader(path: String, loader: GDExtensionLoader)` |
| 戻り値 | `LoadStatus` |

#### 45. GDExtensionManager.reload_extension

GDExtensionを再読み込みする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtensionManager` |
| ソースファイル | `core/extension/gdextension_manager.h:74` |
| メソッド | `reload_extension(path: String)` |
| 戻り値 | `LoadStatus` |

#### 46. GDExtensionManager.unload_extension

GDExtensionをアンロードする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtensionManager` |
| ソースファイル | `core/extension/gdextension_manager.h:75` |
| メソッド | `unload_extension(path: String)` |
| 戻り値 | `LoadStatus` |

#### 47. GDExtensionManager.is_extension_loaded

GDExtensionが読み込み済みかを確認する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtensionManager` |
| ソースファイル | `core/extension/gdextension_manager.h:76` |
| メソッド | `is_extension_loaded(path: String)` |
| 戻り値 | `bool` |

#### 48. GDExtensionManager.get_loaded_extensions

読み込み済みのGDExtension一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtensionManager` |
| ソースファイル | `core/extension/gdextension_manager.h:77` |
| メソッド | `get_loaded_extensions()` |
| 戻り値 | `PackedStringArray` |

#### 49. GDExtensionManager.get_extension

指定パスのGDExtensionを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `GDExtensionManager` |
| ソースファイル | `core/extension/gdextension_manager.h:78` |
| メソッド | `get_extension(path: String)` |
| 戻り値 | `GDExtension` |

---

## 備考

### プラットフォーム固有の実装

- **HTTPClientTCP** (`core/io/http_client_tcp.h`): TCP接続ベースのHTTPクライアント実装
- **HTTPClientWeb** (`platform/web/http_client_web.h`): Web(JavaScript)プラットフォーム向け実装

### 関連クラス

- **TLSOptions**: TLS接続設定用クラス
- **StreamPeer**: ストリームデータ送受信の基底クラス
- **PacketPeer**: パケットデータ送受信の基底クラス
- **MultiplayerPeer**: マルチプレイヤー通信の基底クラス
- **IPAddress**: IPアドレスを表すクラス
- **RefCounted**: 参照カウント付きオブジェクトの基底クラス
- **Resource**: リソースオブジェクトの基底クラス
- **Node**: シーンツリーノードの基底クラス

### デフォルト定数

| 定数 | 値 | 説明 |
| --- | --- | --- |
| PORT_HTTP | 80 | HTTPのデフォルトポート |
| PORT_HTTPS | 443 | HTTPSのデフォルトポート |
| DEFAULT_BUFFER_SIZE | 65535 | WebSocketのデフォルトバッファサイズ |
| HOST_MIN_LEN | 4 | ホスト名の最小長 |

### 参考資料

- Godot公式ドキュメント: https://docs.godotengine.org/
- GDExtension API: https://docs.godotengine.org/en/stable/tutorials/scripting/gdextension/
- HTTPClient: https://docs.godotengine.org/en/stable/classes/class_httpclient.html
- HTTPRequest: https://docs.godotengine.org/en/stable/classes/class_httprequest.html
- WebSocketPeer: https://docs.godotengine.org/en/stable/classes/class_websocketpeer.html
- JSON: https://docs.godotengine.org/en/stable/classes/class_json.html
- JSONRPC: https://docs.godotengine.org/en/stable/classes/class_jsonrpc.html
