# アーキテクチャ設計書

## 概要

本ドキュメントはGodot Engine 4.6のアーキテクチャ設計について記載する。Godotは2Dおよび3Dゲーム開発のためのクロスプラットフォームゲームエンジンであり、C++で実装されている。MIT Licenseのもとオープンソースで開発されており、多様なプラットフォームへのエクスポートをサポートする。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| Vulkan/Direct3D12/Metal/OpenGL | グラフィックスAPI呼び出し | レンダリング |
| OS API (Windows/macOS/Linux/Android/iOS/Web) | プラットフォーム固有API | ファイルシステム、入力、ウィンドウ管理 |
| 物理エンジン (Jolt Physics/Godot Physics) | 内部統合 | 2D/3D物理シミュレーション |
| オーディオAPI (WASAPI/CoreAudio/PulseAudio/ALSA) | プラットフォーム固有API | 音声再生 |
| ネットワーク (ENet/WebSocket/WebRTC) | ライブラリ統合 | マルチプレイヤー通信 |
| GDExtension | バイナリインターフェース | サードパーティ拡張 |

## レイヤー構成

### アーキテクチャスタイル

**コンポーネントベースアーキテクチャ + レイヤードアーキテクチャ**

Godotは以下の特徴を持つアーキテクチャを採用している：
- **シーングラフ構造**: ノード（Node）のツリー構造でゲームシーンを表現
- **サーバーパターン**: レンダリング、物理、オーディオなどの重い処理をサーバープロセスとして分離
- **モジュラー設計**: 機能をモジュール単位で分離し、コンパイル時に有効/無効を切り替え可能

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Editor | ゲーム開発環境の提供 | EditorNode, EditorInterface, Inspector, SceneTree編集 |
| Scene | シーングラフとノードの管理 | Node, Node2D, Node3D, Control, SceneTree |
| Servers | バックエンド処理の実行 | RenderingServer, PhysicsServer, AudioServer, DisplayServer |
| Modules | 拡張機能の提供 | GDScript, Mono, Navigation, Multiplayer, 各種フォーマット対応 |
| Drivers | プラットフォーム抽象化 | Vulkan, GLES3, Metal, D3D12, Audio Drivers |
| Core | エンジンの基盤機能 | Object, Variant, Resource, IO, Math, Templates |
| Platform | OS固有実装 | Windows, macOS, Linux/BSD, Android, iOS, Web |

### レイヤー間の依存関係ルール

```
[Editor] → [Scene] → [Servers] → [Core]
                 ↘    ↗
              [Modules]
                  ↓
[Drivers] → [Core] ← [Platform]
```

- **上位レイヤーは下位レイヤーを参照可能**: Editor → Scene → Servers → Core
- **Coreは他のレイヤーに依存しない**: 最も基盤となるレイヤー
- **Driversは抽象化レイヤーを提供**: 上位レイヤーからは抽象インターフェース経由でアクセス
- **Platformは実行環境を抽象化**: OSの差異をCoreに吸収
- **Modulesは選択的依存**: 各モジュールは必要なレイヤーのみに依存

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| スクリプティング | ゲームロジック記述 | gdscript, mono |
| グラフィックス | 2D/3D描画 | servers/rendering, betsy, lightmapper_rd |
| 物理 | 衝突検出・シミュレーション | godot_physics_2d, godot_physics_3d, jolt_physics |
| オーディオ | 音声再生・処理 | servers/audio, interactive_music, ogg, vorbis, mp3 |
| ネットワーク | 通信機能 | multiplayer, enet, websocket, webrtc |
| UI | ユーザーインターフェース | scene/gui, text_server_adv, text_server_fb |
| インポート/エクスポート | アセット変換 | gltf, fbx, svg, 各種画像フォーマット |
| ナビゲーション | AI経路探索 | navigation_2d, navigation_3d |
| XR | VR/AR対応 | openxr, webxr, mobile_vr |

### パッケージ構造

```
godot-master/
├── core/                       # コアレイヤー
│   ├── config/                 # エンジン・プロジェクト設定
│   ├── crypto/                 # 暗号化機能
│   ├── debugger/               # デバッグ機能
│   ├── error/                  # エラー処理
│   ├── extension/              # GDExtension基盤
│   ├── input/                  # 入力システム
│   ├── io/                     # ファイル・ネットワークI/O
│   ├── math/                   # 数学ライブラリ
│   ├── object/                 # オブジェクトシステム (Object, ClassDB)
│   ├── os/                     # OS抽象化
│   ├── profiling/              # プロファイリング
│   ├── string/                 # 文字列処理
│   ├── templates/              # テンプレートコンテナ
│   └── variant/                # Variant型システム
├── servers/                    # サーバーレイヤー
│   ├── audio/                  # オーディオサーバー
│   ├── camera/                 # カメラサーバー
│   ├── display/                # ディスプレイサーバー
│   ├── movie_writer/           # 動画出力
│   ├── navigation_2d/          # 2Dナビゲーション
│   ├── navigation_3d/          # 3Dナビゲーション
│   ├── physics_2d/             # 2D物理サーバー
│   ├── physics_3d/             # 3D物理サーバー
│   ├── rendering/              # レンダリングサーバー
│   ├── text/                   # テキストサーバー
│   └── xr/                     # XRサーバー
├── scene/                      # シーンレイヤー
│   ├── 2d/                     # 2Dノード
│   ├── 3d/                     # 3Dノード
│   ├── animation/              # アニメーションシステム
│   ├── audio/                  # オーディオノード
│   ├── gui/                    # UIコントロール
│   ├── main/                   # SceneTree, Window
│   ├── resources/              # リソースクラス
│   └── theme/                  # テーマシステム
├── modules/                    # モジュールレイヤー (60+モジュール)
│   ├── gdscript/               # GDScriptスクリプト言語
│   ├── mono/                   # C#サポート
│   ├── multiplayer/            # マルチプレイヤー機能
│   └── ...                     # その他多数のモジュール
├── drivers/                    # ドライバーレイヤー
│   ├── gles3/                  # OpenGL ES 3.0
│   ├── vulkan/                 # Vulkan
│   ├── metal/                  # Metal (Apple)
│   ├── d3d12/                  # Direct3D 12
│   └── ...                     # オーディオ、その他ドライバー
├── editor/                     # エディタレイヤー
│   ├── animation/              # アニメーションエディタ
│   ├── debugger/               # デバッガー
│   ├── export/                 # エクスポート機能
│   ├── gui/                    # エディタUI
│   ├── import/                 # インポート機能
│   ├── inspector/              # インスペクター
│   ├── plugins/                # エディタプラグイン
│   └── project_manager/        # プロジェクトマネージャー
├── platform/                   # プラットフォームレイヤー
│   ├── windows/                # Windows実装
│   ├── macos/                  # macOS実装
│   ├── linuxbsd/               # Linux/BSD実装
│   ├── android/                # Android実装
│   ├── ios/                    # iOS実装
│   ├── visionos/               # visionOS実装
│   └── web/                    # Web (HTML5)実装
├── main/                       # エンジンエントリポイント
├── thirdparty/                 # サードパーティライブラリ
└── tests/                      # テストコード
```

### コンポーネント依存関係

主要コンポーネント間の依存関係は以下の通り：

1. **Object → ClassDB**: 全クラスの基底クラス、リフレクションシステム
2. **Node → Object**: シーングラフの基本単位
3. **SceneTree → Node**: ノードツリーの管理
4. **RenderingServer → Object**: レンダリングコマンドの受付・実行
5. **PhysicsServer → Object**: 物理シミュレーションの実行
6. **GDScript → ScriptLanguage → Object**: スクリプト言語実装

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| 内部DB | 独自実装 | - | リソースUID管理、エディタキャッシュ |
| 構造化データ | JSON/Binary | - | プロジェクト設定、シーンファイル |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| ResourceCache | 内部実装 | リソースの重複ロード防止 | セッション中 |
| ShaderCache | 内部実装 | コンパイル済みシェーダー | 永続 |
| ImportCache | 内部実装 | インポート済みアセット | 永続 |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| MessageQueue | 内部実装 | スレッド間オブジェクトメッセージング |
| CommandQueueMT | 内部実装 | レンダリングコマンドの非同期実行 |
| CallQueue | 内部実装 | ノード処理グループ管理 |

### 検索エンジン

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| FileSystemDock | 内部実装 | エディタ内ファイル検索 |
| EditorHelp | 内部実装 | ドキュメント検索 |

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Vulkan | 1.3+ | 3Dレンダリング (主要バックエンド) |
| OpenGL ES 3.0 | 3.0 | 互換性レンダリング |
| Metal | - | macOS/iOSレンダリング |
| Direct3D 12 | 12 | Windows高性能レンダリング |
| mbedTLS | 統合版 | TLS/暗号化 |
| zlib/zstd | 統合版 | データ圧縮 |
| FreeType | 統合版 | フォントレンダリング |
| HarfBuzz | 統合版 | テキストシェーピング |
| ICU | 統合版 | 国際化対応 |
| ENet | 統合版 | 低レイテンシネットワーク |

## データフロー

### リクエスト処理の流れ

ゲームフレームの処理フロー：

1. **OS::run()**: プラットフォーム固有のメインループ開始
2. **Main::iteration()**: フレーム処理のエントリポイント
3. **Input処理**: InputMapによる入力イベントの変換
4. **SceneTree::process()**: ノードの_process()呼び出し
5. **PhysicsServer::step()**: 物理シミュレーション実行
6. **SceneTree::physics_process()**: ノードの_physics_process()呼び出し
7. **RenderingServer::draw()**: シーンの描画コマンド発行
8. **DisplayServer::swap_buffers()**: フレームバッファのスワップ

### 非同期処理の流れ

レンダリングの非同期処理（RenderingServerDefault）：

1. **メインスレッド**: draw()呼び出しでコマンドをキューに追加
2. **CommandQueueMT**: コマンドをスレッドセーフに蓄積
3. **レンダースレッド**: _thread_loop()でコマンドを順次実行
4. **GPU同期**: フェンスによるGPU完了待機
5. **完了通知**: フレームコールバックの実行

リソースの非同期ロード（ResourceLoader）：

1. **load_threaded_request()**: ロード要求をキューに追加
2. **WorkerThreadPool**: バックグラウンドスレッドでロード実行
3. **load_threaded_get_status()**: ロード状況のポーリング
4. **load_threaded_get()**: 完了したリソースの取得

### データ永続化の流れ

リソースの保存処理：

1. **ResourceSaver::save()**: 保存要求
2. **フォーマット判定**: 拡張子からResourceFormatSaverを選択
3. **バイナリ/テキスト変換**: Variantのシリアライズ
4. **ファイル書き込み**: FileAccessによる書き込み
5. **UID更新**: ResourceUIDへの登録

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| 暗号化リソース | core/crypto | 有料アセット保護 |
| PCK暗号化 | core/io/file_access_pack | エクスポートビルド |
| エディタ秘密情報 | PROPERTY_USAGE_SECRET | API Key等の保護 |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| エンジンログ | print_line(), print_error() | stdout/stderr, エディタログパネル |
| スクリプトログ | print(), push_error() | stdout/stderr, リモートデバッグ |
| プロファイリング | EngineProfiler | エディタ/外部ツール |
| デバッガー通信 | EngineDebugger | TCP/DAP |

### エラーハンドリング

エラーハンドリングの方針と実装箇所：

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| 致命的エラー | CRASH_NOW(), CRASH_NOW_MSG() | プロセス終了 |
| 回復可能エラー | ERR_FAIL_COND_V(), ERR_FAIL_V() | エラー出力+戻り値 |
| 警告 | WARN_PRINT() | 警告出力のみ |
| デバッグ用 | DEV_ASSERT() | 開発ビルドのみアサート |
| スクリプトエラー | push_error() | スタックトレース+継続 |

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| Undo/Redo | UndoRedo, EditorUndoRedoManager | アクション単位 |
| リソース変更 | Resource::_resource_saved_changed() | リソース単位 |
| シーン変更 | SceneTree通知 | シーン単位 |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 継承よりコンポジション | Node + スクリプト | ノードにスクリプトをアタッチ |
| サーバー分離 | RenderingServer, PhysicsServer | メインスレッドとバックエンドの分離 |
| リソース共有 | RefCounted, Ref<T> | 参照カウントによるメモリ管理 |
| プラットフォーム抽象化 | OS, DisplayServer | 仮想関数によるインターフェース統一 |
| モジュール化 | modules/ | コンパイル時有効/無効切り替え |
| シグナル/スロット | Object::emit_signal() | イベント駆動設計 |

### コーディング規約

- **命名規則**: snake_case（関数・変数）、PascalCase（クラス）、UPPER_CASE（定数・マクロ）
- **インデント**: タブ文字使用
- **括弧**: K&Rスタイル（開き括弧は同一行）
- **ヘッダーガード**: `#pragma once`使用
- **Godot独自マクロ**: GDCLASS(), _FORCE_INLINE_, GDREGISTER_CLASS()等
- **エラー処理**: ERR_FAIL_* マクロによる統一的なエラー報告
- **ドキュメント**: XMLフォーマットのクラスリファレンス（doc/classes/）

## 備考

- 本ドキュメントはGodot Engine 4.6 RC版のソースコードを基に作成
- ビルドシステムはSCons (Python)を使用
- thirdparty/には60以上のサードパーティライブラリが統合されている
- コンパイル時オプションにより機能の有効/無効を細かく制御可能
