# Godot Engine コードリーディングガイドライン

## はじめに

このガイドラインは、Godot Engine のコードベースを効率的に理解するための手引きです。
C++ に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、C++ の基本構文と Godot 固有の概念を解説します。

### 1.1 プログラム構造

Godot Engine は C++17 を使用して開発されています。各ファイルは以下の構造を持ちます。

**ヘッダファイル構造（.h）:**
```cpp
// ファイル: core/object/object.h:31-42

#pragma once  // インクルードガード

#include "core/extension/gdextension_interface.gen.h"
#include "core/object/gdtype.h"
#include "core/object/message_queue.h"
// ... その他のインクルード

template <typename T>
class TypedArray;

template <typename T>
class Ref;
```

**ソースファイル構造（.cpp）:**
```cpp
// ファイル: main/main.cpp:31-79

#include "main.h"

#include "core/config/project_settings.h"
#include "core/core_globals.h"
// ... その他のインクルード

/* Static members */

// Singletons
static Engine *engine = nullptr;
static ProjectSettings *globals = nullptr;
```

### 1.2 データ型と変数

Godot は独自の型システムを持ち、`Variant` クラスが動的型として機能します。

**Variant 型の定義:**
```cpp
// ファイル: core/variant/variant.h:93-146

class Variant {
public:
	enum Type {
		NIL,

		// atomic types
		BOOL,
		INT,
		FLOAT,
		STRING,

		// math types
		VECTOR2,
		VECTOR2I,
		RECT2,
		// ... 続く

		// misc types
		COLOR,
		STRING_NAME,
		NODE_PATH,
		RID,
		OBJECT,
		CALLABLE,
		SIGNAL,
		DICTIONARY,
		ARRAY,

		// typed arrays
		PACKED_BYTE_ARRAY,
		// ... 続く

		VARIANT_MAX
	};
};
```

**型エイリアス:**
```cpp
// ファイル: core/variant/variant.h:81-91

typedef Vector<uint8_t> PackedByteArray;
typedef Vector<int32_t> PackedInt32Array;
typedef Vector<int64_t> PackedInt64Array;
typedef Vector<float> PackedFloat32Array;
typedef Vector<double> PackedFloat64Array;
typedef Vector<String> PackedStringArray;
typedef Vector<Vector2> PackedVector2Array;
typedef Vector<Vector3> PackedVector3Array;
```

### 1.3 制御構造

Godot では標準的な C++ の制御構造に加え、マクロベースの制御パターンを使用します。

**プリプロセッサによる条件コンパイル:**
```cpp
// ファイル: main/main.cpp:82-106

// 2D
#ifndef NAVIGATION_2D_DISABLED
#include "servers/navigation_2d/navigation_server_2d.h"
#endif // NAVIGATION_2D_DISABLED

#ifndef PHYSICS_2D_DISABLED
#include "servers/physics_2d/physics_server_2d.h"
#endif // PHYSICS_2D_DISABLED

// 3D
#ifndef NAVIGATION_3D_DISABLED
#include "servers/navigation_3d/navigation_server_3d.h"
#endif // NAVIGATION_3D_DISABLED

#ifdef TOOLS_ENABLED
#include "editor/debugger/editor_debugger_node.h"
#endif // TOOLS_ENABLED
```

### 1.4 関数/メソッド定義

Godot では、getter/setter パターンと `_bind_methods()` によるスクリプトバインディングが一般的です。

**メソッド定義パターン:**
```cpp
// ファイル: scene/2d/node_2d.h:78-95

void set_position(const Point2 &p_pos);
void set_rotation(real_t p_radians);
void set_rotation_degrees(real_t p_degrees);
void set_skew(real_t p_radians);
void set_scale(const Size2 &p_scale);

void rotate(real_t p_radians);
void move_x(real_t p_delta, bool p_scaled = false);
void move_y(real_t p_delta, bool p_scaled = false);

Point2 get_position() const;
real_t get_rotation() const;
real_t get_rotation_degrees() const;
```

**スクリプトバインディング用マクロ:**
```cpp
// ファイル: core/object/object.h:135-143

#define ADD_SIGNAL(m_signal) ::ClassDB::add_signal(get_class_static(), m_signal)
#define ADD_PROPERTY(m_property, m_setter, m_getter) ::ClassDB::add_property(get_class_static(), m_property, StringName(m_setter), StringName(m_getter))
#define ADD_GROUP(m_name, m_prefix) ::ClassDB::add_property_group(get_class_static(), m_name, m_prefix)
#define ADD_SUBGROUP(m_name, m_prefix) ::ClassDB::add_property_subgroup(get_class_static(), m_name, m_prefix)
```

### 1.5 モジュール/インクルード

Godot では相対パスベースのインクルードシステムを使用します。

**インクルードパターン:**
```cpp
// ファイル: scene/main/node.h:33-39

#include "core/input/input_event.h"
#include "core/io/resource.h"
#include "core/string/node_path.h"
#include "core/templates/iterable.h"
#include "core/variant/typed_array.h"
#include "scene/main/scene_tree.h"
#include "scene/scene_string_names.h"
```

---

## 2. プロジェクト固有の概念

> このセクションでは、Godot Engine 特有の概念を解説します。

### 2.1 オブジェクトシステム

Godot の全てのクラスは `Object` クラスから継承され、`GDCLASS` マクロで宣言されます。

**クラス宣言パターン:**
```cpp
// ファイル: scene/main/node.h:50-51

class Node : public Object {
	GDCLASS(Node, Object);
```

```cpp
// ファイル: scene/2d/node_2d.h:35-36

class Node2D : public CanvasItem {
	GDCLASS(Node2D, CanvasItem);
```

```cpp
// ファイル: modules/gdscript/gdscript.h:60-61

class GDScript : public Script {
	GDCLASS(GDScript, Script);
```

**GDCLASS マクロの意味:**
- 第1引数: 現在のクラス名
- 第2引数: 親クラス名
- ランタイム型情報（RTTI）と反射機能を提供

### 2.2 シグナル・スロットシステム

Godot は独自のシグナル・スロット機構を持ち、オブジェクト間の通信に使用します。

```cpp
// ファイル: core/object/object.h:109 のシグナルマップ参照

HashMap<StringName, MethodInfo> _signals;
```

### 2.3 リソースシステム

`Resource` クラスはシリアライズ可能なデータの基底クラスです。

```cpp
// ファイル: core/io/resource.h:53-60

class Resource : public RefCounted {
	GDCLASS(Resource, RefCounted);

public:
	static constexpr AncestralClass static_ancestral_class = AncestralClass::RESOURCE;

	static void register_custom_data_to_otdb() { ClassDB::add_resource_base_extension("res", get_class_static()); }
	virtual String get_base_extension() const { return "res"; }
```

### 2.4 参照カウント

`RefCounted` を継承したクラスは自動的にメモリ管理されます。`Ref<T>` テンプレートでスマートポインタとして扱います。

```cpp
// ファイル: modules/gdscript/gdscript.h:94-96

Ref<GDScriptNativeClass> native;
Ref<GDScript> base;
GDScript *_owner = nullptr; //for subclasses
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `snake_case.h/.cpp` | 全てのソースファイル | `node_2d.h`, `scene_tree.cpp` |
| `register_*_types.cpp` | 型登録用ファイル | `register_core_types.cpp`, `register_scene_types.cpp` |
| `*_server.h/.cpp` | サーバーシングルトン | `rendering_server.h`, `audio_server.cpp` |
| `*_bind.cpp/.h` | スクリプトバインディング | `core_bind.cpp` |
| `*.gen.h/.cpp` | 自動生成ファイル | `gdextension_interface.gen.h` |
| `*.compat.inc` | 互換性コード | `core_bind.compat.inc` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `p_` | 関数パラメータ | `p_name`, `p_value`, `p_pos` |
| `r_` | 参照パラメータ（出力） | `r_error`, `r_result` |
| `m_` | マクロパラメータ | `m_class`, `m_property` |
| `_` (先頭) | プライベートメソッド | `_bind_methods()`, `_notification()` |
| `*Server` | シングルトンサーバークラス | `RenderingServer`, `AudioServer` |
| `*2D` / `*3D` | 2D/3D 専用クラス | `Node2D`, `Node3D`, `PhysicsServer2D` |
| `GD*` | GDScript関連 | `GDScript`, `GDScriptFunction` |
| `Editor*` | エディタ専用 | `EditorNode`, `EditorPlugin` |

### 3.3 マクロ命名規則

| パターン | 意味 | 例 |
|---------|------|-----|
| `GDCLASS(...)` | クラス登録マクロ | `GDCLASS(Node, Object)` |
| `GDREGISTER_CLASS(...)` | クラスDBへの登録 | `GDREGISTER_CLASS(Object)` |
| `_FORCE_INLINE_` | 強制インライン化 | メソッド定義前に付与 |
| `PROPERTY_HINT_*` | プロパティヒント定数 | `PROPERTY_HINT_RANGE` |
| `PROPERTY_USAGE_*` | プロパティ用途フラグ | `PROPERTY_USAGE_DEFAULT` |

### 3.4 定数・列挙型命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `SCREAMING_SNAKE_CASE` | 定数・列挙値 | `PROCESS_MODE_INHERIT`, `VARIANT_MAX` |
| `PascalCase` | 列挙型名 | `ProcessMode`, `Type`, `PropertyHint` |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
godot-master/
├── core/           # エンジンコア機能
│   ├── config/     # 設定システム
│   ├── crypto/     # 暗号化機能
│   ├── debugger/   # デバッガー
│   ├── error/      # エラー処理
│   ├── extension/  # GDExtension システム
│   ├── input/      # 入力処理
│   ├── io/         # ファイルI/O、ネットワーク
│   ├── math/       # 数学ライブラリ
│   ├── object/     # オブジェクトシステム
│   ├── os/         # OS抽象化層
│   ├── string/     # 文字列処理
│   ├── templates/  # テンプレートクラス
│   └── variant/    # Variant型システム
├── drivers/        # グラフィックス・オーディオドライバ
│   ├── gles3/      # OpenGL ES 3.0
│   ├── vulkan/     # Vulkan
│   ├── metal/      # Metal (Apple)
│   └── d3d12/      # Direct3D 12
├── editor/         # エディタ機能（TOOLS_ENABLED時のみ）
│   ├── plugins/    # エディタプラグイン
│   ├── gui/        # エディタGUI
│   └── import/     # インポーター
├── main/           # メインエントリーポイント
├── modules/        # オプショナルモジュール
│   ├── gdscript/   # GDScriptモジュール
│   ├── mono/       # C#サポート
│   └── ...         # その他モジュール
├── platform/       # プラットフォーム固有コード
│   ├── android/
│   ├── ios/
│   ├── linuxbsd/
│   ├── macos/
│   ├── web/
│   └── windows/
├── scene/          # シーンシステム
│   ├── 2d/         # 2Dノード
│   ├── 3d/         # 3Dノード
│   ├── animation/  # アニメーション
│   ├── gui/        # GUIコントロール
│   ├── main/       # Node, SceneTree等
│   └── resources/  # シーンリソース
├── servers/        # サーバーシングルトン
│   ├── audio/      # オーディオサーバー
│   ├── display/    # ディスプレイサーバー
│   ├── physics_2d/ # 2D物理サーバー
│   ├── physics_3d/ # 3D物理サーバー
│   └── rendering/  # レンダリングサーバー
├── tests/          # ユニットテスト
├── thirdparty/     # サードパーティライブラリ
├── SConstruct      # SCons ビルド設定
└── methods.py      # ビルドヘルパー
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `core/` | エンジンの基盤機能（オブジェクト、バリアント、I/O） | `object.h`, `variant.h`, `resource.h` |
| `scene/` | シーンツリーとノードシステム | `node.h`, `scene_tree.h`, `node_2d.h` |
| `servers/` | レンダリング・物理等のシングルトンサーバー | `rendering_server.h`, `physics_server_2d.h` |
| `editor/` | Godotエディタ（`TOOLS_ENABLED`時のみ） | `editor_node.h`, `editor_plugin.h` |
| `modules/` | オプショナル機能（GDScript, C#等） | `gdscript.h`, `mono_gc_handle.h` |
| `platform/` | OS固有の実装 | `os_windows.cpp`, `os_macos.mm` |
| `drivers/` | グラフィックス・オーディオの低レベル実装 | `rasterizer_gles3.cpp` |
| `main/` | エンジン起動・メインループ | `main.cpp`, `main.h` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

Godot Engine は**サーバー・クライアントアーキテクチャ**を採用しています。

```mermaid
graph TB
    subgraph "User Space"
        GDScript[GDScript]
        CSharp[C#]
        GDExtension[GDExtension]
    end

    subgraph "Scene Layer"
        SceneTree[SceneTree]
        Node[Node]
        Node2D[Node2D]
        Node3D[Node3D]
    end

    subgraph "Server Layer"
        RS[RenderingServer]
        PS2D[PhysicsServer2D]
        PS3D[PhysicsServer3D]
        AS[AudioServer]
        DS[DisplayServer]
    end

    subgraph "Driver Layer"
        Vulkan[Vulkan]
        GLES3[OpenGL ES 3]
        Metal[Metal]
    end

    subgraph "Platform Layer"
        Windows[Windows]
        macOS[macOS]
        Linux[Linux]
        Android[Android]
        iOS[iOS]
        Web[Web]
    end

    GDScript --> SceneTree
    CSharp --> SceneTree
    GDExtension --> SceneTree

    SceneTree --> Node
    Node --> Node2D
    Node --> Node3D

    Node --> RS
    Node --> PS2D
    Node --> PS3D
    Node --> AS

    RS --> Vulkan
    RS --> GLES3
    RS --> Metal

    DS --> Windows
    DS --> macOS
    DS --> Linux
    DS --> Android
    DS --> iOS
    DS --> Web
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| **Platform** | OS固有機能の抽象化 | `platform/*/os_*.cpp` |
| **Driver** | グラフィックスAPIの抽象化 | `drivers/vulkan/`, `drivers/gles3/` |
| **Server** | 機能別シングルトンサーバー | `servers/rendering/rendering_server.h` |
| **Core** | 基盤システム（Object, Variant等） | `core/object/object.h`, `core/variant/variant.h` |
| **Scene** | ノードツリーとシーン管理 | `scene/main/node.h`, `scene/main/scene_tree.h` |
| **Module** | オプショナル機能 | `modules/gdscript/gdscript.h` |
| **Editor** | 開発ツール | `editor/editor_node.h` |

### 5.3 データフロー

**ゲーム実行時のデータフロー:**

1. **初期化フェーズ**: `main.cpp` がエンジンを初期化
2. **シーンロード**: `SceneTree` がシーンを構築
3. **メインループ**:
   - 入力処理 (`Input`)
   - 物理更新 (`PhysicsServer`)
   - スクリプト実行 (`_process`, `_physics_process`)
   - レンダリング (`RenderingServer`)
4. **終了**: リソース解放

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

アプリケーションのエントリーポイントは `main/main.cpp` です。

```cpp
// ファイル: main/main.cpp:156-200

/* Static members */

// Singletons

// Initialized in setup()
static Engine *engine = nullptr;
static ProjectSettings *globals = nullptr;
static Input *input = nullptr;
static InputMap *input_map = nullptr;
static TranslationServer *translation_server = nullptr;
static Performance *performance = nullptr;
static PackedData *packed_data = nullptr;
static MessageQueue *message_queue = nullptr;

// Initialized in setup2()
static AudioServer *audio_server = nullptr;
static CameraServer *camera_server = nullptr;
static DisplayServer *display_server = nullptr;
static RenderingServer *rendering_server = nullptr;
static TextServerManager *tsman = nullptr;
static ThemeDB *theme_db = nullptr;
```

主な初期化関数:
- `setup()`: コアシステムの初期化
- `setup2()`: サーバーの初期化
- `start()`: メインループ開始

### 6.2 オブジェクトシステム

`Object` クラスは全てのGodotクラスの基底です。

```cpp
// ファイル: core/object/object.h のプロパティシステム

enum PropertyHint {
	PROPERTY_HINT_NONE,
	PROPERTY_HINT_RANGE,
	PROPERTY_HINT_ENUM,
	PROPERTY_HINT_FLAGS,
	PROPERTY_HINT_FILE,
	PROPERTY_HINT_DIR,
	// ...
};

struct PropertyInfo {
	Variant::Type type = Variant::NIL;
	String name;
	StringName class_name;
	PropertyHint hint = PROPERTY_HINT_NONE;
	String hint_string;
	uint32_t usage = PROPERTY_USAGE_DEFAULT;
};
```

### 6.3 ノードシステム

`Node` は SceneTree の基本単位です。

```cpp
// ファイル: scene/main/node.h:70-131

class Node : public Object {
	GDCLASS(Node, Object);

public:
	enum ProcessMode : unsigned int {
		PROCESS_MODE_INHERIT,
		PROCESS_MODE_PAUSABLE,
		PROCESS_MODE_WHEN_PAUSED,
		PROCESS_MODE_ALWAYS,
		PROCESS_MODE_DISABLED,
	};

	enum PhysicsInterpolationMode : unsigned int {
		PHYSICS_INTERPOLATION_MODE_INHERIT,
		PHYSICS_INTERPOLATION_MODE_ON,
		PHYSICS_INTERPOLATION_MODE_OFF,
	};
```

### 6.4 シーンツリー

`SceneTree` はノードのライフサイクルとメインループを管理します。

```cpp
// ファイル: scene/main/scene_tree.h:85-150

class SceneTree : public MainLoop {
	GDCLASS(SceneTree, MainLoop);

private:
	Window *root = nullptr;

	double physics_process_time = 0.0;
	double process_time = 0.0;
	bool accept_quit = true;
	bool paused = false;

	HashMap<StringName, Group> group_map;
```

### 6.5 サーバーシステム

サーバーはスレッドセーフなシングルトンとして実装されています。

```cpp
// ファイル: servers/rendering/rendering_server_default.h:44-92

class RenderingServerDefault : public RenderingServer {
	GDSOFTCLASS(RenderingServerDefault, RenderingServer);

	mutable CommandQueueMT command_queue;

	Thread::ID server_thread = Thread::MAIN_ID;
	bool create_thread = false;

	void _draw(bool p_swap_buffers, double frame_step);
	void _init();
	void _finish();
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| GDCLASS マクロ | クラス登録 | 高 | 全ての.hファイル |
| _bind_methods | スクリプトバインディング | 高 | 全てのクラス |
| Ref<T> | 参照カウントスマートポインタ | 高 | `core/object/ref_counted.h` |
| シングルトンサーバー | グローバルサービス | 中 | `servers/` |
| 条件コンパイル | 機能トグル | 中 | 全体 |

### 各パターンの詳細

#### パターン1: GDCLASS マクロ

**目的:** クラスをGodotの型システムに登録し、ランタイム型情報を提供する

**実装例:**
```cpp
// ファイル: scene/main/node.h:50-51
class Node : public Object {
	GDCLASS(Node, Object);

protected:
	static void _bind_methods();
```

**解説:**
- `GDCLASS(ClassName, ParentClass)` は自動的に以下を生成:
  - `get_class_static()` - 静的クラス名取得
  - `is_class()` - 型チェック
  - `cast_to<T>()` - 安全なキャスト

#### パターン2: プロパティバインディング

**目的:** C++のメンバをスクリプトから参照可能にする

**実装例:**
```cpp
// ファイル: core/object/object.h:136-137
#define ADD_PROPERTY(m_property, m_setter, m_getter) \
    ::ClassDB::add_property(get_class_static(), m_property, StringName(m_setter), StringName(m_getter))

// 使用例
ADD_PROPERTY(PropertyInfo(Variant::VECTOR2, "position"), "set_position", "get_position");
```

#### パターン3: 参照カウント (Ref<T>)

**目的:** メモリの自動管理

**実装例:**
```cpp
// ファイル: modules/gdscript/gdscript.h:94-96
Ref<GDScriptNativeClass> native;
Ref<GDScript> base;
```

**解説:**
- `Ref<T>` はスコープを抜けると自動的に参照カウントをデクリメント
- 参照カウントが0になるとオブジェクトは解放される

#### パターン4: シングルトンサーバー

**目的:** グローバルサービスの提供

**実装例:**
```cpp
// ファイル: servers/rendering/rendering_server.h
class RenderingServer : public Object {
	GDCLASS(RenderingServer, Object);
	static RenderingServer *singleton;

public:
	static RenderingServer *get_singleton() { return singleton; }
```

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: エンジン起動フロー

**概要:** Godotエンジンの起動から最初のフレームまでの流れ

**処理フロー:**
```
main() → Main::setup() → Main::setup2() → Main::start() → SceneTree::initialize()
```

**詳細な追跡:**

1. **main関数** (`platform/*/godot_*.cpp`)
   - プラットフォーム固有のエントリーポイント
   - `Main::setup()` を呼び出し

2. **Main::setup()** (`main/main.cpp`)
   ```cpp
   // コアシステムの初期化
   register_core_types();
   register_core_driver_types();
   ```

3. **Main::setup2()** (`main/main.cpp`)
   ```cpp
   // サーバーの初期化
   rendering_server = memnew(RenderingServerDefault);
   audio_server = memnew(AudioServer);
   ```

4. **Main::start()** (`main/main.cpp`)
   - SceneTree の作成とルートノードの設定

#### 例2: Node追加フロー

**概要:** シーンツリーにノードが追加される際の処理

**処理フロー:**
```
Node::add_child() → SceneTree::add_to_group() → Node::_enter_tree()
```

**詳細な追跡:**

1. **add_child呼び出し** (`scene/main/node.cpp`)
   - 親子関係の設定
   - 内部インデックスの更新

2. **ツリー通知** (`scene/main/node.cpp`)
   - `NOTIFICATION_ENTER_TREE` の送信
   - グループへの登録

3. **仮想関数呼び出し**
   - `_enter_tree()` - スクリプトへの通知
   - `_ready()` - 初期化完了通知

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `CONTRIBUTING.md` - 開発ガイドライン
3. 公式ドキュメント: https://docs.godotengine.org/

#### 特定機能を理解したい場合
1. `modules/*/` - 該当モジュールのディレクトリ
2. `scene/*/` - 関連するシーンコンポーネント
3. `servers/*/` - 対応するサーバー実装

#### 改修作業を行う場合
1. `CONTRIBUTING.md` - コーディング規約
2. `.clang-format` - フォーマット設定
3. `tests/` - 既存テストケース

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | プロジェクト概要 | 初回参照 |
| `CONTRIBUTING.md` | 貢献ガイドライン | 開発開始時 |
| `CHANGELOG.md` | 変更履歴 | バージョン確認時 |
| `COPYRIGHT.txt` | 著作権・ライセンス | 法的確認時 |
| `doc/` | クラスリファレンスXML | API調査時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `GDCLASS` と `GDSOFTCLASS` の違いは？
A: `GDCLASS` は完全な型情報を持つクラス用、`GDSOFTCLASS` はサーバークラスなど軽量な型情報で十分なクラス用です。

#### Q: `#ifdef TOOLS_ENABLED` で囲まれたコードは何？
A: エディタビルド時のみ含まれるコードです。ゲームエクスポート時には除外されます。

#### Q: `memnew()` と `new` の違いは？
A: `memnew()` はGodotのメモリ追跡システムを使用します。全てのGodotオブジェクトは `memnew()` で作成してください。

#### Q: `StringName` と `String` の違いは？
A: `StringName` はインターン化された文字列で、比較が高速です。メソッド名やプロパティ名に使用されます。

### デバッグのヒント

1. **ブレークポイント設置場所:**
   - `_notification()` - ライフサイクルイベント
   - `_bind_methods()` - プロパティ登録確認
   - `Main::setup()` / `Main::setup2()` - 初期化追跡

2. **ログ出力:**
   ```cpp
   print_line("Debug message");  // 標準出力
   WARN_PRINT("Warning");        // 警告
   ERR_PRINT("Error");           // エラー
   ```

3. **条件コンパイルの確認:**
   - `DEV_ENABLED` - 開発ビルド
   - `DEBUG_ENABLED` - デバッグビルド
   - `TOOLS_ENABLED` - エディタビルド

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Node | シーンツリーの基本単位 |
| Resource | シリアライズ可能なデータオブジェクト |
| Server | スレッドセーフなシングルトンサービス |
| Variant | 動的型コンテナ |
| RefCounted | 参照カウント基底クラス |
| GDExtension | ネイティブ拡張システム |
| SceneTree | ノードのライフサイクル管理 |
| ClassDB | クラス登録データベース |
| PropertyInfo | プロパティメタデータ |
| RID | リソース識別子（Resource ID） |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `SConstruct` | ビルド設定 | SConsビルドシステム設定 |
| `methods.py` | ビルドヘルパー | ビルド用Python関数 |
| `version.py` | バージョン定義 | バージョン番号 |
| `core/object/object.h` | Object基底クラス | 全クラスの基底 |
| `core/variant/variant.h` | Variant型 | 動的型システム |
| `scene/main/node.h` | Node基底クラス | ノードシステム |
| `main/main.cpp` | メインエントリー | エンジン起動 |

### C. 参考資料

- [Godot公式ドキュメント](https://docs.godotengine.org/)
- [Godot Contributing Documentation](https://contributing.godotengine.org/)
- [Godot GitHub リポジトリ](https://github.com/godotengine/godot)
- [C++17 リファレンス](https://en.cppreference.com/w/cpp/17)
- [SCons ビルドシステム](https://scons.org/documentation.html)
