# セキュリティ設計書

## 概要

本ドキュメントはGodot Engineのセキュリティ設計について記述する。Godot Engineはオープンソースのゲームエンジンであり、ネットワーク通信、暗号化、ファイルアクセス制御などのセキュリティ機能を提供している。本設計書では、これらのセキュリティ機能の実装方式と設計意図を明確にする。

## 認証設計

### 認証方式

Godot Engineは汎用ゲームエンジンであり、特定の認証システムを内蔵していない。ただし、以下の暗号化基盤を提供し、開発者がカスタム認証システムを構築できるようにしている。

- **RSA鍵生成**: `Crypto::generate_rsa()`によるRSA鍵ペアの生成
- **署名/検証**: `Crypto::sign()`/`Crypto::verify()`によるデジタル署名
- **X.509証明書**: `X509Certificate`クラスによる証明書の読み込み・生成
- **HMAC**: `HMACContext`クラスによるメッセージ認証コードの生成

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | アプリケーション依存 | エンジン自体はセッション管理機能を提供しない |
| セッション固定化対策 | 開発者実装 | TLS接続によるセッション保護を推奨 |

## 認可設計

### 権限体系

Godot Engineはファイルアクセスに対して以下の権限体系を持つ。

| ロール | 権限 | 説明 |
| --- | --- | --- |
| ACCESS_RESOURCES | 読み取り | res://パス配下のリソースへのアクセス |
| ACCESS_USERDATA | 読み書き | user://パス配下のユーザーデータへのアクセス |
| ACCESS_FILESYSTEM | 読み書き | システムファイルシステムへの直接アクセス |
| ACCESS_PIPE | 読み書き | パイプ経由のファイルアクセス |

### アクセス制御

ファイルアクセスは`FileAccess`クラスと`DirAccess`クラスによって制御される。主な制御機構は以下の通り。

- **パスプレフィックス制御**: `res://`（リソース）、`user://`（ユーザーデータ）による論理的なアクセス制御
- **Unixパーミッション**: `UnixPermissionFlags`による標準的なUNIX権限管理（読み取り/書き込み/実行）
- **読み取り専用属性**: ファイル・ディレクトリの読み取り専用フラグ
- **隠しファイル属性**: ファイルの表示/非表示制御

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS/TLS | mbedTLSライブラリによるTLS 1.2/1.3サポート |
| DTLS | UDPベースの暗号化通信（ENet連携） |
| 証明書検証 | X.509証明書チェーンの検証、ホスト名検証 |
| CA証明書 | システムCA証明書またはビルトイン証明書の使用 |

### TLS実装詳細

- **StreamPeerTLS**: TCPストリームのTLS暗号化
- **PacketPeerDTLS**: UDPパケットのDTLS暗号化
- **DTLSServer**: DTLSサーバー機能
- **TLSOptions**: TLS接続オプション（クライアント/サーバーモード、証明書設定）

### TLSモード

```cpp
enum Mode {
    MODE_CLIENT = 0,       // 標準クライアントモード（証明書検証あり）
    MODE_CLIENT_UNSAFE = 1, // 非推奨: 証明書検証なし
    MODE_SERVER = 2,       // サーバーモード
};
```

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | TLS 1.2/1.3 (mbedTLS実装) |
| ファイル保存 | AES-256-CFB |
| パスワードハッシュ | MD5 (互換性目的)、SHA-1、SHA-256 |
| スクリプト暗号化 | AES-256 |
| PCKファイル暗号化 | AES-256-CFB |

### 暗号化アルゴリズム実装

#### AESContext
```cpp
enum Mode : int32_t {
    MODE_ECB_ENCRYPT,  // ECBモード暗号化
    MODE_ECB_DECRYPT,  // ECBモード復号
    MODE_CBC_ENCRYPT,  // CBCモード暗号化
    MODE_CBC_DECRYPT,  // CBCモード復号
};
```

#### ハッシュアルゴリズム
```cpp
enum HashType : int32_t {
    HASH_MD5,    // 16バイト出力
    HASH_SHA1,   // 20バイト出力
    HASH_SHA256  // 32バイト出力
};
```

### 機密情報管理

- **鍵のゼロ化**: 暗号鍵使用後は`mbedtls_platform_zeroize()`でメモリをゼロ化
- **秘密鍵の保護**: `CryptoKey`クラスによる秘密鍵の安全な管理
- **スクリプト暗号化鍵**: ビルド時に`SCRIPT_AES256_ENCRYPTION_KEY`環境変数で設定
- **タイミング攻撃対策**: `Crypto::constant_time_compare()`による定数時間比較

### ファイル暗号化

`FileAccessEncrypted`クラスによる暗号化ファイルアクセス。

- **暗号化方式**: AES-256-CFB
- **IV生成**: `CryptoCore::RandomGenerator`による安全な乱数生成
- **整合性検証**: MD5ハッシュによるデータ整合性チェック
- **ヘッダーマジック**: `0x43454447` ("GDEC")による識別

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | Godot Engineはブラウザベースではないため対象外。ただしHTML5エクスポートではブラウザのセキュリティ機能に依存 |
| SQLインジェクション | 内蔵DBなし。外部DB接続時は開発者責任 |
| パストラバーサル | `fix_path()`による正規化、`simplify_path()`による危険なパス要素の除去 |
| バッファオーバーフロー | C++実装での境界チェック、`ERR_FAIL_COND_V`マクロによる入力検証 |

### 入力検証

- **パス検証**: ファイルパスの正規化と検証
- **バッファサイズ検証**: 暗号化操作時のデータサイズチェック
- **鍵長検証**: AES-256は32バイト、IVは16バイトの厳密なチェック

## 乱数生成

### セキュア乱数生成器

```cpp
class RandomGenerator {
    void *entropy = nullptr;  // エントロピーソース
    void *ctx = nullptr;      // CTR_DRBGコンテキスト

    Error init();
    Error get_random_bytes(uint8_t *r_buffer, size_t p_bytes);
};
```

- **実装**: mbedTLSの`mbedtls_ctr_drbg`（CTR_DRBG NIST SP 800-90A準拠）
- **エントロピー源**: プラットフォーム固有のエントロピー収集
- **用途**: IV生成、鍵生成、証明書シリアル番号生成

## PCKファイル暗号化

### 暗号化フラグ

```cpp
enum PackFlags {
    PACK_DIR_ENCRYPTED = 1 << 0,  // ディレクトリ構造の暗号化
    PACK_REL_FILEBASE = 1 << 1,   // 相対ファイルベース
    PACK_SPARSE_BUNDLE = 1 << 2,  // スパースバンドル
};

enum PackFileFlags {
    PACK_FILE_ENCRYPTED = 1 << 0,  // ファイル内容の暗号化
    PACK_FILE_REMOVAL = 1 << 1,    // ファイル削除マーク
    PACK_FILE_DELTA = 1 << 2,      // デルタパッチ
};
```

### スクリプト暗号化

- **鍵設定**: ビルド時に`SCRIPT_AES256_ENCRYPTION_KEY`環境変数で256ビット鍵を指定
- **保存先**: `script_encryption_key.gen.cpp`にコンパイル時に埋め込み
- **用途**: GDScriptソースコードの保護

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| エラーログ | 暗号化/復号エラー、証明書検証エラー | ランタイム中 |
| デバッグログ | TLSハンドシェイク詳細（デバッグビルド時） | ランタイム中 |
| 警告ログ | 非推奨API使用、証明書有効期限警告 | ランタイム中 |

### ログ出力機構

- **Logger**クラス: 標準出力、ファイル出力、システムログへの出力
- **ERR_PRINT/ERR_FAIL**マクロ: エラー条件の記録
- **print_verbose**: 詳細ログ出力（verbose有効時）

## 備考

### セキュリティに関する注意事項

1. **MODE_CLIENT_UNSAFE**: TLSOptionsのunsafeクライアントモードは証明書検証を行わないため、本番環境での使用は非推奨
2. **MD5の使用**: ファイル整合性チェックにMD5を使用しているが、暗号学的用途では非推奨。将来的にSHA-256への移行が望ましい
3. **鍵の埋め込み**: スクリプト暗号化鍵はバイナリに埋め込まれるため、リバースエンジニアリングによる抽出リスクがある
4. **サンドボックス**: Godot Engine自体はサンドボックス機能を持たず、OSレベルのサンドボックス（iOS App Sandbox、Android Permissions等）に依存

### 推奨事項

- 機密データの暗号化にはAES-256を使用
- ネットワーク通信にはTLSを使用し、証明書検証を有効化
- パスワード保存にはSHA-256以上のハッシュアルゴリズムを使用
- スクリプト暗号化は難読化目的であり、完全な保護ではないことを認識

### 依存ライブラリ

- **mbedTLS**: TLS/SSL、暗号化プリミティブの実装
- **zlib/zstd**: 圧縮機能（暗号化前の圧縮に使用可能）
