# 機能設計書 11-シェーダー

## 概要

本ドキュメントは、Godotエンジンにおけるシェーダー機能の設計仕様を記述する。シェーダー機能は、マテリアルの視覚的表現をカスタマイズするための核心的な機能であり、テキストベースのGodotシェーダー言語（.gdshader）とノードベースのVisual Shaderの両方をサポートする。

### 本機能の処理概要

**業務上の目的・背景**：シェーダー機能は、ゲーム開発において独自の視覚効果やマテリアル表現を実現するために必要不可欠な機能である。標準的なマテリアル設定では表現できない特殊な描画効果（水の波紋、炎のエフェクト、トゥーンシェーディング等）を実現するために、GPUに対してカスタムの描画命令を記述できる仕組みを提供する。

**機能の利用シーン**：
- 独自のマテリアル表現が必要な場合（メタリック、サブサーフェススキャッタリング等）
- ポストプロセスエフェクトの実装（ブルーム、ビネット、カラーグレーディング等）
- パーティクルの動作制御
- スカイボックスやフォグのカスタマイズ
- プロシージャルテクスチャの生成

**主要な処理内容**：
1. シェーダーコードのパース・コンパイル（ShaderLanguageクラス）
2. シェーダーインクルード依存関係の解決（ShaderPreprocessor）
3. RenderingServerへのシェーダーRID登録
4. Visual Shaderノードからのコード生成
5. シェーダーパラメータ（uniform）の管理

**関連システム・外部連携**：
- RenderingServer：シェーダーのGPU登録と管理
- MaterialクラスとのShader参照関係
- ShaderIncludeによるコード再利用

**権限による制御**：特になし（エディタ・ランタイム共通で利用可能）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 18 | シェーダーエディタ | 主画面 | GLSL/Godotシェーダーコードの編集 |
| 42 | ビジュアルシェーダーエディタ | 主画面 | ノードベースのビジュアルシェーダー編集 |
| 54 | マテリアルエディタ | 参照画面 | マテリアル・シェーダーのプレビュー表示 |

## 機能種別

リソース管理 / GPU描画制御

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| code | String | Yes | シェーダーソースコード | 有効なシェーダー構文であること |
| mode | Mode | No | シェーダーモード（spatial/canvas_item/particles/sky/fog） | 自動判定可能 |
| default_textures | HashMap | No | デフォルトテクスチャパラメータ | テクスチャRIDが有効であること |

### 入力データソース

- .gdshaderファイル（テキストベースシェーダー）
- .tres/.resファイル（VisualShaderリソース）
- スクリプトからの動的生成

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| shader_rid | RID | RenderingServerに登録されたシェーダーID |
| mode | Mode | 解析されたシェーダーモード |
| preprocessed_code | String | プリプロセス済みシェーダーコード |
| uniform_list | List<PropertyInfo> | シェーダーパラメータ一覧 |

### 出力先

- RenderingServer（GPUシェーダープログラム）
- Materialインスタンス

## 処理フロー

### 処理シーケンス

```
1. シェーダーコードの設定（set_code）
   └─ include依存関係の切断
2. ShaderPreprocessorによるプリプロセス
   └─ #include文の解決、新しい依存関係の構築
3. シェーダータイプの判定
   └─ shader_type宣言からmode決定（spatial/canvas_item/particles/sky/fog）
4. RenderingServerへの登録
   └─ shader_set_codeによるGPUコンパイル
5. 変更通知の発行
   └─ emit_changed()で参照側に通知
```

### フローチャート

```mermaid
flowchart TD
    A[set_code呼び出し] --> B[既存include依存の切断]
    B --> C[ShaderPreprocessor実行]
    C --> D{プリプロセス成功?}
    D -->|Yes| E[include_dependenciesを更新]
    D -->|No| F[エラー状態維持]
    E --> G[shader_typeを解析]
    G --> H[modeを決定]
    H --> I{shader_ridが有効?}
    I -->|Yes| J[RenderingServerにコード送信]
    I -->|No| K[遅延初期化のため保持]
    J --> L[emit_changed発行]
    K --> L
    L --> M[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | シェーダータイプ自動判定 | コード内のshader_type宣言からモードを自動判定 | set_code実行時 |
| BR-02 | 依存関係の自動管理 | ShaderIncludeリソースの変更を自動検知して再コンパイル | _dependency_changed時 |
| BR-03 | RID遅延生成 | shader_ridは実際に必要になるまで生成しない | get_rid初回呼び出し時 |

### 計算ロジック

特になし

## データベース操作仕様

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 構文エラー | シェーダーコードが不正 | エディタにエラー表示 |
| - | インクルードエラー | 参照先ファイルが見つからない | パス確認、ファイル存在確認 |
| ERR_FILE_CANT_OPEN | ファイルエラー | シェーダーファイル読み込み失敗 | ファイルパス確認 |

### リトライ仕様

なし

## トランザクション仕様

なし（リソース管理のみ）

## パフォーマンス要件

- シェーダーコンパイルはバックグラウンドで実行
- RIDの遅延生成により起動時間を短縮

## セキュリティ考慮事項

- シェーダーコードはサンドボックス化されたGPU上で実行
- ファイルシステムへのアクセスは不可

## 備考

- Visual Shaderはvisual_shader.cppで実装され、内部的にShaderクラスを使用
- シェーダーモード: MODE_SPATIAL(0), MODE_CANVAS_ITEM(1), MODE_PARTICLES(2), MODE_SKY(3), MODE_FOG(4)

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

シェーダーのデータ構造を理解するため、ヘッダファイルから確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | shader.h | `scene/resources/shader.h` | Shaderクラスのメンバ変数（mode, shader_rid, code, default_textures） |
| 1-2 | visual_shader.h | `scene/resources/visual_shader.h` | VisualShaderクラスとVisualShaderNodeの継承関係 |

**読解のコツ**: Shaderクラスはmode（シェーダータイプ）とshader_rid（RenderingServerへの参照）を管理する。shader_ridは必要になるまで生成されない遅延初期化パターンを採用している。

#### Step 2: エントリーポイントを理解する

シェーダーコード設定の流れを追う。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | shader.cpp | `scene/resources/shader.cpp` | set_code関数がエントリーポイント |

**主要処理フロー**:
1. **85-90行目**: include依存関係の切断
2. **93-108行目**: ShaderPreprocessorによるプリプロセス実行
3. **111-123行目**: shader_typeからモード判定
4. **125-131行目**: RenderingServerへのコード送信
5. **134行目**: emit_changed()で変更通知

#### Step 3: プリプロセッサを理解する

シェーダーインクルードの解決処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | shader_preprocessor.cpp | `servers/rendering/shader_preprocessor.cpp` | #include文の解決処理 |
| 3-2 | shader_include.cpp | `scene/resources/shader_include.cpp` | インクルードファイルのリソース管理 |

**主要処理フロー**:
- プリプロセッサはinclude文を再帰的に解決
- 依存関係はHashSet<Ref<ShaderInclude>>で管理

#### Step 4: Visual Shaderを理解する

ノードベースシェーダーのコード生成を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | visual_shader.cpp | `scene/resources/visual_shader.cpp` | ノードグラフからシェーダーコード生成 |
| 4-2 | visual_shader_nodes.cpp | `scene/resources/visual_shader_nodes.cpp` | 各種ノードの実装 |

### プログラム呼び出し階層図

```
Shader::set_code()
    │
    ├─ ShaderPreprocessor::preprocess()
    │      └─ ShaderInclude::get_code()
    │
    ├─ ShaderLanguage::get_shader_type()
    │
    └─ RenderingServer::shader_set_code()

VisualShader::_update_shader() [継承元: Shader]
    │
    └─ VisualShader::generate_code()
           └─ VisualShaderNode::generate_code()
```

### データフロー図

```
[入力]                    [処理]                         [出力]

.gdshaderファイル ──────▶ ResourceFormatLoaderShader
                              │
                              ▼
                         Shader::set_code()
                              │
                              ▼
                         ShaderPreprocessor
                              │
                              ▼
                         RenderingServer ──────▶ GPUシェーダープログラム
                              │
                              ▼
                         Material参照
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| shader.cpp | `scene/resources/shader.cpp` | ソース | Shaderリソースクラスの実装 |
| shader.h | `scene/resources/shader.h` | ヘッダ | Shaderクラス定義 |
| visual_shader.cpp | `scene/resources/visual_shader.cpp` | ソース | Visual Shaderの実装 |
| visual_shader.h | `scene/resources/visual_shader.h` | ヘッダ | Visual Shaderクラス定義 |
| visual_shader_nodes.cpp | `scene/resources/visual_shader_nodes.cpp` | ソース | 各種ビジュアルシェーダーノード |
| shader_include.cpp | `scene/resources/shader_include.cpp` | ソース | シェーダーインクルードリソース |
| shader_preprocessor.cpp | `servers/rendering/shader_preprocessor.cpp` | ソース | シェーダープリプロセッサ |
| shader_language.cpp | `servers/rendering/shader_language.cpp` | ソース | シェーダー言語パーサー |
