# 機能設計書 19-オーディオ再生

## 概要

本ドキュメントは、Godotエンジンにおけるオーディオ再生機能の設計仕様を記述する。オーディオ再生機能は、AudioStreamPlayerノードを使用して、ゲーム内でBGM、効果音、ボイスなどの音声を再生する機能である。非空間オーディオ（2D/3D空間に依存しない）を提供する。

### 本機能の処理概要

**業務上の目的・背景**：ゲーム開発において、オーディオはゲームプレイ体験の重要な要素である。BGM、効果音、UI音、ボイスなど、様々な種類の音声を適切なタイミングで再生することで、ゲームの没入感を高める。AudioStreamPlayerは、最もシンプルな非空間オーディオ再生を提供する。

**機能の利用シーン**：
- BGMの再生
- UI効果音の再生
- カットシーンのボイス再生
- ステレオまたはサラウンド出力が必要な場合
- 空間に依存しない環境音

**主要な処理内容**：
1. AudioStreamリソースの設定
2. AudioServerへの再生開始/停止
3. 音量（dB）とピッチの制御
4. バス（オーディオミキサー）への出力
5. ポリフォニー（同時発音数）の管理
6. 自動再生機能

**関連システム・外部連携**：
- AudioServer：オーディオミキシングとバス管理
- AudioStream：オーディオリソース（OGG、WAV、MP3等）
- AudioStreamPlayback：再生状態の管理

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | オーディオバスレイアウト | 設定画面 | オーディオバスの設定 |

## 機能種別

オーディオ再生 / サウンド管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| stream | Ref<AudioStream> | No | 再生するオーディオストリーム | 有効なAudioStreamリソース |
| volume_db | float | No | 音量（dB）、範囲: -80 to 24 | NaNでないこと |
| pitch_scale | float | No | ピッチ倍率、範囲: 0.01 to 4+ | 0より大きいこと |
| bus | StringName | No | 出力先バス名 | 有効なバス名 |
| autoplay | bool | No | シーン開始時に自動再生 | - |
| max_polyphony | int | No | 最大同時発音数 | 1以上 |
| mix_target | MixTarget | No | ミックスターゲット（Stereo/Surround/Center） | - |

### 入力データソース

- AudioStreamリソースファイル（.ogg, .wav, .mp3等）
- エディタでのプロパティ設定
- スクリプトからの動的設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| is_playing | bool | 再生中かどうか |
| playback_position | float | 現在の再生位置（秒） |
| stream_playback | Ref<AudioStreamPlayback> | 再生状態オブジェクト |

### 出力先

- AudioServer（オーディオミキサー）
- 指定されたオーディオバス
- スピーカー/ヘッドフォン

## 処理フロー

### 処理シーケンス

```
1. ストリームの設定
   └─ set_stream()でAudioStreamを設定
2. 再生開始
   └─ play()関数呼び出し
3. AudioStreamPlaybackの作成
   └─ internal->play_basic()
4. AudioServerへの登録
   └─ start_playback_stream()
5. 音量ベクトルの計算
   └─ _get_volume_vector()
6. 再生位置の追跡
   └─ get_playback_position()
```

### フローチャート

```mermaid
flowchart TD
    A[play呼び出し] --> B[play_basic]
    B --> C{streamが有効?}
    C -->|No| D[リターン]
    C -->|Yes| E[AudioStreamPlayback作成]
    E --> F[start_playback_stream]
    F --> G[_get_volume_vector]
    G --> H[音量ベクトル計算]
    H --> I{サンプル再生?}
    I -->|Yes| J[start_sample_playback]
    I -->|No| K[ストリーム再生継続]
    J --> L[ensure_playback_limit]
    K --> L
    L --> M[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ポリフォニー制限 | max_polyphonyを超える再生は古いものから停止 | 同時発音数超過時 |
| BR-02 | NaN拒否 | volume_dbにNaNは設定不可 | set_volume_db時 |
| BR-03 | 自動再生 | autoplayがtrueの場合、シーン開始時に自動再生 | NOTIFICATION_READY時 |

### 計算ロジック

音量ベクトル計算（ステレオ）:
```cpp
float volume_linear = Math::db_to_linear(internal->volume_db);
if (AudioServer::get_singleton()->get_speaker_mode() == AudioServer::SPEAKER_MODE_STEREO) {
    volume_vector.write[0] = AudioFrame(volume_linear, volume_linear);
}
```

サラウンド時のミックスターゲット対応:
- MIX_TARGET_STEREO: フロントL/Rのみ
- MIX_TARGET_SURROUND: 全チャンネル
- MIX_TARGET_CENTER: センターチャンネルのみ

## データベース操作仕様

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | エラー | volume_dbにNaNを設定 | 有効な数値を設定 |
| - | 警告 | streamが未設定で再生 | AudioStreamリソースを設定 |

### リトライ仕様

なし

## トランザクション仕様

なし

## パフォーマンス要件

- オーディオ処理は専用スレッドで実行
- サンプル再生モードでレイテンシ削減可能

## セキュリティ考慮事項

特になし

## 備考

- MixTarget: MIX_TARGET_STEREO(0), MIX_TARGET_SURROUND(1), MIX_TARGET_CENTER(2)
- PlaybackType: Default, Stream, Sample

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

AudioStreamPlayerの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | audio_stream_player.h | `scene/audio/audio_stream_player.h` | AudioStreamPlayerクラスのメンバ変数 |
| 1-2 | audio_stream_player_internal.h | `scene/audio/audio_stream_player_internal.h` | 内部実装クラス |

**読解のコツ**: AudioStreamPlayerはinternal（AudioStreamPlayerInternal）に多くの処理を委譲している。internalはstream_playbacks、volume_db、pitch_scaleなどを管理する。

#### Step 2: エントリーポイントを理解する

play関数の実装を追う。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | audio_stream_player.cpp | `scene/audio/audio_stream_player.cpp` | play関数 |

**主要処理フロー**:
1. **107-124行目**: play関数 - 再生開始処理
2. **108行目**: internal->play_basic()で再生準備
3. **112行目**: AudioServer::start_playback_stream()で再生開始
4. **116-123行目**: サンプル再生モードの処理

#### Step 3: 音量計算を理解する

_get_volume_vector関数を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | audio_stream_player.cpp | `scene/audio/audio_stream_player.cpp` | _get_volume_vector関数 |

**主要処理フロー**:
- **181-216行目**: _get_volume_vector - 音量ベクトル計算
- **191行目**: dBからリニア変換
- **195-214行目**: スピーカーモードとミックスターゲットに応じた配分

### プログラム呼び出し階層図

```
AudioStreamPlayer::play()
    │
    ├─ AudioStreamPlayerInternal::play_basic()
    │      │
    │      └─ AudioStream::instantiate_playback()
    │
    ├─ AudioStreamPlayer::_get_volume_vector()
    │
    └─ AudioServer::start_playback_stream()
           │
           └─ オーディオミキサースレッドで再生

AudioStreamPlayer::stop()
    │
    └─ AudioStreamPlayerInternal::stop_basic()
           │
           └─ AudioServer::stop_playback_stream()
```

### データフロー図

```
[入力]                    [処理]                         [出力]

AudioStreamリソース ────▶ AudioStreamPlayer
       │                      │
       ▼                      ▼
  stream, volume_db      play() / stop()
  pitch_scale, bus            │
                              ▼
                       AudioServer
                              │
                              ▼
                       オーディオバス ──────▶ スピーカー出力
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| audio_stream_player.cpp | `scene/audio/audio_stream_player.cpp` | ソース | AudioStreamPlayer実装 |
| audio_stream_player.h | `scene/audio/audio_stream_player.h` | ヘッダ | AudioStreamPlayer定義 |
| audio_stream_player_internal.cpp | `scene/audio/audio_stream_player_internal.cpp` | ソース | 内部実装 |
| audio_stream.cpp | `servers/audio/audio_stream.cpp` | ソース | AudioStreamリソース |
| audio_server.cpp | `servers/audio/audio_server.cpp` | ソース | オーディオサーバー |
