# 機能設計書 21-オーディオエフェクト

## 概要

本ドキュメントは、Godot Engineのオーディオエフェクト機能について、その設計と実装の詳細を記述した機能設計書である。オーディオエフェクトは、ゲーム内の音声に対してリバーブ、コンプレッサー、イコライザーなどの音響処理を適用するための機能を提供する。

### 本機能の処理概要

**業務上の目的・背景**：ゲーム開発において、サウンドデザインは没入感を高める重要な要素である。オーディオエフェクト機能により、開発者はプログラミング知識がなくても、リアルな環境音響効果（洞窟のリバーブ、屋外の開放感など）や、ダイナミクス制御（音量の均一化）、周波数調整などを実現できる。これにより、プロフェッショナルな品質のサウンドデザインが可能となる。

**機能の利用シーン**：
- 洞窟や大聖堂などの閉鎖空間でリバーブエフェクトを適用して残響効果を演出
- BGMやSEの音量差を均一化するためにコンプレッサーを適用
- 特定の周波数帯域を強調・減衰させるためにイコライザーを使用
- ラジオやテレビからの音声を表現するためにフィルターエフェクトを適用

**主要な処理内容**：
1. AudioEffectリソースの作成とパラメータ設定
2. AudioServerのバスへのエフェクト追加
3. リアルタイムでの音声フレーム処理（process関数）
4. エフェクトパラメータのランタイム変更

**関連システム・外部連携**：AudioServer（バスシステム）、AudioStreamPlayer（音声再生ノード）と連携して動作する。

**権限による制御**：特に権限による制御はなく、すべてのユーザーがエフェクトを使用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 16 | オーディオバス | 主画面 | バスへのエフェクト追加・パラメータ調整 |

## 機能種別

計算処理 / リアルタイム信号処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| p_src_frames | AudioFrame* | Yes | 入力オーディオフレーム配列 | NULL不可 |
| p_dst_frames | AudioFrame* | Yes | 出力オーディオフレーム配列 | NULL不可 |
| p_frame_count | int | Yes | 処理するフレーム数 | 正の整数 |

### 入力データソース

AudioServerのミキシングバッファから供給されるオーディオフレームデータ。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| p_dst_frames | AudioFrame* | エフェクト適用後のオーディオフレーム |

### 出力先

AudioServerのミキシングパイプラインに戻され、最終的にオーディオドライバーへ出力される。

## 処理フロー

### 処理シーケンス

```
1. AudioServerがミックスステップを開始
   └─ 各バスに対してエフェクトチェーンを処理
2. AudioEffectInstanceのprocess関数が呼び出される
   └─ 入力フレームを受け取り、エフェクト処理を適用
3. 各エフェクト固有のDSPアルゴリズムを実行
   └─ リバーブ：畳み込み/遅延処理
   └─ コンプレッサー：ダイナミクス処理
   └─ EQ：フィルターバンク処理
4. 処理済みフレームを出力バッファに書き込み
```

### フローチャート

```mermaid
flowchart TD
    A[オーディオフレーム入力] --> B{エフェクト有効?}
    B -->|Yes| C[エフェクトパラメータ取得]
    B -->|No| G[バイパス出力]
    C --> D[DSPアルゴリズム実行]
    D --> E[左右チャンネル処理]
    E --> F[出力フレームに書き込み]
    F --> H[次のフレームバッチへ]
    G --> H
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-21-01 | サンプルレート同期 | エフェクトはAudioServerのミックスレートに同期して動作する | 常時 |
| BR-21-02 | ステレオ処理 | 左右チャンネルは独立または連携して処理される | エフェクト種別による |
| BR-21-03 | リアルタイム制約 | 処理はオーディオスレッドで実行され、低レイテンシが要求される | 常時 |

### 計算ロジック

**リバーブ処理（AudioEffectReverb）**：
- プリディレイ：入力信号を指定ミリ秒遅延
- ルームサイズ：残響時間を制御（0.0〜1.0）
- ダンピング：高周波の減衰率
- Dry/Wet：原音とエフェクト音のミックス比率

**コンプレッサー処理（AudioEffectCompressor）**：
- threshold：圧縮開始閾値（dB）
- ratio：圧縮比率
- attack_us：アタックタイム（マイクロ秒）
- release_ms：リリースタイム（ミリ秒）
- gain reduction = -overdb * (cratio - 1) / cratio

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作はなし。すべての処理はメモリ上で行われる。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | バッファオーバーフロー | フレーム数がバッファサイズを超過 | MIN関数で制限 |
| - | NULLポインタ | 無効なエフェクトインスタンス | ERR_FAIL_CONDで早期リターン |

### リトライ仕様

リアルタイム処理のため、リトライ機構はなし。エラー時はフレームをスキップ。

## トランザクション仕様

トランザクション管理は不要。オーディオ処理はフレーム単位で完結する。

## パフォーマンス要件

- レイテンシ：オーディオバッファサイズに依存（通常5-20ms）
- スループット：44100Hz / 48000Hz等のサンプルレートでリアルタイム処理
- CPU使用率：エフェクト数とアルゴリズム複雑度に比例

## セキュリティ考慮事項

- 外部入力なし（内部オーディオデータのみ処理）
- メモリ安全性：固定サイズバッファによる境界チェック

## 備考

- エフェクトチェーンは直列処理され、順序が結果に影響する
- 一部エフェクト（コンプレッサー）はサイドチェーン入力をサポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

オーディオエフェクトの基底クラスとインスタンス構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | audio_effect.h | `servers/audio/audio_effect.h` | AudioEffectとAudioEffectInstanceの基底クラス定義 |
| 1-2 | audio_frame.h | `core/math/audio_frame.h` | AudioFrame構造体（左右チャンネルの浮動小数点ペア） |

**読解のコツ**: AudioEffectは`instantiate()`メソッドでAudioEffectInstanceを生成するファクトリパターンを使用している。

#### Step 2: エントリーポイントを理解する

AudioServerからエフェクトが呼び出される流れを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | audio_server.h | `servers/audio/audio_server.h` | Bus構造体のeffects配列、Channel構造体のeffect_instances |
| 2-2 | audio_server.cpp | `servers/audio/audio_server.cpp` | _mix_step関数でのエフェクト呼び出し |

**主要処理フロー**:
1. **238-271行目**: Bus構造体の定義、Effect構造体を含む
2. **261-279行目**: _driver_process関数でミックスステップを開始

#### Step 3: 個別エフェクトの実装を理解する

代表的なエフェクトの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | audio_effect_reverb.cpp | `servers/audio/effects/audio_effect_reverb.cpp` | リバーブエフェクトのprocess実装 |
| 3-2 | audio_effect_compressor.cpp | `servers/audio/effects/audio_effect_compressor.cpp` | コンプレッサーのダイナミクス処理 |
| 3-3 | audio_effect_eq.cpp | `servers/audio/effects/audio_effect_eq.cpp` | イコライザーのバンド処理 |

**主要処理フロー**:
- **35-75行目（reverb）**: 左右チャンネル独立処理、Reverbクラスへの委譲
- **34-113行目（compressor）**: 閾値比較、ゲインリダクション計算
- **35-61行目（eq）**: バンドごとのフィルター処理

### プログラム呼び出し階層図

```
AudioServer::_driver_process
    │
    ├─ AudioServer::_mix_step
    │      │
    │      └─ Bus::Channel::effect_instances[i]->process()
    │             │
    │             ├─ AudioEffectReverbInstance::process
    │             │      └─ Reverb::process
    │             │
    │             ├─ AudioEffectCompressorInstance::process
    │             │
    │             └─ AudioEffectEQInstance::process
    │                    └─ EQ::BandProcess::process_one
    │
    └─ AudioDriver::audio_server_process
```

### データフロー図

```
[入力]                    [処理]                         [出力]

AudioStreamPlayback   ──▶  AudioServer               ──▶  AudioDriver
   (音声データ)              (バスミキシング)                (スピーカー出力)
                                │
                                ▼
                         AudioEffectInstance
                           (エフェクト処理)
                                │
                     ┌──────────┼──────────┐
                     ▼          ▼          ▼
                  Reverb   Compressor     EQ
                   処理       処理        処理
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| audio_effect.h | `servers/audio/audio_effect.h` | ソース | エフェクト基底クラス |
| audio_server.h | `servers/audio/audio_server.h` | ソース | オーディオサーバー定義 |
| audio_server.cpp | `servers/audio/audio_server.cpp` | ソース | バスとエフェクト処理 |
| audio_effect_reverb.h | `servers/audio/effects/audio_effect_reverb.h` | ソース | リバーブエフェクト定義 |
| audio_effect_reverb.cpp | `servers/audio/effects/audio_effect_reverb.cpp` | ソース | リバーブ実装 |
| audio_effect_compressor.h | `servers/audio/effects/audio_effect_compressor.h` | ソース | コンプレッサー定義 |
| audio_effect_compressor.cpp | `servers/audio/effects/audio_effect_compressor.cpp` | ソース | コンプレッサー実装 |
| audio_effect_eq.h | `servers/audio/effects/audio_effect_eq.h` | ソース | イコライザー定義 |
| audio_effect_eq.cpp | `servers/audio/effects/audio_effect_eq.cpp` | ソース | イコライザー実装 |
| reverb_filter.h | `servers/audio/effects/reverb_filter.h` | ソース | リバーブフィルター |
| eq_filter.h | `servers/audio/effects/eq_filter.h` | ソース | EQフィルター |
