# 機能設計書 22-オーディオバス

## 概要

本ドキュメントは、Godot Engineのオーディオバス機能について、その設計と実装の詳細を記述した機能設計書である。オーディオバスは、複数のオーディオソースをグループ化してミキシングし、ボリューム調整やエフェクト適用を一括管理するためのルーティングシステムを提供する。

### 本機能の処理概要

**業務上の目的・背景**：ゲームオーディオでは、BGM、SE（効果音）、ボイス、環境音など、異なる種類の音声を独立して制御する必要がある。オーディオバス機能により、カテゴリごとに音量を一括調整したり、特定のカテゴリにのみエフェクトを適用したりすることが可能となる。これは、プロフェッショナルなオーディオミキシングの基本的なワークフローを実現する。

**機能の利用シーン**：
- ゲーム設定画面でBGM、SE、ボイスの音量を個別に調整
- 特定のバス（例：環境音）にのみリバーブエフェクトを適用
- マスターバスでゲーム全体の音量を一括制御
- サイドチェーンコンプレッションでBGMをボイスに連動させて自動減衰

**主要な処理内容**：
1. バスの作成・削除・並び替え
2. バス間のルーティング設定（send先の指定）
3. バスごとのボリューム、ソロ、ミュート制御
4. バスへのオーディオエフェクトの追加・削除・有効化
5. リアルタイムでのオーディオフレームミキシング

**関連システム・外部連携**：AudioStreamPlayer（2D/3D）から出力バスを指定して音声を送信。AudioEffectと連携してエフェクト処理を適用。

**権限による制御**：特に権限による制御はなく、すべてのユーザーがバスシステムを使用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 16 | オーディオバス | 主画面 | オーディオバスレイアウトの編集・ミキシング設定 |

## 機能種別

リアルタイム信号処理 / 設定管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| p_bus | int | Yes | バスインデックス | 0以上、バス数未満 |
| p_volume_db | float | No | ボリューム（dB） | -80.0〜24.0推奨 |
| p_send | StringName | No | 送信先バス名 | 有効なバス名 |
| p_effect | Ref<AudioEffect> | No | 追加するエフェクト | NULL不可 |

### 入力データソース

- AudioStreamPlaybackからのオーディオフレーム
- プロジェクト設定からのデフォルトバスレイアウト
- AudioBusLayoutリソースファイル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ミキシング結果 | AudioFrame[] | 全バスをミックスした最終オーディオ |
| ピークボリューム | float | 各バスの左右チャンネルピーク値（dB） |

### 出力先

マスターバスからAudioDriverへ出力され、最終的にシステムのオーディオ出力デバイスへ送信される。

## 処理フロー

### 処理シーケンス

```
1. AudioStreamPlaybackが指定バスにオーディオを送信
   └─ バスごとのチャンネルバッファに蓄積
2. AudioServerの_mix_step関数が実行
   └─ 各バスのチャンネルバッファを処理
3. バスごとにエフェクトチェーンを適用
   └─ 有効なエフェクトのみ順次処理
4. ボリューム・ソロ・ミュート設定を反映
   └─ soledフラグの計算とミュート判定
5. send先バスへミックス結果を加算
   └─ 再帰的にマスターバスへ集約
6. マスターバスの出力をAudioDriverへ送信
```

### フローチャート

```mermaid
flowchart TD
    A[AudioStreamPlayback] --> B[指定バスへ送信]
    B --> C{エフェクト有効?}
    C -->|Yes| D[エフェクト処理]
    C -->|No| E[バイパス]
    D --> F[ボリューム適用]
    E --> F
    F --> G{ミュート?}
    G -->|Yes| H[出力なし]
    G -->|No| I{send先あり?}
    I -->|Yes| J[send先バスへ加算]
    I -->|No| K[マスターへ直接]
    J --> L[マスターバス]
    K --> L
    L --> M[AudioDriver出力]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-22-01 | マスターバス必須 | インデックス0は常にMasterバスとして存在 | 常時 |
| BR-22-02 | ソロ排他 | ソロが有効なバスがあると、他のバスはミュート扱い | ソロ有効時 |
| BR-22-03 | 循環参照禁止 | send先の設定で循環参照が発生しないこと | send設定時 |
| BR-22-04 | チャンネル数同期 | バスのチャンネル数はスピーカーモードに依存 | 常時 |

### 計算ロジック

**ボリューム計算**：
- volume_db（dB値）をlinear値に変換して適用
- db_to_linear関数: `pow(10.0, p_db / 20.0)`

**ソロ判定**：
- soledフラグ = 自身がソロ、またはsend先がsoled
- 全バスでソロがなければ全バスが有効

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作はなし。バスレイアウトはリソースファイル（.tres）として保存。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_FAIL_INDEX | インデックス範囲外 | 存在しないバスインデックス指定 | ERR_FAIL_INDEX_V で早期リターン |
| - | バス名重複 | 既存バス名と同名を設定 | 自動的に連番を付与 |

### リトライ仕様

リアルタイム処理のため、リトライ機構はなし。

## トランザクション仕様

バス設定変更時はlock/unlockでスレッドセーフティを確保。

## パフォーマンス要件

- バス数：最大128バス（実用上は10-20程度推奨）
- チャンネル数：スピーカーモードにより1〜4チャンネル/バス
- 処理時間：オーディオバッファサイズ内でミキシング完了

## セキュリティ考慮事項

- バスレイアウトはプロジェクト内部リソースとして管理
- 外部からの直接操作は不可

## 備考

- デフォルトでMasterバスのみ存在
- エディタのオーディオバスパネルでGUI編集可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

バスとチャンネルの内部構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | audio_server.h | `servers/audio/audio_server.h` | Bus構造体、Channel構造体の定義 |

**読解のコツ**: Bus構造体は238-271行目に定義されており、name、solo、mute、bypass、channels、effects、volume_db、sendなどのメンバを持つ。Channel構造体はバッファとエフェクトインスタンスを保持。

#### Step 2: エントリーポイントを理解する

AudioServerのバス管理APIを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | audio_server.h | `servers/audio/audio_server.h` | set_bus_count、add_bus、remove_bus等のAPI定義（381-423行目） |
| 2-2 | audio_server.cpp | `servers/audio/audio_server.cpp` | バス操作メソッドの実装 |

**主要処理フロー**:
1. **381行目**: set_bus_count - バス数の設定
2. **389行目**: set_bus_name - バス名の設定
3. **395-396行目**: set_bus_volume_db / get_bus_volume_db - ボリューム設定
4. **401-402行目**: set_bus_send / get_bus_send - 送信先設定
5. **413-418行目**: add_bus_effect / remove_bus_effect - エフェクト管理

#### Step 3: ミキシング処理を理解する

リアルタイムミキシングの流れを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | audio_server.cpp | `servers/audio/audio_server.cpp` | _driver_process、_mix_step関数 |

**主要処理フロー**:
- **261行目〜**: _driver_process - ドライバーからの呼び出しエントリ
- **276-279行目**: _mix_stepの呼び出し
- **283-295行目**: マスターバスから出力バッファへの書き込み

### プログラム呼び出し階層図

```
AudioDriver::audio_server_process
    │
    └─ AudioServer::_driver_process
           │
           ├─ AudioServer::_mix_step
           │      │
           │      ├─ playback_list処理（各AudioStreamPlayback）
           │      │
           │      ├─ Bus::Channel::effect_instances処理
           │      │
           │      └─ send先バスへの加算処理
           │
           └─ マスターバス出力処理
                  │
                  └─ p_buffer（int32_t*）への書き込み
```

### データフロー図

```
[入力]                      [処理]                        [出力]

AudioStreamPlayer ────┐
                      │
AudioStreamPlayer2D ──┼──▶ 指定Bus ──▶ Effect Chain ──┐
                      │                                │
AudioStreamPlayer3D ──┘                                ▼
                                                  send先Bus
                                                       │
                                                       ▼
                                                   Master Bus
                                                       │
                                                       ▼
                                                  AudioDriver
                                                  (スピーカー)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| audio_server.h | `servers/audio/audio_server.h` | ソース | AudioServer、Bus、Channelクラス定義 |
| audio_server.cpp | `servers/audio/audio_server.cpp` | ソース | バス管理とミキシング実装 |
| audio_effect.h | `servers/audio/audio_effect.h` | ソース | エフェクト基底クラス |
| audio_stream_player.cpp | `scene/audio/audio_stream_player.cpp` | ソース | バスへの出力元 |
| audio_stream_player_2d.cpp | `scene/2d/audio_stream_player_2d.cpp` | ソース | 2D空間オーディオ |
| audio_stream_player_3d.cpp | `scene/3d/audio_stream_player_3d.cpp` | ソース | 3D空間オーディオ |
