# 機能設計書 26-入力マッピング

## 概要

本ドキュメントは、Godot Engineの入力マッピング機能について、その設計と実装の詳細を記述した機能設計書である。本機能は、キーボード、マウス、ゲームパッド等の物理的な入力を抽象的な「アクション」として定義し、ゲームロジックから入力デバイスの詳細を隠蔽するための機能を提供する。

### 本機能の処理概要

**業務上の目的・背景**：ゲーム開発において、プレイヤーが使用する入力デバイスは多様である。入力マッピング機能により、開発者は「ジャンプ」「攻撃」「移動」といったゲーム固有のアクションを定義し、そのアクションに複数の入力イベント（スペースキー、Aボタン等）を割り当てることができる。これにより、キーコンフィグ機能の実装や、異なるプラットフォームへの対応が容易になる。

**機能の利用シーン**：
- プレイヤーがジャンプキーをスペースからEnterに変更
- Xbox コントローラーとPlayStationコントローラーで同じアクションを使用
- アクセシビリティ対応のためのカスタムキーバインド
- デバッグ用のショートカットキー設定

**主要な処理内容**：
1. アクションの追加・削除・取得
2. アクションへの入力イベント（キー、ボタン、軸）の割り当て
3. 入力イベントからアクションへのマッチング
4. デッドゾーン設定と適用
5. プロジェクト設定からのアクション読み込み

**関連システム・外部連携**：Inputシングルトン（アクション状態の問い合わせ）、ProjectSettings（アクション定義の保存）

**権限による制御**：特に権限による制御はなし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 21 | プロジェクト設定 | 主機能 | Input Map（アクション・キーバインド）設定 |
| 36 | 入力イベント設定ダイアログ | 主機能 | 入力イベントのキー/ボタン割り当て |

## 機能種別

設定管理 / イベント照合

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| p_action | StringName | Yes | アクション名 | 空文字列不可 |
| p_deadzone | float | No | デッドゾーン（0.0〜1.0） | DEFAULT_DEADZONE = 0.2 |
| p_event | Ref<InputEvent> | Yes | 割り当てる入力イベント | NULL不可 |

### 入力データソース

- プロジェクト設定ファイル（project.godot）のinput/セクション
- スクリプトからのAPI呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| has_action | bool | アクションが存在するか |
| get_actions | TypedArray<StringName> | 全アクション名の配列 |
| action_get_events | List<Ref<InputEvent>>* | アクションに割り当てられた入力イベント一覧 |
| event_is_action | bool | 入力イベントがアクションにマッチするか |

### 出力先

- ゲームスクリプトからのAPI呼び出し結果
- Input::is_action_pressed等の判定に使用

## 処理フロー

### 処理シーケンス

```
1. アクションの定義
   └─ add_action(p_action, p_deadzone)
2. 入力イベントの割り当て
   └─ action_add_event(p_action, p_event)
3. 入力発生時のマッチング
   └─ event_is_action(p_event, p_action) or event_get_action_status
4. デッドゾーン適用
   └─ |value| < deadzone → pressed = false
5. 結果をInputシングルトンへ返却
   └─ action_statesの更新に使用
```

### フローチャート

```mermaid
flowchart TD
    A[アクション定義] --> B[入力イベント割り当て]
    B --> C[入力発生]
    C --> D{event_is_action}
    D -->|検索| E[_find_event]
    E --> F{イベントマッチ?}
    F -->|Yes| G[デッドゾーン適用]
    F -->|No| H[次のイベント検索]
    G --> I{|value| >= deadzone?}
    I -->|Yes| J[pressed = true, strength計算]
    I -->|No| K[pressed = false]
    H --> F
    J --> L[結果返却]
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-26-01 | デフォルトデッドゾーン | DEFAULT_DEADZONE = 0.2 | アクション追加時 |
| BR-26-02 | トグルデッドゾーン | DEFAULT_TOGGLE_DEADZONE = 0.5（ui_*アクション用） | ui_*アクション |
| BR-26-03 | デバイスフィルタ | ALL_DEVICES = -1で全デバイスにマッチ | イベントマッチング時 |
| BR-26-04 | 完全一致 | exact_match = trueで修飾キーも含めて完全一致 | オプション |

### 計算ロジック

**strength計算**：
- デジタル入力：pressed ? 1.0 : 0.0
- アナログ入力：(|value| - deadzone) / (1.0 - deadzone)

**マッチング優先順位**：
- アクションに登録された順序で検索
- 最初にマッチしたイベントの結果を使用

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作はなし。アクションはinput_map（HashMap<StringName, Action>）としてメモリ上で管理。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | アクション未登録 | 存在しないアクション名指定 | ERR_FAIL_COND_MSG + サジェスト表示 |
| - | アクション重複 | 既存アクション名で追加 | ERR_FAIL_COND_MSG |
| - | 無効なイベント | NULLイベント指定 | EXTRACT_PARAM_OR_FAIL_MSG |

### リトライ仕様

リトライ機構はなし。エラー時は操作をスキップ。

## トランザクション仕様

トランザクション管理は不要。

## パフォーマンス要件

- イベントマッチング：O(n)（nはアクションに登録されたイベント数）
- アクション検索：O(1)（HashMapによる）

## セキュリティ考慮事項

- アクション名・イベントは信頼されたソースからのみ設定

## 備考

- ビルトインアクション（ui_accept、ui_cancel等）はload_defaultで自動登録
- suggest_actions関数で類似アクション名を提案

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

アクションと入力マップの内部構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | input_map.h | `core/input/input_map.h` | InputMapシングルトン、Action構造体 |

**読解のコツ**: Action構造体（49-53行目）はid、deadzone、inputs（List<Ref<InputEvent>>）を持ち、1つのアクションに対する全設定を保持。

#### Step 2: エントリーポイントを理解する

アクション管理APIを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | input_map.cpp | `core/input/input_map.cpp` | add_action、action_add_event、event_is_action |
| 2-2 | input_map.h | `core/input/input_map.h` | _find_event、event_get_action_status |

**主要処理フロー**:
1. **115-122行目**: add_action - アクション追加
2. **199行目〜**: action_add_event - イベント追加
3. **144-162行目**: _find_event - イベント検索
4. **96-97行目**: event_is_action、event_get_action_status - マッチング判定

#### Step 3: プロジェクト設定との連携を理解する

設定ファイルからの読み込みを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | input_map.cpp | `core/input/input_map.cpp` | load_from_project_settings |

**主要処理フロー**:
- **101行目**: load_from_project_settings - プロジェクト設定読み込み
- **102行目**: load_default - ビルトインアクション読み込み

### プログラム呼び出し階層図

```
InputMap（シングルトン）
    │
    ├─ add_action / erase_action
    │      └─ input_map HashMap操作
    │
    ├─ action_add_event / action_erase_event
    │      └─ Action::inputs List操作
    │
    └─ event_is_action / event_get_action_status
           │
           └─ _find_event
                  │
                  └─ InputEvent::action_match
                         └─ デッドゾーン適用、strength計算
```

### データフロー図

```
[入力]                       [処理]                        [出力]

project.godot ────────┐
(input/セクション)      │
                      ├──▶ InputMap::load_from_project_settings
スクリプトAPI ────────┘                │
(add_action等)                        ▼
                               input_map HashMap
                                      │
                      ┌───────────────┼───────────────┐
                      │               │               │
                      ▼               ▼               ▼
              Action定義        Event登録      Matching
              (deadzone)      (inputs List)   (_find_event)
                                                    │
                                                    ▼
                                          Input::action_states更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| input_map.h | `core/input/input_map.h` | ソース | InputMapシングルトン、Action構造体定義 |
| input_map.cpp | `core/input/input_map.cpp` | ソース | アクション管理実装 |
| input.h | `core/input/input.h` | ソース | ActionState、アクション状態問い合わせ |
| input.cpp | `core/input/input.cpp` | ソース | アクション状態更新 |
| input_event.h | `core/input/input_event.h` | ソース | action_matchメソッド定義 |
| input_event.cpp | `core/input/input_event.cpp` | ソース | action_match実装 |
