# 機能設計書 27-UIコントロール

## 概要

本ドキュメントは、Godot EngineのUIコントロール機能について、その設計と実装の詳細を記述した機能設計書である。本機能は、Button、Label、TextEdit等のユーザーインターフェース要素を提供し、ゲームやアプリケーションのGUI構築を可能にする。

### 本機能の処理概要

**業務上の目的・背景**：ゲームやアプリケーションには、メニュー、HUD、設定画面など多様なユーザーインターフェースが必要である。UIコントロール機能により、開発者は標準的なUI要素（ボタン、ラベル、入力フィールド等）を使用して、一貫性のあるユーザー体験を構築できる。また、Controlクラスを基底として継承階層が整理されており、カスタムUIの作成も容易である。

**機能の利用シーン**：
- ゲームメニュー（スタート、設定、終了ボタン）
- インベントリ画面（アイテムリスト、詳細表示）
- ダイアログボックス（確認、警告メッセージ）
- HUD（体力バー、スコア表示）
- フォーム入力（テキストフィールド、チェックボックス）

**主要な処理内容**：
1. レイアウト管理（アンカー、マージン、サイズフラグ）
2. 入力イベント処理（_gui_input）
3. フォーカス管理
4. テーマによるスタイリング
5. 最小サイズ計算と描画

**関連システム・外部連携**：Theme（スタイリング）、Container（レイアウト）、Viewport（イベント伝播）

**権限による制御**：特に権限による制御はなし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 9 | インスペクタードック | 補助機能 | プロパティエディタのUI表示 |

## 機能種別

GUI構築 / イベント処理 / レイアウト管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| p_anchor | Anchor | No | アンカー位置（ANCHOR_BEGIN/END） | 0.0〜1.0 |
| p_size_flags | SizeFlags | No | サイズフラグ（FILL/EXPAND等） | 有効なフラグ組み合わせ |
| p_focus_mode | FocusMode | No | フォーカスモード | NONE/CLICK/ALL |

### 入力データソース

- シーンファイル（.tscn）からのプロパティ設定
- スクリプトからのAPI呼び出し
- Themeリソースからのスタイル情報

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| get_minimum_size | Size2 | コントロールの最小サイズ |
| get_combined_minimum_size | Size2 | 子要素含む最小サイズ |
| get_rect | Rect2 | 現在の矩形領域 |

### 出力先

- 画面への描画（CanvasItemを通じて）
- シグナル発行（pressed、focus_entered等）

## 処理フロー

### 処理シーケンス

```
1. Controlノードがシーンツリーに追加
   └─ _enter_treeで初期化
2. 親Containerまたはアンカーに基づきレイアウト計算
   └─ _notification(NOTIFICATION_RESIZED)
3. テーマからスタイル情報を取得
   └─ _update_theme_item_cache
4. 入力イベントを受信
   └─ _gui_input → pressed等のシグナル発行
5. 描画
   └─ _notification(NOTIFICATION_DRAW)
```

### フローチャート

```mermaid
flowchart TD
    A[Controlノード作成] --> B[シーンツリー追加]
    B --> C[レイアウト計算]
    C --> D[テーマ適用]
    D --> E[描画]
    E --> F{入力イベント?}
    F -->|Yes| G[_gui_input処理]
    G --> H{フォーカス可能?}
    H -->|Yes| I[フォーカス取得]
    H -->|No| J[イベント伝播]
    I --> K[シグナル発行]
    F -->|No| E
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-27-01 | アンカーシステム | ANCHOR_BEGIN(0)〜ANCHOR_END(1)で親に対する相対位置を指定 | レイアウト時 |
| BR-27-02 | サイズフラグ | FILL、EXPAND、SHRINK_CENTER等でレイアウト動作を制御 | Container内 |
| BR-27-03 | フォーカス順序 | Tab/Shift+Tabでフォーカス移動、focus_next/prevで明示指定可 | キーボード操作時 |
| BR-27-04 | マウスフィルター | STOP/PASS/IGNOREでマウスイベントの伝播を制御 | マウス入力時 |

### 計算ロジック

**アンカー計算**：
- 位置 = 親サイズ * anchor + offset
- GrowDirection（BEGIN/END/BOTH）でサイズ変更方向を指定

**最小サイズ計算**：
- custom_minimum_sizeと内部計算の最大値
- 子要素の最小サイズを再帰的に考慮

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作はなし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 無効なアンカー値 | 範囲外の値指定 | CLAMPで0.0〜1.0に制限 |

### リトライ仕様

リトライ機構はなし。

## トランザクション仕様

トランザクション管理は不要。

## パフォーマンス要件

- 描画：フレームレートに同期（60FPS時16.6ms以内）
- レイアウト再計算：queue_sortで遅延実行、バッチ処理

## セキュリティ考慮事項

- TextEdit等の入力フィールドは適切な入力検証が必要

## 備考

- RTL（右から左）レイアウトをサポート
- アクセシビリティ対応のフォーカス管理

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Controlクラスの構造とenum定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | control.h | `scene/gui/control.h` | Control クラス、Anchor、SizeFlags、FocusMode等のenum定義 |

**読解のコツ**: Controlクラスは多数のenumを持つ。Anchor(54-58)、GrowDirection(59-63)、FocusMode(65-70)、SizeFlags(78-86)、MouseFilter(88-92)等を先に把握すると理解が進む。

#### Step 2: エントリーポイントを理解する

レイアウトと描画の流れを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | control.cpp | `scene/gui/control.cpp` | _notification、_gui_input |

**主要処理フロー**:
1. **NOTIFICATION_RESIZED**: サイズ変更通知
2. **NOTIFICATION_DRAW**: 描画通知
3. **NOTIFICATION_FOCUS_ENTER/EXIT**: フォーカス変更

#### Step 3: 具体的なコントロールを理解する

代表的なUIコントロールの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | button.h/cpp | `scene/gui/button.h`, `scene/gui/button.cpp` | Buttonクラスの実装 |
| 3-2 | label.h/cpp | `scene/gui/label.h`, `scene/gui/label.cpp` | Labelクラスの実装 |
| 3-3 | text_edit.h/cpp | `scene/gui/text_edit.h`, `scene/gui/text_edit.cpp` | TextEditクラスの実装 |

### プログラム呼び出し階層図

```
Control（基底クラス）
    │
    ├─ BaseButton（ボタン基底）
    │      ├─ Button
    │      ├─ CheckBox
    │      ├─ CheckButton
    │      └─ TextureButton
    │
    ├─ Range（範囲入力基底）
    │      ├─ ScrollBar
    │      ├─ Slider
    │      ├─ SpinBox
    │      └─ ProgressBar
    │
    ├─ Label
    │
    ├─ TextEdit
    │      └─ CodeEdit
    │
    └─ Container（レイアウト基底）
           ├─ BoxContainer
           ├─ GridContainer
           └─ ...
```

### データフロー図

```
[入力]                       [処理]                         [出力]

シーンファイル ─────────┐
                       │
スクリプトAPI ─────────┼──▶ Control インスタンス ──▶ レイアウト計算
                       │              │
Themeリソース ─────────┘              │
                                      ▼
                           ┌──────────┴──────────┐
                           │                      │
                           ▼                      ▼
                       描画処理              入力処理
                    (CanvasItem)          (_gui_input)
                           │                      │
                           ▼                      ▼
                       画面表示              シグナル発行
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| control.h | `scene/gui/control.h` | ソース | Control基底クラス定義 |
| control.cpp | `scene/gui/control.cpp` | ソース | Control実装 |
| base_button.h | `scene/gui/base_button.h` | ソース | ボタン基底クラス |
| button.h | `scene/gui/button.h` | ソース | Buttonクラス定義 |
| label.h | `scene/gui/label.h` | ソース | Labelクラス定義 |
| text_edit.h | `scene/gui/text_edit.h` | ソース | TextEditクラス定義 |
| range.h | `scene/gui/range.h` | ソース | 範囲入力基底クラス |
| container.h | `scene/gui/container.h` | ソース | コンテナ基底クラス |
