# 機能設計書: AnimationPlayer

## 1. 機能概要

### 1.1 機能名
AnimationPlayer - キーフレームアニメーション再生システム

### 1.2 機能説明
AnimationPlayerは、Godotエンジンにおけるキーフレームベースのアニメーション再生を担当するノードである。Animationリソースを管理し、指定されたアニメーションを再生・停止・ブレンドする機能を提供する。AnimationLibraryを通じて複数のアニメーションを整理し、トラックベースでノードのプロパティ（位置、回転、スケール、カスタムプロパティ等）を時間軸に沿って変化させる。

### 1.3 関連画面
- アニメーションエディタパネル
- インスペクターパネル（AnimationPlayerプロパティ）

## 2. 機能詳細

### 2.1 主要機能一覧

| 機能ID | 機能名 | 説明 |
|--------|--------|------|
| F31-01 | アニメーション再生 | 指定したアニメーションを再生する |
| F31-02 | アニメーション停止 | 現在再生中のアニメーションを停止する |
| F31-03 | アニメーションブレンド | 複数アニメーション間のクロスフェード |
| F31-04 | アニメーションライブラリ管理 | AnimationLibraryの追加・削除・管理 |
| F31-05 | 再生制御 | 再生速度、ループ、逆再生の制御 |
| F31-06 | アニメーションキュー | アニメーションのキュー再生 |
| F31-07 | コールバック処理 | アニメーション終了時などのシグナル発行 |

### 2.2 クラス構造

```
AnimationMixer (基底クラス)
    └── AnimationPlayer (派生クラス)
```

### 2.3 主要プロパティ

| プロパティ名 | 型 | デフォルト値 | 説明 |
|-------------|-----|-------------|------|
| current_animation | StringName | "" | 現在再生中のアニメーション名 |
| assigned_animation | StringName | "" | 割り当てられたアニメーション名 |
| autoplay | String | "" | シーン開始時に自動再生するアニメーション |
| current_animation_position | double | 0.0 | 現在の再生位置（秒） |
| current_animation_length | double | 0.0 | 現在のアニメーション長さ（秒） |
| playback_default_blend_time | double | 0.0 | デフォルトのブレンド時間 |
| speed_scale | float | 1.0 | 再生速度スケール |
| movie_quit_on_finish | bool | false | ムービー終了時にゲーム終了 |

### 2.4 主要メソッド

| メソッド名 | 戻り値 | 説明 |
|-----------|--------|------|
| play(name, custom_blend, custom_speed, from_end) | void | アニメーションを再生 |
| play_backwards(name, custom_blend) | void | アニメーションを逆再生 |
| pause() | void | 再生を一時停止 |
| stop(keep_state) | void | 再生を停止 |
| is_playing() | bool | 再生中かどうかを返す |
| queue(name) | void | アニメーションをキューに追加 |
| clear_queue() | void | キューをクリア |
| get_queue() | PackedStringArray | キュー内のアニメーション一覧を取得 |
| seek(seconds, update, update_only) | void | 指定位置にシーク |
| set_blend_time(from, to, sec) | void | ブレンド時間を設定 |
| get_blend_time(from, to) | double | ブレンド時間を取得 |

### 2.5 シグナル

| シグナル名 | 引数 | 説明 |
|-----------|------|------|
| animation_finished | anim_name: StringName | アニメーション再生完了時 |
| animation_changed | old_name, new_name | アニメーション変更時 |
| animation_started | anim_name: StringName | アニメーション開始時 |
| current_animation_changed | name: String | 現在のアニメーション変更時 |

## 3. 処理フロー

### 3.1 アニメーション再生フロー

```
[play()呼び出し]
      │
      ▼
[アニメーション存在確認]
      │
      ├─ 存在しない → エラー終了
      │
      ▼
[ブレンド処理判定]
      │
      ├─ ブレンドあり → [_blend_pre_process()]
      │                      │
      │                      ▼
      │               [ブレンド設定適用]
      │
      ▼
[PlaybackData設定]
      │
      ▼
[_process_animation() - 毎フレーム]
      │
      ├─ アニメーション完了 → [animation_finished シグナル発行]
      │                           │
      │                           ▼
      │                     [次のキュー確認]
      │                           │
      │                           ├─ キューあり → [次アニメーション再生]
      │                           │
      │                           └─ キューなし → [停止]
      │
      └─ 再生継続 → [トラック値更新]
```

### 3.2 データフロー図

```
[AnimationLibrary]        [Animation Resource]
        │                         │
        ▼                         ▼
   ┌─────────────────────────────────────┐
   │         AnimationMixer              │
   │  - animation_set (キャッシュ)       │
   │  - track_cache (トラックキャッシュ) │
   └─────────────────────────────────────┘
                    │
                    ▼
   ┌─────────────────────────────────────┐
   │         AnimationPlayer             │
   │  - playback_queue                   │
   │  - blend_times                      │
   │  - playback (PlaybackData)          │
   └─────────────────────────────────────┘
                    │
                    ▼
           [ターゲットノード]
           (プロパティ更新)
```

## 4. コードリーディングガイド

### 4.1 推奨読解順序

1. **データ構造の理解**
   - `scene/animation/animation_player.h` (31-134行目): AnimationPlayerクラス定義
   - `scene/animation/animation_mixer.h`: 基底クラスAnimationMixer定義

2. **エントリーポイント**
   - `animation_player.cpp` **_bind_methods()** (674-770行目): GDScriptバインディング
   - `animation_player.cpp` **play()** (134-206行目): 再生開始処理

3. **コア処理**
   - `animation_mixer.cpp` **_process_animation()**: アニメーション処理のメインループ
   - `animation_player.cpp` **_animation_process()** (254-350行目): フレーム更新処理

### 4.2 重要な処理ポイント

#### play()メソッドの処理（animation_player.cpp: 134-206行目）
```cpp
void AnimationPlayer::play(const StringName &p_name, double p_custom_blend,
                           float p_custom_scale, bool p_from_end) {
    // アニメーションの存在確認
    // ブレンド時間の決定
    // 再生位置の設定
    // 再生状態の更新
}
```

#### _animation_process_data()の処理
- キーフレーム補間計算
- トラック値の適用
- ループ処理の判定

### 4.3 プログラム呼び出し階層図

```
AnimationPlayer::play()
    │
    ├── AnimationMixer::_check_animation()
    │       └── animation_set.has(name)
    │
    ├── AnimationPlayer::_get_blend_time()
    │       └── blend_times.get()
    │
    ├── AnimationPlayer::_set_process()
    │       └── set_process_internal() / set_physics_process_internal()
    │
    └── AnimationMixer::_clear_playing_caches()
            └── track_cache.clear()

AnimationPlayer::_animation_process()  [毎フレーム呼び出し]
    │
    ├── AnimationMixer::_process_animation()
    │       │
    │       ├── _blend_process()
    │       │       └── _blend_calc_total_weight()
    │       │
    │       └── _apply_animation()
    │               └── _apply_track_value()
    │
    └── AnimationPlayer::_check_animation_finished()
            │
            ├── emit_signal("animation_finished")
            │
            └── _queue_process()
                    └── play(next_queued)
```

### 4.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| scene/animation/animation_player.h | ヘッダー | AnimationPlayerクラス定義 |
| scene/animation/animation_player.cpp | 実装 | AnimationPlayerの実装 |
| scene/animation/animation_mixer.h | ヘッダー | 基底クラスAnimationMixer定義 |
| scene/animation/animation_mixer.cpp | 実装 | アニメーションミキシング処理 |
| scene/resources/animation.h | ヘッダー | Animationリソース定義 |
| scene/resources/animation.cpp | 実装 | キーフレームデータ管理 |
| scene/resources/animation_library.h | ヘッダー | AnimationLibrary定義 |
| scene/resources/animation_library.cpp | 実装 | アニメーションライブラリ管理 |

## 5. 設計上の考慮事項

### 5.1 パフォーマンス
- トラックキャッシュによる高速なプロパティアクセス
- 増分更新による不要な計算の削減
- デルタタイム制御によるフレームレート非依存の再生

### 5.2 拡張性
- AnimationMixerを基底クラスとした階層設計
- AnimationLibraryによるモジュラーなアニメーション管理
- カスタムトラックタイプのサポート

### 5.3 互換性
- Godot 3.xからの移行サポート（anims/プレフィックス対応）
- 旧形式のアニメーションデータの自動変換

## 6. 使用例

### 6.1 基本的なアニメーション再生
```gdscript
# アニメーションプレイヤーの取得
@onready var anim_player = $AnimationPlayer

func _ready():
    # アニメーション再生
    anim_player.play("walk")

    # 逆再生
    anim_player.play_backwards("jump")

    # ブレンド付き再生
    anim_player.play("run", 0.5)  # 0.5秒でブレンド

func _on_animation_player_animation_finished(anim_name):
    if anim_name == "attack":
        anim_player.play("idle")
```

### 6.2 キュー再生
```gdscript
func play_combo():
    anim_player.play("attack1")
    anim_player.queue("attack2")
    anim_player.queue("attack3")
```

## 7. 関連機能
- [No.32 AnimationTree](./32-AnimationTree.md) - 複雑なアニメーションブレンド
- [No.33 Tween](./33-Tween.md) - プログラマティックな補間アニメーション
