# 機能設計書 57-タイルマップシステム

## 概要

本ドキュメントは、Godot Engineにおける2Dタイルマップシステム（TileMapLayer）の設計を記載するものである。

### 本機能の処理概要

**業務上の目的・背景**：2Dゲーム開発において、タイルマップは背景、地形、障害物などを効率的に配置するための基本的なツールである。TileMapLayerは、TileSetリソースに基づいてタイルを配置・描画し、物理コリジョン、ナビゲーション、シーンインスタンス化などの機能を提供する。

**機能の利用シーン**：
- 2Dプラットフォームゲームの地形作成
- トップダウンRPGのマップ作成
- ローグライクゲームのダンジョン生成
- パズルゲームのグリッドベース配置

**主要な処理内容**：
1. タイルの配置・削除・取得
2. レンダリング（クアドラント単位での描画最適化）
3. 物理コリジョンの生成と更新
4. ナビゲーションメッシュの生成
5. Y-Sort（深度ソート）対応
6. アニメーションタイルの再生
7. テレインマッチング（自動タイル接続）

**関連システム・外部連携**：
- TileSet（タイルデータ定義）
- PhysicsServer2D（コリジョン処理）
- NavigationServer2D（経路探索）
- RenderingServer（描画）

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | TileMapエディタ | エディタ | タイルの配置・編集 |
| - | TileSetエディタ | エディタ | タイルセットの定義 |

## 機能種別

2Dゲームオブジェクト / シーンノード

## 入力仕様

### プロパティ

| プロパティ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| tile_set | TileSet | Yes | 使用するタイルセット | - |
| enabled | Bool | No | レイヤー有効/無効 | デフォルトtrue |
| y_sort_enabled | Bool | No | Y-Sort有効化 | - |
| y_sort_origin | Int | No | Y-Sortの原点オフセット | - |
| x_draw_order_reversed | Bool | No | X軸描画順序反転 | - |
| rendering_quadrant_size | Int | No | レンダリングクアドラントサイズ | デフォルト16 |

### メソッド入力

| メソッド名 | パラメータ | 説明 |
|-----------|-----------|------|
| set_cell | coords: Vector2i, source_id: int, atlas_coords: Vector2i, alternative_tile: int | タイルを配置 |
| get_cell_source_id | coords: Vector2i | 指定座標のソースIDを取得 |
| get_cell_atlas_coords | coords: Vector2i | 指定座標のアトラス座標を取得 |
| get_used_cells | - | 使用中のセル座標リストを取得 |
| local_to_map | local_position: Vector2 | ローカル座標をマップ座標に変換 |
| map_to_local | map_position: Vector2i | マップ座標をローカル座標に変換 |

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 描画済みタイル | CanvasItem | レンダリングされたタイル画像 |
| コリジョン形状 | PhysicsBody2D | 物理衝突形状 |
| ナビゲーション領域 | NavigationRegion2D | 経路探索用領域 |

## 処理フロー

### 処理シーケンス

```
1. タイルデータの更新検出
   └─ dirty flagによる変更追跡
2. レンダリングクアドラントの更新
   └─ 変更されたセルを含むクアドラントの再構築
3. 物理クアドラントの更新
   └─ コリジョン形状の再生成
4. ナビゲーションの更新
   └─ ナビゲーションメッシュの再生成
5. シーンの更新
   └─ インスタンス化されたシーンの更新
```

### フローチャート

```mermaid
flowchart TD
    A[set_cell呼び出し] --> B[セルデータ更新]
    B --> C[dirty flag設定]
    C --> D{update処理}
    D --> E[_rendering_update]
    D --> F[_physics_update]
    D --> G[_navigation_update]
    D --> H[_scenes_update]
    E --> I[クアドラント描画]
    F --> J[コリジョン生成]
    G --> K[NavMesh生成]
    H --> L[シーンインスタンス化]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-57-01 | クアドラント最適化 | 同一マテリアル・Z-indexのタイルは同一CanvasItemで描画 | 常時 |
| BR-57-02 | Y-Sort時の描画順序 | Y座標 + y_sort_originでソート | y_sort_enabled時 |
| BR-57-03 | 座標変換 | タイル形状（正方形/アイソメトリック/六角形）に応じた座標変換 | 常時 |

### 計算ロジック

クアドラント座標計算:
```
quadrant_coords.x = coords.x > 0 ? coords.x / quadrant_size : (coords.x - (quadrant_size - 1)) / quadrant_size
quadrant_coords.y = coords.y > 0 ? coords.y / quadrant_size : (coords.y - (quadrant_size - 1)) / quadrant_size
```

## データベース操作仕様

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_INVALID_DATA | データ不正 | TileSetが未設定 | TileSetを設定 |
| ERR_INVALID_PARAMETER | パラメータ不正 | 無効なsource_id | 有効なソースIDを指定 |

## トランザクション仕様

即時反映。トランザクション管理なし。

## パフォーマンス要件

- クアドラント単位での描画最適化により大量タイルに対応
- dirty flagによる差分更新で不要な再計算を回避

## セキュリティ考慮事項

特になし。

## 備考

- Godot 4.0でTileMapからTileMapLayerに分離
- 複数レイヤーは複数のTileMapLayerノードで実現

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | tile_map_layer.h | `scene/2d/tile_map_layer.h` | TileMapLayerクラス定義、CellData構造体 |

**読解のコツ**: CellData構造体がセルごとのデータを保持し、dirty flagsで変更を追跡する。

#### Step 2: レンダリング処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | tile_map_layer.cpp | `scene/2d/tile_map_layer.cpp` | _rendering_update関数 |

**主要処理フロー**:
- **53-57行目**: クアドラント座標計算（_coords_to_quadrant_coords）
- **220-400行目**: _rendering_update関数でレンダリングクアドラント更新
- **282-312行目**: クアドラント内のセルソート処理
- **318-385行目**: CanvasItem生成とタイル描画

#### Step 3: 物理処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | tile_map_layer.cpp | `scene/2d/tile_map_layer.cpp` | _physics_update関数 |

**主要処理フロー**:
- 物理クアドラント単位でStaticBody2Dを生成
- TileDataからコリジョン形状を取得して配置

#### Step 4: デバッグ描画を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | tile_map_layer.cpp | `scene/2d/tile_map_layer.cpp` | _debug_update関数 |

**主要処理フロー**:
- **63-203行目**: デバッグクアドラントの管理と描画
- **176-177行目**: 物理・ナビゲーション・シーンのデバッグ描画

### プログラム呼び出し階層図

```
TileMapLayer
    │
    ├─ set_cell
    │      └─ _set_cell_internal
    │             └─ dirty flag設定
    │
    ├─ _internal_update
    │      ├─ _rendering_update
    │      │      ├─ _rendering_quadrants_update_cell
    │      │      └─ draw_tile (描画)
    │      │
    │      ├─ _physics_update
    │      │      └─ PhysicsServer2D::body_add_shape
    │      │
    │      ├─ _navigation_update
    │      │      └─ NavigationServer2D::region_set_navigation_polygon
    │      │
    │      └─ _scenes_update
    │             └─ シーンインスタンス化
    │
    └─ _debug_update
           └─ デバッグ描画
```

### データフロー図

```
[入力]                      [処理]                           [出力]

set_cell呼び出し ──────────▶ CellData更新 ─────────────────▶ tile_map_layer_data

dirty flags ───────────────▶ _internal_update ─────────────▶ 各種サーバー更新

TileSet ───────────────────▶ 座標変換・描画データ取得 ─────▶ CanvasItem描画
                                                              PhysicsBody2D
                                                              NavigationRegion2D
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| tile_map_layer.cpp | `scene/2d/tile_map_layer.cpp` | ソース | TileMapLayer本体 |
| tile_map_layer.h | `scene/2d/tile_map_layer.h` | ヘッダ | クラス定義 |
| tile_map.cpp | `scene/2d/tile_map.cpp` | ソース | 旧TileMap互換クラス |
| tile_set.cpp | `scene/resources/2d/tile_set.cpp` | ソース | TileSetリソース |
| tile_set.h | `scene/resources/2d/tile_set.h` | ヘッダ | TileSetクラス定義 |
