# 機能設計書 76-暗号化

## 概要

本ドキュメントは、Godotエンジンにおける暗号化機能の設計を記述する。RSA暗号、X.509証明書、HMAC、TLS通信などのセキュリティ機能を提供する。

### 本機能の処理概要

暗号化機能は、ゲームやアプリケーションにおけるセキュリティ要件を満たすための暗号化基盤を提供する。mbedTLSライブラリをベースに、RSA鍵生成、証明書管理、署名・検証、暗号化・復号化、HMAC認証などの機能を実装する。

**業務上の目的・背景**：オンラインゲームやネットワーク通信を行うアプリケーションでは、データの機密性、完全性、認証性を確保することが重要である。暗号化機能は、セキュアな通信チャネルの確立、データ改ざん検出、ユーザー認証などのセキュリティ機能を提供する。

**機能の利用シーン**：
- HTTPSによるサーバーとのセキュア通信
- セーブデータの暗号化保護
- ライセンスキーの検証
- マルチプレイヤーゲームでの通信認証
- デジタル署名によるMODの検証

**主要な処理内容**：
1. 暗号論的擬似乱数生成（generate_random_bytes）
2. RSA鍵ペア生成（generate_rsa）
3. 自己署名証明書生成（generate_self_signed_certificate）
4. デジタル署名（sign）と検証（verify）
5. RSA暗号化（encrypt）と復号化（decrypt）
6. HMAC認証コード生成（hmac_digest）
7. タイミング攻撃耐性のある比較（constant_time_compare）
8. TLSオプション設定（client/server）

**関連システム・外部連携**：mbedTLS、HTTPクライアント、WebSocketClient、StreamPeerTLS

**権限による制御**：特になし（すべてのスクリプトから利用可能）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 該当画面なし | - | 暗号化は特定画面に限定されない基盤機能 |

## 機能種別

ユーティリティ / セキュリティ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| size | int | Yes | 生成するバイト数/RSA鍵サイズ | 正の整数 |
| hash_type | HashingContext.HashType | Yes | ハッシュアルゴリズム種別 | enum値チェック |
| key | CryptoKey | Yes | 暗号鍵 | null check |
| hash | PackedByteArray | Yes | ハッシュ値 | - |
| signature | PackedByteArray | Yes(verify) | 署名データ | - |
| plaintext/ciphertext | PackedByteArray | Yes | 平文/暗号文 | - |

### 入力データソース

- スクリプトからのAPI呼び出し
- 鍵/証明書ファイル（.key, .pub, .crt）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| random_bytes | PackedByteArray | 乱数バイト列 |
| crypto_key | CryptoKey | 生成されたRSA鍵 |
| certificate | X509Certificate | 生成された証明書 |
| signature | PackedByteArray | デジタル署名 |
| verification_result | bool | 署名検証結果 |
| encrypted/decrypted | PackedByteArray | 暗号化/復号化結果 |
| hmac | PackedByteArray | HMAC認証コード |

### 出力先

- スクリプトへの結果返却
- ファイルシステム（鍵/証明書保存時）

## 処理フロー

### 処理シーケンス

```
1. Cryptoインスタンス取得
   └─ Crypto.create()でmbedTLS実装を取得
2. 鍵/証明書準備
   └─ generate_rsa()または load()で鍵を準備
3. 暗号操作実行
   └─ sign()/verify()/encrypt()/decrypt()/hmac_digest()
4. 結果取得
   └─ PackedByteArrayまたはboolで結果を返却
```

### フローチャート

```mermaid
flowchart TD
    A[Crypto生成] --> B{操作種別}
    B -->|鍵生成| C[generate_rsa]
    B -->|証明書生成| D[generate_self_signed_certificate]
    B -->|署名| E[sign]
    B -->|検証| F[verify]
    B -->|暗号化| G[encrypt]
    B -->|復号化| H[decrypt]
    B -->|HMAC| I[hmac_digest]
    C --> J[CryptoKey返却]
    D --> K[X509Certificate返却]
    E --> L[署名バイト列返却]
    F --> M[検証結果返却]
    G --> N[暗号文返却]
    H --> O[平文返却]
    I --> P[HMAC返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-76-01 | mbedTLS依存 | 暗号機能はmbedTLSモジュール有効時のみ使用可能 | モジュール無効時はnull返却 |
| BR-76-02 | タイミング攻撃対策 | constant_time_compareで定時間比較を実行 | HMAC検証時 |
| BR-76-03 | 証明書有効期間 | デフォルトは2014-2034年の20年間有効 | generate_self_signed_certificate |
| BR-76-04 | 公開鍵保存 | .pub拡張子は公開鍵のみ保存 | save()時 |

### ハッシュアルゴリズム

```
MD5, SHA1, SHA256, SHA384, SHA512
```

## データベース操作仕様

本機能はデータベース操作を行わない（インメモリ処理）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_FAIL_V_MSG | モジュール無効 | mbedTLSモジュール無効 | モジュールを有効化してビルド |
| ERR_FAIL_COND_V | パラメータエラー | null鍵/不正なデータ | 有効なパラメータを指定 |
| FAILED | 暗号処理エラー | 鍵不一致/データ破損 | 鍵/データを確認 |

### リトライ仕様

暗号処理にリトライ機構はない。

## トランザクション仕様

該当なし（インメモリ処理）

## パフォーマンス要件

- RSA鍵生成は計算量が大きいため、起動時やバックグラウンドでの実行を推奨
- 大きなデータの暗号化は分割処理を検討
- HMACContext使用でストリーミング処理が可能

## セキュリティ考慮事項

- 秘密鍵のメモリ上での保護に注意
- constant_time_compareによるタイミング攻撃防止
- TLSOptions.client_unsafeは証明書検証をスキップするため本番環境での使用に注意

## 備考

- mbedTLSモジュール無効時はCrypto機能は使用不可
- TLSOptionsでクライアント/サーバー両方の設定に対応
- リソースローダー/セーバーで.key/.pub/.crt形式をサポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

暗号関連クラスの基本構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | crypto.h | `core/crypto/crypto.h` | Crypto, CryptoKey, X509Certificate定義 |
| 1-2 | crypto.cpp | `core/crypto/crypto.cpp` | 基底クラス実装 |

**読解のコツ**: Crypto, CryptoKey, X509Certificateはファクトリパターンで実装。_createポインタがmbedTLS実装を指す。

#### Step 2: mbedTLS実装を理解する

mbedTLSを使用した実装を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | crypto_mbedtls.cpp | `modules/mbedtls/crypto_mbedtls.cpp` | mbedTLS実装 |

**主要処理フロー**:
1. **generate_random_bytes**: mbedtls_ctr_drbg_randomで乱数生成
2. **generate_rsa**: mbedtls_pk_setup + mbedtls_rsa_gen_keyでRSA鍵生成
3. **sign/verify**: mbedtls_pk_sign/mbedtls_pk_verifyで署名操作

### プログラム呼び出し階層図

```
Crypto::generate_rsa()
    │
    └─ CryptoMbedTLS::generate_rsa() [modules/mbedtls]
           │
           ├─ mbedtls_pk_setup()
           │
           └─ mbedtls_rsa_gen_key()
                  │
                  └─ CryptoKeyMbedTLS [結果格納]

Crypto::hmac_digest()
    │
    ├─ HMACContext::create()
    │      └─ HMACContextMbedTLS [mbedTLS実装]
    │
    ├─ HMACContext::start()
    │
    ├─ HMACContext::update()
    │
    └─ HMACContext::finish()
```

### データフロー図

```
[入力]               [処理]                      [出力]

鍵サイズ ───────────▶ generate_rsa ──────────▶ CryptoKey
     │                    │
データ + 鍵 ────────▶ sign/encrypt ──────────▶ 署名/暗号文
     │                    │
署名 + 鍵 + データ ──▶ verify ──────────────▶ bool
     │                    │
暗号文 + 鍵 ─────────▶ decrypt ─────────────▶ 平文
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| crypto.cpp | `core/crypto/crypto.cpp` | ソース | 基底クラス実装 |
| crypto.h | `core/crypto/crypto.h` | ヘッダー | 基底クラス定義 |
| crypto_mbedtls.cpp | `modules/mbedtls/crypto_mbedtls.cpp` | ソース | mbedTLS実装 |
| hashing_context.cpp | `core/crypto/hashing_context.cpp` | ソース | ハッシュコンテキスト |
| stream_peer_tls.cpp | `core/io/stream_peer_tls.cpp` | ソース | TLSストリーム |
