# 画面設計書: デバッガ

## 1. 画面概要

| 項目 | 内容 |
|------|------|
| 画面ID | 15 |
| 画面名 | デバッガ（ScriptEditorDebugger） |
| 概要 | スクリプトのデバッグ実行時に、ブレークポイント、スタックトレース、変数インスペクション、エラー表示、プロファイリングなどの機能を提供する統合デバッグインターフェース |
| 主要機能 | ステップ実行、ブレークポイント管理、スタック変数表示、エラー/警告表示、式評価、プロファイリング、VRAM使用状況表示 |

## 2. UI構成

### 2.1 全体レイアウト

```
+-----------------------------------------------------------------------------------+
| デバッガ (MarginContainer)                                                         |
| +-------------------------------------------------------------------------------+ |
| | TabContainer                                                                  | |
| | +---+--------+------+---------+----------------+----------+-----------+------+| |
| | |Stack|Errors|Evalu-|Profiler |Visual Profiler |Monitors  |Video RAM  |Misc  || |
| | |Trace|      |ator  |         |                |          |           |      || |
| | +---+--------+------+---------+----------------+----------+-----------+------+| |
| | +-----------------------------------------------------------------------------+| |
| | | [タブコンテンツ]                                                            || |
| | +-----------------------------------------------------------------------------+| |
| +-------------------------------------------------------------------------------+ |
+-----------------------------------------------------------------------------------+
```

### 2.2 Stack Traceタブ構造

```
+-----------------------------------------------------------------------------------+
| HBoxContainer (ツールバー)                                                         |
| +---------------------------+---+---+---+---+---+---+---+---+---+---+---+--------+ |
| | RichTextLabel (reason)    |   |Skip|Ign|   |Cp |   |Stp|Nxt|Out|   |Brk|Cont  | |
| |  [停止理由・エラー情報]    |   |BP  |EB |   |   |   |   |   |   |   |   |      | |
| +---------------------------+---+---+---+---+---+---+---+---+---+---+---+--------+ |
+-----------------------------------------------------------------------------------+
| HSplitContainer                                                                   |
| +-------------------+---------------------+------------------------------------+   |
| | VBoxContainer     | VBoxContainer       | Tree (breakpoints_tree)            |   |
| | (stack_vb)        | (inspector_vbox)    |                                    |   |
| | +---------------+ | +---------------+   | +--------------------------------+ |   |
| | |HBox[Thread:]  | | |HBox[Search]   |   | | Breakpoints                    | |   |
| | |[OptionButton] | | |[LineEdit]     |   | | - file1.gd:10                 | |   |
| | +---------------+ | +---------------+   | | - file2.gd:25                 | |   |
| | +---------------+ | +---------------+   | +--------------------------------+ |   |
| | | Tree          | | |EditorDebugger |   |                                    |   |
| | | (stack_dump)  | | | Inspector     |   |                                    |   |
| | | Stack Frames  | | | [変数一覧]    |   |                                    |   |
| | | - 0 - func1   | | |               |   |                                    |   |
| | | - 1 - func2   | | |               |   |                                    |   |
| | +---------------+ | +---------------+   |                                    |   |
| +-------------------+---------------------+------------------------------------+   |
+-----------------------------------------------------------------------------------+
```

### 2.3 Errorsタブ構造

```
+-----------------------------------------------------------------------------------+
| HBoxContainer (error_hbox)                                                        |
| +-------------+---------------+------------------------+-------------------------+ |
| | [Expand All]| [Collapse All]| <space>                | [Clear]                 | |
| +-------------+---------------+------------------------+-------------------------+ |
+-----------------------------------------------------------------------------------+
| Tree (error_tree) - 2カラム                                                       |
| +---------------------------------------+-----------------------------------------+ |
| | エラー種別/ファイル                    | メッセージ                             | |
| +---------------------------------------+-----------------------------------------+ |
| | [Error] script.gd:42                  | Invalid assignment...                  | |
| |   +-- C++ Source                      | scene/main/node.cpp:123                | |
| | [Warning] script.gd:15                | Unused variable...                     | |
| +---------------------------------------+-----------------------------------------+ |
+-----------------------------------------------------------------------------------+
```

### 2.4 Video RAMタブ構造

```
+-----------------------------------------------------------------------------------+
| HBoxContainer (vmem_hb)                                                           |
| +--------------------------------------------+---+--------+---+----------+------+ |
| | Label [List of Video Memory Usage...]      |[i]| <space>|Tot|[LineEdit]|Rf|Ex | |
| +--------------------------------------------+---+--------+---+----------+------+ |
+-----------------------------------------------------------------------------------+
| Tree (vmem_tree) - 4カラム                                                        |
| +-----------------------------+----------+------------+---------+                 |
| | Resource Path               | Type     | Format     | Usage   |                 |
| +-----------------------------+----------+------------+---------+                 |
| | res://textures/icon.png     | Texture  | RGBA8      | 256 KiB |                 |
| | res://models/player.mesh    | Mesh     | Vertices   | 1.2 MiB |                 |
| +-----------------------------+----------+------------+---------+                 |
+-----------------------------------------------------------------------------------+
```

### 2.5 Miscタブ構造

```
+-----------------------------------------------------------------------------------+
| GridContainer (info_left) - 2カラム                                               |
| +---------------------------+---------------------------------------------------+ |
| | Clicked Control:          | [LineEdit - clicked_ctrl]                         | |
| +---------------------------+---------------------------------------------------+ |
| | Clicked Control Type:     | [LineEdit - clicked_ctrl_type]                    | |
| +---------------------------+---------------------------------------------------+ |
| | Live Edit Root:           | [LineEdit] [Set From Tree] [Clear]                | |
| +---------------------------+---------------------------------------------------+ |
+-----------------------------------------------------------------------------------+
| VSeparator                                                                        |
+-----------------------------------------------------------------------------------+
| HBoxContainer (buttons)                                                           |
| +-------------------------+                                                       |
| | [Export measures as CSV]|                                                       |
| +-------------------------+                                                       |
+-----------------------------------------------------------------------------------+
```

## 3. コンポーネント詳細

### 3.1 クラス構造

```
ScriptEditorDebugger : MarginContainer
├── GDCLASS(ScriptEditorDebugger, MarginContainer)
├── friend class EditorDebuggerNode
├── friend class DebugAdapterProtocol
└── friend class DebugAdapterParser
```

### 3.2 主要コンポーネント

| コンポーネント名 | 型 | 説明 |
|------------------|-----|------|
| tabs | TabContainer | メインタブコンテナ |
| reason | RichTextLabel | 停止理由・エラーメッセージ表示 |
| skip_breakpoints | Button | ブレークポイントスキップ切替 |
| ignore_error_breaks | Button | エラーブレーク無視切替 |
| copy | Button | エラー情報コピー |
| step | Button | ステップイン |
| next | Button | ステップオーバー |
| out | Button | ステップアウト |
| dobreak | Button | ブレーク |
| docontinue | Button | 続行 |
| threads | OptionButton | スレッド選択 |
| stack_dump | Tree | スタックフレーム表示 |
| inspector | EditorDebuggerInspector | 変数インスペクタ |
| breakpoints_tree | Tree | ブレークポイント一覧 |
| error_tree | Tree | エラー/警告一覧 |
| expression_evaluator | EditorExpressionEvaluator | 式評価器 |
| profiler | EditorProfiler | スクリプトプロファイラ |
| visual_profiler | EditorVisualProfiler | ビジュアルプロファイラ |
| performance_profiler | EditorPerformanceProfiler | パフォーマンスモニター |
| vmem_tree | Tree | VRAM使用状況ツリー |
| scene_tree | SceneDebuggerTree | リモートシーンツリー |

### 3.3 タブ構成

| タブNo | タブ名 | 対応コンポーネント |
|--------|--------|-------------------|
| 0 | Stack Trace | VBoxContainer (dbg) |
| 1 | Errors | VBoxContainer (errors_tab) |
| 2 | Evaluator | EditorExpressionEvaluator |
| 3 | Profiler | EditorProfiler |
| 4 | Visual Profiler | EditorVisualProfiler |
| 5 | Monitors | EditorPerformanceProfiler |
| 6 | Video RAM | VBoxContainer (vmem_vb) |
| 7 | Misc | VBoxContainer (misc) |

## 4. データ構造

### 4.1 MessageType列挙型

```cpp
enum MessageType {
    MESSAGE_ERROR,
    MESSAGE_WARNING,
    MESSAGE_SUCCESS,
};
```

### 4.2 ProfilerType列挙型

```cpp
enum ProfilerType {
    PROFILER_VISUAL,
    PROFILER_SCRIPTS_SERVERS
};
```

### 4.3 Actions列挙型

```cpp
enum Actions {
    ACTION_COPY_ERROR,
    ACTION_OPEN_SOURCE,
    ACTION_DELETE_BREAKPOINT,
    ACTION_DELETE_BREAKPOINTS_IN_FILE,
    ACTION_DELETE_ALL_BREAKPOINTS,
};
```

### 4.4 ThreadDebugged構造体

```cpp
struct ThreadDebugged {
    String name;           // スレッド名
    String error;          // エラーメッセージ
    bool can_debug;        // デバッグ可能フラグ
    bool has_stackdump;    // スタックダンプ有無
    uint32_t debug_order;  // デバッグ順序
    uint64_t thread_id;    // スレッドID
};
```

### 4.5 FileDialogPurpose列挙型

```cpp
enum FileDialogPurpose {
    SAVE_MONITORS_CSV,
    SAVE_VRAM_CSV,
};
```

## 5. 処理フロー

### 5.1 デバッグセッション開始フロー

```
start(Ref<RemoteDebuggerPeer> p_peer)
    ↓
peer = p_peer (RemoteDebuggerPeer接続)
    ↓
[接続確立]
    ↓
_msg_set_pid受信 → remote_pid設定
    ↓
emit_signal("started")
```

### 5.2 ブレークポイント停止フロー

```
_msg_debug_enter(p_thread_id, p_data)
    ↓
ThreadDebugged構造体作成
    ↓
threads_debugged.insert(p_thread_id, td)
    ↓
最初のスレッドの場合:
    ├── debugging_thread_id = p_thread_id
    ├── _thread_debug_enter(p_thread_id)
    ├── ウィンドウ前面移動
    └── プロファイラ無効化
    ↓
_update_buttons_state()
```

### 5.3 スタックダンプ表示フロー

```
_msg_stack_dump(p_thread_id, p_data)
    ↓
stack.deserialize(p_data)
    ↓
stack_dump->clear()
    ↓
各フレームをTreeItemとして追加
    ↓
フレーム0を選択状態に
    ↓
emit_signal("stack_dump", stack_dump_info)
```

### 5.4 ステップ実行フロー

```
debug_step() / debug_next() / debug_out()
    ↓
ERR_FAIL_COND(!is_breaked())
    ↓
_put_msg("step"/"next"/"out", Array(), debugging_thread_id)
    ↓
_clear_execution()
```

### 5.5 デバッグ続行フロー

```
debug_continue()
    ↓
ERR_FAIL_COND(!is_breaked())
    ↓
フォーカス窃取許可（子プロセスの場合）
    ↓
_clear_execution()
    ↓
_put_msg("continue", Array(), debugging_thread_id)
    ↓
_put_msg("servers:foreground", Array())
    ↓
_mute_audio_on_break(false)
```

## 6. イベント処理

### 6.1 シグナル定義

| シグナル名 | パラメータ | 説明 |
|------------|------------|------|
| started | なし | デバッグセッション開始 |
| breaked | bool, bool, String, bool | ブレーク状態変更 |
| stack_dump | Array | スタックダンプ受信 |
| stack_frame_vars | int | スタックフレーム変数数 |
| stack_frame_var | Array | スタックフレーム変数 |
| output | String, int | 出力メッセージ |
| remote_tree_updated | なし | リモートツリー更新 |
| remote_objects_updated | Object | リモートオブジェクト更新 |
| remote_objects_requested | Array | リモートオブジェクト要求 |
| remote_object_property_updated | ObjectID, String | プロパティ更新 |

### 6.2 メッセージハンドラ

```cpp
static HashMap<String, ParseMessageFunc> parse_message_handlers;

// 登録されるハンドラ例:
"debug_enter" → _msg_debug_enter
"debug_exit" → _msg_debug_exit
"set_pid" → _msg_set_pid
"scene:click_ctrl" → _msg_scene_click_ctrl
"scene:scene_tree" → _msg_scene_scene_tree
"stack_dump" → _msg_stack_dump
"stack_frame_vars" → _msg_stack_frame_vars
"output" → _msg_output
"error" → _msg_error
"servers:memory" → _msg_servers_memory_usage
...
```

## 7. 状態管理

### 7.1 デバッグ状態

| 変数名 | 型 | 説明 |
|--------|-----|------|
| threads_debugged | HashMap<uint64_t, ThreadDebugged> | デバッグ中スレッド |
| debugging_thread_id | uint64_t | 現在デバッグ中スレッドID |
| skip_breakpoints_value | bool | ブレークポイントスキップ状態 |
| ignore_error_breaks_value | bool | エラーブレーク無視状態 |
| error_count | int | エラー数 |
| warning_count | int | 警告数 |

### 7.2 接続状態

| 変数名 | 型 | 説明 |
|--------|-----|------|
| peer | Ref<RemoteDebuggerPeer> | リモートデバッガ接続 |
| remote_pid | OS::ProcessID | リモートプロセスID |
| live_debug | bool | ライブデバッグ有効 |

### 7.3 ボタン状態更新

```cpp
void _update_buttons_state() {
    bool is_active = is_session_active();
    bool has_tree = scene_tree && !scene_tree->nodes.is_empty();
    bool is_debugging = is_debuggable();
    bool is_break = is_breaked();

    // 各ボタンの有効/無効設定
    step->set_disabled(!is_debugging);
    next->set_disabled(!is_debugging);
    out->set_disabled(!is_debugging);
    dobreak->set_disabled(!is_active || is_break);
    docontinue->set_disabled(!is_break);
    // ...
}
```

## 8. 外部連携

### 8.1 RemoteDebuggerPeer通信

```cpp
void _put_msg(const String &p_message, const Array &p_data, uint64_t p_thread_id) {
    if (is_session_active()) {
        Array msg = { p_message, p_thread_id, p_data };
        peer->put_message(msg);
    }
}
```

### 8.2 EditorLogへの出力転送

```cpp
void _msg_output(uint64_t p_thread_id, const Array &p_data) {
    // 出力メッセージをEditorLogに転送
    EditorNode::get_log()->add_message(output_strings[i], msg_type);
    emit_signal("output", output_strings[i], msg_type);
}
```

### 8.3 InspectorDock連携

```cpp
// コンストラクタで接続
InspectorDock::get_inspector_singleton()->connect(
    "object_id_selected",
    callable_mp(this, &ScriptEditorDebugger::_remote_object_selected)
);
```

## 9. コードリーディングガイド

### 9.1 データ構造の把握

1. **ThreadDebugged構造体** (script_editor_debugger.h:168-175)
   - デバッグ中スレッドの状態を管理
   - name, error, can_debug, has_stackdump, debug_order, thread_id

2. **MessageType/Actions列挙型** (script_editor_debugger.h:67-84)
   - メッセージ種別とコンテキストメニューアクションを定義

### 9.2 エントリーポイント

1. **start(Ref<RemoteDebuggerPeer>)** (script_editor_debugger.cpp:未掲載)
   - デバッグセッション開始の起点
   - RemoteDebuggerPeerの接続確立

2. **_parse_message()** (script_editor_debugger.cpp:未掲載)
   - リモートデバッガからのメッセージ処理
   - parse_message_handlersマップを使用

### 9.3 処理層の読み解き

```
メッセージ受信
    ↓
_parse_message() [メッセージルーティング]
    ↓
_msg_* ハンドラ群 [個別メッセージ処理]
    ↓
UI更新 / シグナル発行
```

### 9.4 プログラム呼び出し階層図

```
ScriptEditorDebugger
├── デバッグ制御
│   ├── debug_step() → _put_msg("step")
│   ├── debug_next() → _put_msg("next")
│   ├── debug_out() → _put_msg("out")
│   ├── debug_break() → _put_msg("break")
│   └── debug_continue() → _put_msg("continue")
├── メッセージ処理
│   ├── _msg_debug_enter() → _thread_debug_enter()
│   ├── _msg_debug_exit() → _clear_execution()
│   ├── _msg_stack_dump() → stack_dump更新
│   └── _msg_error() → error_tree更新
├── UI操作
│   ├── _stack_dump_frame_selected() → request_stack_dump()
│   ├── _error_activated() → スクリプトエディタ遷移
│   └── _breakpoint_tree_clicked() → ブレークポイント位置へジャンプ
└── プロファイラ連携
    ├── EditorProfiler
    ├── EditorVisualProfiler
    └── EditorPerformanceProfiler
```

### 9.5 データフロー図

```
[リモートプロセス]
       ↓ RemoteDebuggerPeer
[ScriptEditorDebugger]
       ├── stack_dump → スタックフレーム表示
       ├── inspector → 変数インスペクション
       ├── error_tree → エラー/警告表示
       ├── profiler → プロファイリングデータ
       └── EditorLog → 出力メッセージ転送
```

### 9.6 関連ファイル一覧

| ファイルパス | 役割 |
|-------------|------|
| editor/debugger/script_editor_debugger.h | クラス定義・メンバ宣言 |
| editor/debugger/script_editor_debugger.cpp | 実装・UIコンストラクタ |
| editor/debugger/editor_debugger_node.h/cpp | デバッガノード管理 |
| editor/debugger/editor_debugger_inspector.h/cpp | 変数インスペクタ |
| editor/debugger/editor_expression_evaluator.h/cpp | 式評価器 |
| editor/debugger/editor_profiler.h/cpp | スクリプトプロファイラ |
| editor/debugger/editor_visual_profiler.h/cpp | ビジュアルプロファイラ |
| editor/debugger/editor_performance_profiler.h/cpp | パフォーマンスモニター |
| scene/debugger/scene_debugger.h/cpp | シーンデバッガ（リモート側） |
| core/debugger/remote_debugger.h/cpp | リモートデバッガ通信 |
| core/debugger/debugger_marshalls.h/cpp | デバッグデータシリアライズ |

## 10. 補足情報

### 10.1 キーボードショートカット

| 機能 | ショートカット |
|------|---------------|
| Step Into | debugger/step_into |
| Step Over | debugger/step_over |
| Step Out | debugger/step_out |
| Break | debugger/break |
| Continue | debugger/continue |

### 10.2 エディタ設定

| 設定キー | 説明 |
|---------|------|
| debugger/auto_switch_to_stack_trace | エラー発生時にStack Traceタブに自動切替 |

### 10.3 CSV出力機能

- **モニター出力**: Performance::MONITOR_MAXまでのモニター値 + プロファイラデータ
- **VRAM出力**: vmem_treeの4カラムデータ（Path, Type, Format, Usage）
