# 画面設計書: オーディオバス

## 1. 画面概要

| 項目 | 内容 |
|------|------|
| 画面ID | 16 |
| 画面名 | オーディオバス（EditorAudioBuses） |
| 概要 | オーディオバスのミキサー画面。各バスのボリューム、ソロ/ミュート/バイパス状態、エフェクトの設定、センド先の管理を行う |
| 主要機能 | バス追加/削除/複製、ボリューム調整、VUメーター表示、エフェクト管理、バスレイアウトの保存/読込 |

## 2. UI構成

### 2.1 全体レイアウト

```
+-----------------------------------------------------------------------------------+
| EditorAudioBuses (EditorDock)                                                     |
| +-------------------------------------------------------------------------------+ |
| | VBoxContainer (main_vb)                                                       | |
| | +-------------------------------------------------------------------------+   | |
| | | HBoxContainer (top_hb) - ツールバー                                      |   | |
| | | [Layout:][file_label][Add Bus]|[Load][Save As][Load Default][Create]    |   | |
| | +-------------------------------------------------------------------------+   | |
| | +-------------------------------------------------------------------------+   | |
| | | MarginContainer (bus_mc)                                                |   | |
| | | +---------------------------------------------------------------------+ |   | |
| | | | ScrollContainer (bus_scroll)                                        | |   | |
| | | | +---------------------------------------------------------------+   | |   | |
| | | | | HBoxContainer (bus_hb) - バス一覧                              |   | |   | |
| | | | | +-------+ +-------+ +-------+ +-------+                        |   | |   | |
| | | | | |Master | |Bus 1  | |Bus 2  | | ...   |                        |   | |   | |
| | | | | |       | |       | |       | |       |                        |   | |   | |
| | | | | +-------+ +-------+ +-------+ +-------+                        |   | |   | |
| | | | +---------------------------------------------------------------+   | |   | |
| | | +---------------------------------------------------------------------+ |   | |
| | +-------------------------------------------------------------------------+   | |
| +-------------------------------------------------------------------------------+ |
+-----------------------------------------------------------------------------------+
```

### 2.2 個別バス（EditorAudioBus）構造

```
+------------------------------------------+
| PanelContainer (EditorAudioBus)          |
| +--------------------------------------+ |
| | VBoxContainer                        | |
| | +----------------------------------+ | |
| | | LineEdit (track_name)            | | |
| | +----------------------------------+ | |
| | +----------------------------------+ | |
| | | HBoxContainer                    | | |
| | | [Solo][Mute][Bypass]<spacer>[Opt]| | |
| | +----------------------------------+ | |
| | +----------------------------------+ | |
| | | HSeparator                       | | |
| | +----------------------------------+ | |
| | +----------------------------------+ | |
| | | HBoxContainer (スライダー&VU)    | | |
| | | +--------+----+----+----+------+ | | |
| | | |VSlider |VU_L|VU_R|... |Scale | | | |
| | | |        |    |    |    |      | | | |
| | | +--------+----+----+----+------+ | | |
| | +----------------------------------+ | |
| | +----------------------------------+ | |
| | | Tree (effects)                   | | |
| | | - Effect1 [checkbox]             | | |
| | | - Effect2 [checkbox]             | | |
| | | - Add Effect                     | | |
| | +----------------------------------+ | |
| | +----------------------------------+ | |
| | | OptionButton (send)              | | |
| | +----------------------------------+ | |
| +--------------------------------------+ |
+------------------------------------------+
```

## 3. コンポーネント詳細

### 3.1 クラス構造

```
EditorAudioBuses : EditorDock
├── GDCLASS(EditorAudioBuses, EditorDock)
└── ドック名: "Audio", アイコン: "AudioStreamPlayer"

EditorAudioBus : PanelContainer
├── GDCLASS(EditorAudioBus, PanelContainer)
└── 個別バスパネル

EditorAudioBusDrop : Control
├── GDCLASS(EditorAudioBusDrop, Control)
└── ドラッグ&ドロップ用エンドエリア

EditorAudioMeterNotches : Control
├── GDCLASS(EditorAudioMeterNotches, Control)
└── VUメーター目盛り描画

AudioBusesEditorPlugin : EditorPlugin
└── AudioBusLayoutリソース編集プラグイン
```

### 3.2 EditorAudioBuses主要コンポーネント

| コンポーネント名 | 型 | 説明 |
|------------------|-----|------|
| top_hb | HBoxContainer | 上部ツールバー |
| bus_mc | MarginContainer | バススクロール領域のマージンコンテナ |
| bus_scroll | ScrollContainer | バス一覧スクロールコンテナ |
| bus_hb | HBoxContainer | バスパネル並列コンテナ |
| drop_end | EditorAudioBusDrop | ドラッグ用エンドエリア |
| file | Label | 現在のレイアウトファイル表示 |
| add | Button | バス追加ボタン |
| load | Button | レイアウト読込ボタン |
| save_as | Button | 名前を付けて保存ボタン |
| _default | Button | デフォルトレイアウト読込ボタン |
| _new | Button | 新規レイアウト作成ボタン |
| save_timer | Timer | 自動保存タイマー（0.8秒） |
| file_dialog | EditorFileDialog | ファイル選択ダイアログ |

### 3.3 EditorAudioBus主要コンポーネント

| コンポーネント名 | 型 | 説明 |
|------------------|-----|------|
| track_name | LineEdit | バス名編集 |
| slider | VSlider | ボリュームスライダー |
| solo | Button | ソロボタン（トグル） |
| mute | Button | ミュートボタン（トグル） |
| bypass | Button | エフェクトバイパスボタン（トグル） |
| bus_options | MenuButton | バスオプションメニュー |
| send | OptionButton | センド先選択 |
| effects | Tree | エフェクト一覧ツリー |
| channel[CHANNELS_MAX] | 構造体 | チャンネルVU情報（最大4チャンネル） |
| effect_options | PopupMenu | エフェクト追加メニュー |
| delete_effect_popup | PopupMenu | エフェクト削除メニュー |
| audio_value_preview_box | Panel | ボリュームプレビュー表示 |
| audio_value_preview_label | Label | ボリューム値ラベル |
| preview_timer | Timer | プレビュー非表示タイマー |

### 3.4 チャンネル構造体

```cpp
struct {
    bool prev_active = false;     // 前フレームのアクティブ状態
    float peak_l = 0;             // 左チャンネルピーク値
    float peak_r = 0;             // 右チャンネルピーク値
    TextureProgressBar *vu_l;     // 左VUメーター
    TextureProgressBar *vu_r;     // 右VUメーター
} channel[CHANNELS_MAX];          // CHANNELS_MAX = 4
```

## 4. データ構造

### 4.1 EditorDock設定

```cpp
// コンストラクタ内
set_name(TTRC("Audio"));
set_icon_name("AudioStreamPlayer");
set_dock_shortcut(ED_SHORTCUT_AND_COMMAND(
    "bottom_panels/toggle_audio_bottom_panel",
    TTRC("Toggle Audio Dock"),
    KeyModifierMask::ALT | Key::A
));
set_default_slot(EditorDock::DOCK_SLOT_BOTTOM);
set_available_layouts(
    EditorDock::DOCK_LAYOUT_HORIZONTAL |
    EditorDock::DOCK_LAYOUT_FLOATING
);
```

### 4.2 VUメーター設定

```cpp
// VUメーター範囲
channel[i].vu_l->set_min(-80);  // -80dB
channel[i].vu_l->set_max(24);   // +24dB
channel[i].vu_l->set_step(0.1);
```

### 4.3 AudioNotch構造体（目盛り）

```cpp
struct AudioNotch {
    float relative_position;  // 0.0-1.0の相対位置
    float db_value;           // dB値
    bool render_db_value;     // dB値を描画するか
};
```

## 5. 処理フロー

### 5.1 バス追加フロー

```
_add_bus()
    ↓
EditorUndoRedoManager::create_action("Add Audio Bus")
    ↓
AudioServer::set_bus_count(count + 1)
    ↓
commit_action()
    ↓
bus_layout_changedシグナル → _rebuild_buses()
```

### 5.2 VUメーター更新フロー（NOTIFICATION_PROCESS）

```
毎フレーム処理
    ↓
for each channel (0 to cc):
    ├── AudioServer::is_bus_channel_active()
    ├── AudioServer::get_bus_peak_volume_left_db()
    ├── AudioServer::get_bus_peak_volume_right_db()
    ↓
    if real_peak > channel[i].peak_l:
        channel[i].peak_l = real_peak
    else:
        channel[i].peak_l -= delta_time * 60.0  // 減衰
    ↓
    channel[i].vu_l->set_value(channel[i].peak_l)
    channel[i].vu_r->set_value(channel[i].peak_r)
```

### 5.3 ボリューム変更フロー

```
_volume_changed(p_normalized)
    ↓
_normalized_volume_to_scaled_db(p_normalized)
    ↓
EditorUndoRedoManager::create_action("Change Audio Bus Volume", MERGE_ENDS)
    ↓
AudioServer::set_bus_volume_db(index, p_db)
    ↓
commit_action()
```

### 5.4 エフェクト追加フロー

```
_effect_add(p_which)
    ↓
ClassDB::instantiate(effect_class_name)
    ↓
EditorUndoRedoManager::create_action("Add Audio Bus Effect")
    ↓
AudioServer::add_bus_effect(bus_index, effect, effect_index)
    ↓
commit_action()
```

### 5.5 自動保存フロー

```
NOTIFICATION_PROCESS:
    ↓
AudioServer::is_edited() または effect->is_edited()
    ↓
save_timer->start() (0.8秒)
    ↓
timeout → _server_save()
    ↓
AudioServer::generate_bus_layout()
    ↓
ResourceSaver::save(state, path)
```

## 6. イベント処理

### 6.1 EditorAudioBusシグナル

| シグナル名 | 説明 |
|------------|------|
| duplicate_request | バス複製リクエスト |
| delete_request | バス削除リクエスト |
| vol_reset_request | ボリュームリセットリクエスト |
| drop_end_request | ドラッグエンドエリア表示リクエスト |
| dropped | ドロップ完了 |

### 6.2 EditorAudioBusDropシグナル

| シグナル名 | 説明 |
|------------|------|
| dropped | バスがドロップされた（index, 新位置） |

### 6.3 AudioServer連携

```cpp
// バスレイアウト変更時の再構築
AudioServer::get_singleton()->connect(
    "bus_layout_changed",
    callable_mp(this, &EditorAudioBuses::_rebuild_buses)
);
```

## 7. 状態管理

### 7.1 バス状態

| 変数名 | 型 | 説明 |
|--------|-----|------|
| is_master | bool | マスターバスフラグ |
| updating_bus | bool | 更新中フラグ（再帰防止） |
| cc | int | チャンネル数 |
| hovering_drop | bool | ドラッグホバー状態 |

### 7.2 レイアウト状態

| 変数名 | 型 | 説明 |
|--------|-----|------|
| edited_path | String | 現在編集中のレイアウトパス |
| new_layout | bool | 新規レイアウト作成モード |
| floating | bool | フローティングレイアウトモード |

## 8. ボリューム変換

### 8.1 正規化ボリューム→dB変換

```cpp
float _normalized_volume_to_scaled_db(float normalized) {
    if (normalized > 0.6f) {
        return 22.22f * normalized - 16.2f;
    } else if (normalized < 0.05f) {
        return 830.72 * normalized - 80.0f;
    } else {
        return 45.0f * Math::pow(normalized - 1.0, 3);
    }
}
```

### 8.2 dB→正規化ボリューム変換

```cpp
float _scaled_db_to_normalized_volume(float db) {
    if (db > -2.88) {
        return (db + 16.2f) / 22.22f;
    } else if (db < -38.602f) {
        return (db + 80.00f) / 830.72f;
    } else {
        // 負のdB値に対する特殊処理（NaN回避）
        ...
    }
}
```

## 9. ドラッグ&ドロップ

### 9.1 バス移動

```cpp
Variant get_drag_data(const Point2 &p_point) {
    Dictionary d;
    d["type"] = "move_audio_bus";
    d["index"] = get_index();
    return d;
}
```

### 9.2 エフェクト移動

```cpp
Variant get_drag_data_fw(const Point2 &p_point, Control *p_from) {
    Dictionary d;
    d["type"] = "audio_bus_effect";
    d["bus"] = get_index();
    d["effect"] = item->get_metadata(0);
    return d;
}
```

## 10. コードリーディングガイド

### 10.1 データ構造の把握

1. **EditorAudioBus::channel構造体** (editor_audio_buses.h:63-71)
   - 各チャンネルのVUメーター状態管理
   - peak_l/peak_r, vu_l/vu_r

2. **AudioNotch構造体** (editor_audio_buses.h:222-246)
   - VUメーター目盛り情報
   - relative_position, db_value, render_db_value

### 10.2 エントリーポイント

1. **EditorAudioBuses::register_editor()** (editor_audio_buses.cpp:1123-1127)
   - エディタへの登録処理
   - EditorDockManager経由でドック追加

2. **EditorAudioBuses::open_layout()** (editor_audio_buses.cpp:1467-1489)
   - レイアウトファイルを開く
   - AudioServer::set_bus_layout()呼び出し

### 10.3 処理層の読み解き

```
ユーザー操作
    ↓
EditorAudioBus (個別バスUI)
    ↓
EditorAudioBuses (全体管理)
    ↓
EditorUndoRedoManager (アンドゥ/リドゥ)
    ↓
AudioServer (オーディオ処理)
    ↓
AudioBusLayout (リソース保存)
```

### 10.4 プログラム呼び出し階層図

```
EditorAudioBuses
├── UI操作
│   ├── _add_bus() → AudioServer::set_bus_count()
│   ├── _delete_bus() → AudioServer::remove_bus()
│   └── _duplicate_bus() → AudioServer::add_bus() + 設定コピー
├── レイアウト管理
│   ├── _load_layout() → open_layout()
│   ├── _save_as_layout() → _file_dialog_callback()
│   ├── _new_layout() → AudioServer::set_bus_layout(empty)
│   └── _server_save() → ResourceSaver::save()
└── バス表示更新
    └── _rebuild_buses() → EditorAudioBus生成

EditorAudioBus
├── ボリューム操作
│   ├── _volume_changed() → AudioServer::set_bus_volume_db()
│   └── _show_value() → プレビュー表示
├── トグル操作
│   ├── _solo_toggled() → AudioServer::set_bus_solo()
│   ├── _mute_toggled() → AudioServer::set_bus_mute()
│   └── _bypass_toggled() → AudioServer::set_bus_bypass_effects()
├── エフェクト操作
│   ├── _effect_add() → AudioServer::add_bus_effect()
│   ├── _effect_edited() → AudioServer::set_bus_effect_enabled()
│   └── _delete_effect_pressed() → AudioServer::remove_bus_effect()
└── VUメーター更新
    └── NOTIFICATION_PROCESS → AudioServer::get_bus_peak_volume_*()
```

### 10.5 データフロー図

```
[AudioBusLayout .tres ファイル]
       ↓ ResourceLoader::load()
[AudioServer]
       ↓ get_bus_*() / set_bus_*()
[EditorAudioBuses]
       ├── [EditorAudioBus] × バス数
       │      ├── VSlider (ボリューム)
       │      ├── TextureProgressBar × 2 × 4 (VU)
       │      ├── Tree (エフェクト)
       │      └── OptionButton (センド)
       └── [EditorAudioBusDrop]
              ↓ ドラッグ&ドロップ
[_drop_at_index() → AudioServer::move_bus()]
```

### 10.6 関連ファイル一覧

| ファイルパス | 役割 |
|-------------|------|
| editor/audio/editor_audio_buses.h | クラス定義・メンバ宣言 |
| editor/audio/editor_audio_buses.cpp | 実装・UIコンストラクタ |
| servers/audio/audio_server.h/cpp | オーディオサーバー |
| scene/resources/audio_bus_layout.h/cpp | バスレイアウトリソース |
| editor/docks/editor_dock.h/cpp | EditorDock基底クラス |
| editor/docks/editor_dock_manager.h/cpp | ドック管理 |

## 11. 補足情報

### 11.1 キーボードショートカット

| 機能 | ショートカット |
|------|---------------|
| ドック表示切替 | Alt+A |
| バス複製 | Ctrl/Cmd+D |
| バス削除 | Delete |

### 11.2 バスオプションメニュー

| メニュー項目 | 機能 |
|-------------|------|
| Duplicate Bus | バスを複製 |
| Delete Bus | バスを削除（マスター以外） |
| Reset Volume | ボリュームを0dBにリセット |

### 11.3 VUメーター目盛り

- 6dBから-80dBまで6dB刻み
- 表示対象: -6dB以上、-24dB、-72dB
- 減衰速度: 60dB/秒

### 11.4 エフェクト一覧

ClassDB::get_inheriters_from_class("AudioEffect")から取得される全AudioEffectサブクラスをアルファベット順でリスト化。
