# 画面設計書 19-プロジェクトダイアログ

## 1. 画面概要

### 1.1 基本情報

| 項目 | 内容 |
|------|------|
| 画面名 | プロジェクトダイアログ (Project Dialog) |
| 画面ID | project_dialog |
| 主要クラス | ProjectDialog (ConfirmationDialog継承) |
| ソースファイル | editor/project_manager/project_dialog.cpp, editor/project_manager/project_dialog.h |
| 使用場所 | プロジェクトマネージャー |
| 表示サイズ | 500 x auto (EDSCALE適用) |

### 1.2 機能概要

プロジェクトダイアログは、Godotエディタのプロジェクトマネージャーで新規プロジェクトの作成、既存プロジェクトのインポート、ZIPファイルからのインストール、プロジェクト名の変更、プロジェクトの複製を行うための多目的ダイアログである。モードに応じてUIコンポーネントの表示/非表示が切り替わる。

## 2. 動作モード

### 2.1 Mode列挙型

```cpp
enum Mode {
    MODE_NEW,       // 新規プロジェクト作成
    MODE_IMPORT,    // 既存プロジェクトのインポート
    MODE_INSTALL,   // ZIPからのインストール
    MODE_RENAME,    // プロジェクト名の変更
    MODE_DUPLICATE, // プロジェクトの複製
};
```

### 2.2 モード別タイトルとボタン

| モード | ダイアログタイトル | OKボタンテキスト |
|--------|------------------|-----------------|
| MODE_NEW | "Create New Project" | "Create" |
| MODE_IMPORT | "Import Existing Project" | "Import" |
| MODE_INSTALL | "Install Project: {zip_title}" | "Install" |
| MODE_RENAME | "Rename Project" | "Rename" |
| MODE_DUPLICATE | "Duplicate Project" | "Duplicate" |

### 2.3 モード別表示コンポーネント

| コンポーネント | NEW | IMPORT | INSTALL | RENAME | DUPLICATE |
|---------------|-----|--------|---------|--------|-----------|
| name_container | o | x | o | o | o |
| project_path_container | o | o | o | o | o |
| install_path_container | x | * | x | x | x |
| renderer_container | o | x | x | x | x |
| default_files_container | o | x | x | x | x |
| create_dir | o | o | o | x | o |
| edit_check_box | o | o | o | x | * |

※ * = 条件による表示

## 3. UI構成

### 3.1 全体レイアウト

```
+--------------------------------------------+
| [Dialog Title]                        [X]  |
+--------------------------------------------+
| Project Name:                              |
| [                    project_name        ] |
+--------------------------------------------+
| Project Path:               [Create Folder]|
| [        project_path        ][S][Browse]  |
+--------------------------------------------+
| Project Installation Path: (import mode)   |
| [        install_path        ][S][Browse]  |
+--------------------------------------------+
| [              msg (error/warning)       ] |
+--------------------------------------------+
| Renderer:                                  |
| +------------------+----------------------+|
| | ( ) Forward+     | * Supports desktop.. ||
| | ( ) Mobile       | * Less advanced 3D.. ||
| | (*) Compatibility| * Intended for low.. ||
| | [More information]                     | |
| +------------------+----------------------+|
| [RD not supported warning]                 |
| [The renderer can be changed later...]     |
+--------------------------------------------+
| Version Control Metadata: [Git v]          |
+--------------------------------------------+
| [x] Edit Now                               |
+--------------------------------------------+
|                   [Cancel] [Create/Import] |
+--------------------------------------------+
```

### 3.2 主要コンポーネント

| コンポーネント | タイプ | 説明 |
|---------------|--------|------|
| name_container | VBoxContainer | プロジェクト名入力コンテナ |
| project_name | LineEdit | プロジェクト名入力欄 |
| project_path_container | VBoxContainer | プロジェクトパスコンテナ |
| project_path | LineEdit | プロジェクトパス入力欄 |
| create_dir | CheckButton | フォルダ作成チェック |
| project_browse | Button | パス参照ボタン |
| project_status_rect | TextureRect | パス状態アイコン |
| install_path_container | VBoxContainer | インストールパスコンテナ |
| install_path | LineEdit | インストールパス入力欄 |
| install_browse | Button | インストールパス参照ボタン |
| install_status_rect | TextureRect | インストール状態アイコン |
| msg | Label | メッセージ表示ラベル |
| renderer_container | VBoxContainer | レンダラー選択コンテナ |
| renderer_button_group | ButtonGroup | レンダラーラジオボタングループ |
| renderer_info | Label | レンダラー説明ラベル |
| rd_not_supported | Label | RD非対応警告ラベル |
| default_files_container | HBoxContainer | デフォルトファイルコンテナ |
| vcs_metadata_selection | OptionButton | VCSメタデータ選択 |
| edit_check_box | CheckBox | "Edit Now"チェックボックス |
| fdialog_project | EditorFileDialog | プロジェクトパス選択ダイアログ |
| fdialog_install | EditorFileDialog | インストールパス選択ダイアログ |
| dialog_error | AcceptDialog | エラー表示ダイアログ |
| nonempty_confirmation | ConfirmationDialog | 非空フォルダ確認ダイアログ |

## 4. メッセージタイプ

### 4.1 MessageType列挙型

```cpp
enum MessageType {
    MESSAGE_ERROR,    // エラー（赤色、StatusErrorアイコン）
    MESSAGE_WARNING,  // 警告（オレンジ色、StatusWarningアイコン）
    MESSAGE_SUCCESS,  // 成功（緑色、StatusSuccessアイコン）
};
```

### 4.2 状態フラグ

```cpp
enum InvalidStateFlag {
    INVALID_STATE_FLAG_NONE = 0,
    INVALID_STATE_FLAG_PATH_INPUT = 1 << 0,      // パス入力エラー
    INVALID_STATE_FLAG_RENDERER_SELECT = 1 << 1,  // レンダラー選択エラー
};
```

## 5. レンダラー選択

### 5.1 レンダラーオプション

| レンダラー | rendering_method | 説明 |
|-----------|-----------------|------|
| Forward+ | forward_plus | デスクトップ専用、高度な3Dグラフィックス、RenderingDevice使用 |
| Mobile | mobile | デスクトップ+モバイル、RenderingDevice使用 |
| Compatibility | gl_compatibility | 全プラットフォーム対応、OpenGL 3使用 |

### 5.2 レンダラー説明テキスト

```cpp
// Forward+
"* Supports desktop platforms only."
"* Advanced 3D graphics available."
"* Can scale to large complex scenes."
"* Uses RenderingDevice backend."
"* Slower rendering of simple scenes."

// Mobile
"* Supports desktop + mobile platforms."
"* Less advanced 3D graphics."
"* Less scalable for complex scenes."
"* Uses RenderingDevice backend."
"* Fast rendering of simple scenes."

// Compatibility
"* Supports desktop, mobile + web platforms."
"* Least advanced 3D graphics."
"* Intended for low-end/older devices."
"* Uses OpenGL 3 backend (OpenGL 3.3/ES 3.0/WebGL2)."
"* Fastest rendering of simple scenes."
```

## 6. パス検証

### 6.1 検証ルール

| 検証項目 | エラー条件 | メッセージタイプ |
|---------|-----------|-----------------|
| プロジェクト名 | 空文字 | ERROR |
| 相対パス | 相対パスが指定された | ERROR |
| 無効文字 | ディレクトリ名に無効文字または末尾空白 | ERROR |
| 禁止ディレクトリ | 作業ディレクトリまたは実行ディレクトリ | ERROR |
| ホーム/ドキュメント | ホームまたはドキュメントディレクトリ直下 | ERROR |
| 親ディレクトリ不在 | create_dir時に親ディレクトリが存在しない | ERROR |
| パス不存在 | create_dir無効時にパスが存在しない | ERROR |
| 非空フォルダ | 対象フォルダが空でない | WARNING |
| 自己複製 | 複製先が元プロジェクトのサブディレクトリ | ERROR |

### 6.2 ZIPファイル検証

```cpp
// ZIPファイル検証フロー
if (is_zip_file(path)) {
    // 1. ZIP形式確認
    unzFile pkg = unzOpen2(zip_path);
    if (!pkg) {
        _set_message("Invalid ZIP format", MESSAGE_ERROR);
        return;
    }

    // 2. project.godot存在確認
    while (ret == UNZ_OK) {
        if (name.get_file() == "project.godot") {
            break; // 有効なZIP
        }
        ret = unzGoToNextFile(pkg);
    }
    if (ret == UNZ_END_OF_LIST_OF_FILE) {
        _set_message("No project.godot found", MESSAGE_ERROR);
        return;
    }
}
```

## 7. ディレクトリ命名規則

### 7.1 命名規則オプション

```cpp
int naming_convention = EDITOR_GET("project_manager/directory_naming_convention");
switch (naming_convention) {
    case 0: // No Convention - そのまま
    case 1: // kebab-case - to_kebab_case()
    case 2: // snake_case - to_snake_case()
    case 3: // camelCase - to_camel_case()
    case 4: // PascalCase - to_pascal_case()
    case 5: // Title Case - capitalize()
}
```

### 7.2 自動ディレクトリ名更新

```cpp
void ProjectDialog::_update_target_auto_dir() {
    String new_auto_dir;
    if (mode == MODE_NEW || mode == MODE_INSTALL || mode == MODE_DUPLICATE) {
        new_auto_dir = project_name->get_text();  // プロジェクト名から
    } else if (mode == MODE_IMPORT) {
        new_auto_dir = project_path->get_text().get_file().get_basename();  // ZIPファイル名から
    }
    // 命名規則を適用
    new_auto_dir = OS::get_singleton()->get_safe_dir_name(new_auto_dir);
}
```

## 8. プロジェクト作成処理

### 8.1 新規プロジェクト作成フロー (MODE_NEW)

```
1. フォルダ作成（create_dir有効時）
   └── DirAccess::make_dir(path)

2. project.godot作成
   ├── application/config/name = プロジェクト名
   ├── application/config/icon = "res://icon.svg"
   ├── application/config/features = プロジェクト機能
   ├── rendering/renderer/rendering_method = レンダラー
   └── ProjectSettings::save_custom()

3. icon.svg作成
   └── get_default_project_icon()

4. VCSメタデータ作成
   └── EditorVCSInterface::create_vcs_metadata_files()

5. .editorconfig作成
   └── root = true, charset = utf-8
```

### 8.2 ZIPインストールフロー (MODE_IMPORT/MODE_INSTALL)

```
1. ZIPファイルを開く
   └── unzOpen2(zip_path)

2. project.godotを含むルートディレクトリを検索
   └── zip_root取得

3. 全ファイルを展開
   ├── ディレクトリ: DirAccess::make_dir()
   └── ファイル: FileAccess::store_buffer()

4. project.godot更新（名前変更）
   └── cfg.set_value("application", "config/name", project_name)

5. 失敗ファイルがあればエラー表示
```

### 8.3 複製フロー (MODE_DUPLICATE)

```
1. ディレクトリコピー
   └── DirAccess::copy_dir(original_project_path, path)

2. project.godot更新（名前変更）
   └── ConfigFile::set_value()
```

## 9. VCSメタデータ

### 9.1 VCSMetadata列挙型

```cpp
// EditorVCSInterface::VCSMetadata
enum VCSMetadata {
    NONE,  // なし
    GIT,   // Git（デフォルト）
};
```

### 9.2 Git初期化

```cpp
EditorVCSInterface::create_vcs_metadata_files(
    EditorVCSInterface::VCSMetadata::GIT,
    path
);
// .gitignore, .gitattributes等を作成
```

## 10. シグナル

### 10.1 発行シグナル

| シグナル | 発行タイミング | パラメータ |
|---------|---------------|-----------|
| project_created | MODE_NEW/IMPORT/INSTALL完了時 | path, edit_now |
| project_duplicated | MODE_DUPLICATE完了時 | original_path, new_path, edit_now |
| projects_updated | MODE_RENAME完了時 | なし |

## 11. ファイルダイアログ

### 11.1 fdialog_project設定

```cpp
// MODE_IMPORT時
fdialog_project->set_file_mode(EditorFileDialog::FILE_MODE_OPEN_ANY);
fdialog_project->add_filter("project.godot", "Godot Project");
fdialog_project->add_filter("*.zip", "ZIP File");

// その他のモード
fdialog_project->set_file_mode(EditorFileDialog::FILE_MODE_OPEN_DIR);
```

### 11.2 fdialog_install設定

```cpp
fdialog_install->set_file_mode(EditorFileDialog::FILE_MODE_OPEN_DIR);
```

## 12. イベント処理

### 12.1 シグナル接続

| シグナル送信元 | シグナル名 | ハンドラ |
|---------------|-----------|---------|
| project_name | text_changed | _project_name_changed |
| project_name | text_submitted | ok_pressed |
| project_path | text_changed | _project_path_changed |
| project_path | text_submitted | ok_pressed |
| install_path | text_changed | _install_path_changed |
| install_path | text_submitted | ok_pressed |
| create_dir | toggled | _create_dir_toggled |
| project_browse | pressed | _browse_project_path |
| install_browse | pressed | _browse_install_path |
| fdialog_project | dir_selected | _project_path_selected |
| fdialog_project | file_selected | _project_path_selected |
| fdialog_project | canceled | show_dialog(false, false) |
| fdialog_install | dir_selected | _install_path_selected |
| fdialog_install | file_selected | _install_path_selected |
| renderer_button | pressed | _renderer_selected |
| nonempty_confirmation | ok_pressed | _nonempty_confirmation_ok_pressed |

## 13. テーマ・通知処理

### 13.1 NOTIFICATION_THEME_CHANGED

```cpp
create_dir->set_button_icon(get_editor_theme_icon("FolderCreate"));
project_browse->set_button_icon(get_editor_theme_icon("FolderBrowse"));
install_browse->set_button_icon(get_editor_theme_icon("FolderBrowse"));
```

### 13.2 メッセージカラー

| メッセージタイプ | テーマカラーキー |
|----------------|----------------|
| MESSAGE_ERROR | error_color |
| MESSAGE_WARNING | warning_color |
| MESSAGE_SUCCESS | success_color |

## 14. コードリーディングガイド

### 14.1 プログラム呼び出し階層図

```
ProjectDialog
    │
    ├── show_dialog(reset_name, is_confirmed)
    │   ├── モード別UI設定
    │   │   ├── MODE_NEW: renderer_container表示
    │   │   ├── MODE_IMPORT: name_container非表示
    │   │   ├── MODE_RENAME: project_path編集不可
    │   │   └── MODE_DUPLICATE: edit_check_box条件表示
    │   ├── _update_target_auto_dir()
    │   ├── _create_dir_toggled()
    │   └── _validate_path()
    │
    ├── _validate_path()
    │   ├── プロジェクト名検証
    │   ├── パス検証
    │   │   ├── ZIP検証（MODE_IMPORT時）
    │   │   └── ディレクトリ検証
    │   └── _set_message() / _update_ok_button()
    │
    ├── _renderer_selected()
    │   ├── renderer_info更新
    │   ├── RenderingDevice対応確認
    │   └── rd_not_supported表示制御
    │
    └── ok_pressed()
        ├── 非空フォルダ確認
        │   └── nonempty_confirmation表示
        ├── MODE_NEW処理
        │   ├── フォルダ作成
        │   ├── project.godot作成
        │   ├── icon.svg作成
        │   ├── VCSメタデータ作成
        │   └── .editorconfig作成
        ├── MODE_IMPORT/INSTALL処理
        │   └── ZIP展開
        ├── MODE_DUPLICATE処理
        │   └── ディレクトリコピー
        └── シグナル発行
            ├── project_created
            ├── project_duplicated
            └── projects_updated
```

### 14.2 データフロー図

```
[プロジェクトマネージャー]
         │
         ▼
[ProjectDialog::set_mode()]
         │
         ├── MODE_NEW
         │   └── show_dialog() → UI設定 → レンダラー選択
         │
         ├── MODE_IMPORT
         │   └── ask_for_path_and_show() → ファイル選択
         │
         ├── MODE_INSTALL
         │   └── set_zip_path/title() → show_dialog()
         │
         ├── MODE_RENAME
         │   └── set_project_name/path() → show_dialog()
         │
         └── MODE_DUPLICATE
             └── set_original_project_path() → show_dialog()
                      │
                      ▼
                [_validate_path()]
                      │
                      ├── 検証成功
                      │   └── OKボタン有効化
                      │
                      └── 検証失敗
                          └── OKボタン無効化 + エラーメッセージ
                      │
                      ▼
                [ok_pressed()]
                      │
                      ├── プロジェクト作成/インポート/複製
                      │
                      └── シグナル発行
                          │
                          ▼
                    [プロジェクトマネージャーで処理]
```

### 14.3 関連ファイル一覧

| ファイルパス | 役割 |
|-------------|------|
| editor/project_manager/project_dialog.h | ProjectDialog クラス定義 |
| editor/project_manager/project_dialog.cpp | ProjectDialog 実装（約1244行） |
| editor/project_manager/project_manager.h/cpp | プロジェクトマネージャー本体 |
| editor/gui/editor_file_dialog.h/cpp | ファイル選択ダイアログ |
| editor/version_control/editor_vcs_interface.h/cpp | VCSインターフェース |
| core/config/project_settings.h/cpp | プロジェクト設定 |
| core/io/zip_io.h/cpp | ZIP入出力 |
| scene/gui/dialogs.h | ConfirmationDialog基底クラス |

### 14.4 読解の推奨順序

1. **project_dialog.h** - Mode列挙型とメンバ変数の構造を理解
2. **ProjectDialog()コンストラクタ** - UI構築を追跡（1015-1243行目）
3. **show_dialog()** - モード別初期化処理（849-983行目）
4. **_validate_path()** - パス検証ロジック（94-289行目）
5. **ok_pressed()** - プロジェクト作成/インポート処理（540-814行目）
6. **_renderer_selected()** - レンダラー選択と説明更新（489-533行目）
