# 画面設計書 20-クイック設定ダイアログ

## 1. 画面概要

### 1.1 基本情報

| 項目 | 内容 |
|------|------|
| 画面名 | クイック設定ダイアログ (Quick Settings Dialog) |
| 画面ID | quick_settings_dialog |
| 主要クラス | QuickSettingsDialog (AcceptDialog継承) |
| ソースファイル | editor/project_manager/quick_settings_dialog.cpp, editor/project_manager/quick_settings_dialog.h |
| 使用場所 | プロジェクトマネージャー |
| ダイアログタイトル | "Quick Settings" |
| OKボタンテキスト | "Close" |

### 1.2 機能概要

クイック設定ダイアログは、プロジェクトマネージャーからアクセスできる簡易設定ダイアログである。言語、テーマスタイル、カラープリセット、表示スケール、ネットワークモード、更新確認、ディレクトリ命名規則などの基本的なエディタ設定を素早く変更できる。完全な設定へのアクセスも提供する。

## 2. クラス構造

### 2.1 クラス階層

```
AcceptDialog
  └── QuickSettingsDialog
        ├── VBoxContainer (main_vbox)
        │     ├── PanelContainer (settings_list_panel)
        │     │     └── VBoxContainer (settings_list)
        │     │           ├── HBoxContainer (各設定項目)
        │     │           │     ├── Label (設定名)
        │     │           │     └── OptionButton (設定値)
        │     │           ├── Label (custom_theme_label)
        │     │           └── Button ("Edit All Settings")
        │     └── Label (restart_required_label)
        └── EditorSettingsDialog (editor_settings_dialog) ※遅延生成
```

### 2.2 メンバ変数

```cpp
// 設定値リスト
Vector<String> editor_languages;                    // 言語リスト（Android以外）
Vector<String> editor_styles;                       // スタイルリスト
Vector<String> editor_themes;                       // テーマリスト
Vector<String> editor_scales;                       // スケールリスト
Vector<String> editor_network_modes;                // ネットワークモードリスト
Vector<String> editor_check_for_updates;            // 更新確認オプションリスト
Vector<String> editor_directory_naming_conventions; // 命名規則リスト

// UIコンポーネント
PanelContainer *settings_list_panel;
VBoxContainer *settings_list;
OptionButton *language_option_button;               // Android以外
OptionButton *style_option_button;
OptionButton *theme_option_button;
OptionButton *scale_option_button;
OptionButton *network_mode_option_button;
OptionButton *check_for_update_button;
OptionButton *directory_naming_convention_button;
Label *custom_theme_label;
Label *restart_required_label;
Button *restart_required_button;                    // 遅延生成
EditorSettingsDialog *editor_settings_dialog;       // 遅延生成
```

## 3. UI構成

### 3.1 全体レイアウト

```
+--------------------------------------------------+
| Quick Settings                              [X]  |
+--------------------------------------------------+
| +----------------------------------------------+ |
| | Language          [English           v]      | |
| | Style             [Modern            v]      | |
| | Color Preset      [Default           v]      | |
| | Display Scale     [Auto              v]      | |
| | Network Mode      [Online            v]      | |
| | Check for Updates [Always            v]      | |
| | Directory Naming  [No Convention     v]      | |
| |                                              | |
| |                        [Edit All Settings]   | |
| +----------------------------------------------+ |
|                                                  |
| [Settings changed! The project manager must be   |
|  restarted for changes to take effect.]          |
|                                                  |
|                       [Restart Now] [Close]      |
+--------------------------------------------------+
```

### 3.2 設定項目一覧

| 設定項目 | エディタ設定キー | OptionButton |
|---------|-----------------|--------------|
| Language | interface/editor/editor_language | language_option_button |
| Style | interface/theme/style | style_option_button |
| Color Preset | interface/theme/color_preset | theme_option_button |
| Display Scale | interface/editor/display_scale | scale_option_button |
| Network Mode | network/connection/network_mode | network_mode_option_button |
| Check for Updates | network/connection/check_for_updates | check_for_update_button |
| Directory Naming Convention | project_manager/directory_naming_convention | directory_naming_convention_button |

### 3.3 設定コントロール追加

```cpp
void QuickSettingsDialog::_add_setting_control(const String &p_text, Control *p_control) {
    HBoxContainer *container = memnew(HBoxContainer);
    settings_list->add_child(container);

    Label *label = memnew(Label(p_text));
    label->set_h_size_flags(Control::SIZE_EXPAND_FILL);
    container->add_child(label);

    p_control->set_h_size_flags(Control::SIZE_EXPAND_FILL);
    container->add_child(p_control);
}
```

## 4. 設定値取得

### 4.1 設定値フェッチ処理

```cpp
void QuickSettingsDialog::_fetch_setting_values() {
    // EditorSettingsからプロパティリストを取得
    List<PropertyInfo> editor_settings_properties;
    EditorSettings::get_singleton()->get_property_list(&editor_settings_properties);

    for (const PropertyInfo &pi : editor_settings_properties) {
        if (pi.name == "interface/editor/editor_language") {
            editor_languages = pi.hint_string.split(";", false);
        } else if (pi.name == "interface/theme/style") {
            editor_styles = pi.hint_string.split(",");
        } else if (pi.name == "interface/theme/color_preset") {
            editor_themes = pi.hint_string.split(",");
        } else if (pi.name == "interface/editor/display_scale") {
            editor_scales = pi.hint_string.split(",");
        } else if (pi.name == "network/connection/network_mode") {
            editor_network_modes = pi.hint_string.split(",");
        } else if (pi.name == "network/connection/check_for_updates") {
            editor_check_for_updates = pi.hint_string.split(",");
        } else if (pi.name == "project_manager/directory_naming_convention") {
            editor_directory_naming_conventions = pi.hint_string.split(",");
        }
    }
}
```

### 4.2 現在値更新処理

```cpp
void QuickSettingsDialog::_update_current_values() {
    // 各設定の現在値をOptionButtonに反映
    // - editor_languagesからlanguage_option_button
    // - editor_stylesからstyle_option_button
    // - editor_themesからtheme_option_button
    // - editor_scalesからscale_option_button
    // - editor_network_modesからnetwork_mode_option_button
    // - editor_check_for_updatesからcheck_for_update_button
    // - editor_directory_naming_conventionsからdirectory_naming_convention_button
}
```

## 5. 設定変更処理

### 5.1 設定選択ハンドラ

| 関数 | 設定キー | 再起動必要 |
|------|---------|-----------|
| _language_selected() | interface/editor/editor_language | 暗黙的 |
| _style_selected() | interface/theme/style | なし |
| _theme_selected() | interface/theme/color_preset | なし |
| _scale_selected() | interface/editor/display_scale | あり |
| _network_mode_selected() | network/connection/network_mode | なし |
| _check_for_update_selected() | network/connection/check_for_updates | なし |
| _directory_naming_convention_selected() | project_manager/directory_naming_convention | なし |

### 5.2 設定値保存

```cpp
void QuickSettingsDialog::_set_setting_value(
    const String &p_setting,
    const Variant &p_value,
    bool p_restart_required = false
) {
    // 1. 設定値をEditorSettingsに保存
    EditorSettings::get_singleton()->set(p_setting, p_value);
    EditorSettings::get_singleton()->notify_changes();
    EditorSettings::get_singleton()->save();

    // 2. 再起動必要な場合はUIを更新
    if (p_restart_required) {
        restart_required_label->show();

        if (!restart_required_button) {
            restart_required_button = add_button(TTRC("Restart Now"), ...);
            restart_required_button->connect("pressed", &_request_restart);
        }
    }
}
```

## 6. 言語設定（Android以外）

### 6.1 言語オプション構造

```cpp
// editor_languages の形式: "code/name"
// 例: "en/English", "ja/Japanese"

for (int i = 0; i < editor_languages.size(); i++) {
    const String &lang_code = editor_languages[i].get_slicec('/', 0);  // "en"
    const String &lang_name = editor_languages[i].get_slicec('/', 1);  // "English"
    language_option_button->add_item(lang_name, i);
    language_option_button->set_item_metadata(i, lang_code);
}
```

### 6.2 言語選択処理

```cpp
void QuickSettingsDialog::_language_selected(int p_id) {
    const String selected_language = language_option_button->get_item_metadata(p_id);
    _set_setting_value("interface/editor/editor_language", selected_language);
}
```

## 7. テーマ設定

### 7.1 スタイル選択

```cpp
void QuickSettingsDialog::_style_selected(int p_id) {
    const String selected_style = style_option_button->get_item_text(p_id);
    _set_setting_value("interface/theme/style", selected_style);
}
```

### 7.2 カラープリセット選択

```cpp
void QuickSettingsDialog::_theme_selected(int p_id) {
    const String selected_theme = theme_option_button->get_item_text(p_id);
    _set_setting_value("interface/theme/color_preset", selected_theme);

    // "Custom"選択時はヒントラベルを表示
    custom_theme_label->set_visible(selected_theme == "Custom");
}
```

### 7.3 カスタムテーマラベル

```cpp
custom_theme_label = memnew(Label(TTRC("Custom preset can be further configured in the editor.")));
custom_theme_label->set_horizontal_alignment(HORIZONTAL_ALIGNMENT_RIGHT);
custom_theme_label->set_custom_minimum_size(Size2(220, 0) * EDSCALE);
custom_theme_label->set_autowrap_mode(TextServer::AUTOWRAP_WORD);
custom_theme_label->hide(); // 初期状態は非表示
```

## 8. ネットワーク設定

### 8.1 ネットワークモード選択

```cpp
void QuickSettingsDialog::_network_mode_selected(int p_id) {
    _set_setting_value("network/connection/network_mode", p_id);

    // オフラインモード(p_id == 0)の場合、更新確認を無効化
    check_for_update_button->set_disabled(!p_id);
}
```

### 8.2 更新確認選択

```cpp
void QuickSettingsDialog::_check_for_update_selected(int p_id) {
    _set_setting_value("network/connection/check_for_updates", p_id);
}
```

## 9. 完全な設定へのアクセス

### 9.1 完全設定ダイアログ表示

```cpp
void QuickSettingsDialog::_show_full_settings() {
    if (!editor_settings_dialog) {
        // 遅延初期化
        EditorHelp::generate_doc();

        Ref<EditorInspectorDefaultPlugin> eidp;
        eidp.instantiate();
        EditorInspector::add_inspector_plugin(eidp);

        EditorPropertyNameProcessor *epnp = memnew(EditorPropertyNameProcessor);
        add_child(epnp);

        editor_settings_dialog = memnew(EditorSettingsDialog);
        get_parent()->add_child(editor_settings_dialog);
        editor_settings_dialog->connect("restart_requested", &_request_restart);
    }

    hide();
    editor_settings_dialog->popup_edit_settings();
}
```

### 9.2 Edit All Settingsボタン

```cpp
Button *open_full_settings = memnew(Button);
open_full_settings->set_text(TTRC("Edit All Settings"));
open_full_settings->set_h_size_flags(Control::SIZE_SHRINK_END);
settings_list->add_child(open_full_settings);
open_full_settings->connect("pressed", &_show_full_settings);
```

## 10. 再起動要求

### 10.1 再起動必要ラベル

```cpp
restart_required_label = memnew(Label(TTRC(
    "Settings changed! The project manager must be restarted for changes to take effect."
)));
restart_required_label->set_custom_minimum_size(Size2(560, 0) * EDSCALE);
restart_required_label->set_autowrap_mode(TextServer::AUTOWRAP_WORD);
restart_required_label->set_h_size_flags(Control::SIZE_EXPAND_FILL);
restart_required_label->hide(); // 初期状態は非表示
```

### 10.2 再起動ボタン

```cpp
// 再起動が必要な設定変更時に動的に追加
int ed_swap_cancel_ok = EDITOR_GET("interface/editor/accept_dialog_cancel_ok_buttons");
if (ed_swap_cancel_ok == 0) {
    ed_swap_cancel_ok = DisplayServer::get_singleton()->get_swap_cancel_ok() ? 2 : 1;
}
restart_required_button = add_button(TTRC("Restart Now"), ed_swap_cancel_ok != 2);
restart_required_button->connect("pressed", &_request_restart);
```

### 10.3 再起動シグナル

```cpp
void QuickSettingsDialog::_request_restart() {
    emit_signal("restart_required");
}
```

## 11. テーマ対応

### 11.1 NOTIFICATION_THEME_CHANGED

```cpp
case NOTIFICATION_THEME_CHANGED: {
    // 設定リストパネルにスタイル適用
    settings_list_panel->add_theme_style_override(
        "panel",
        get_theme_stylebox("quick_settings_panel", "ProjectManager")
    );

    // 再起動ラベルに警告色
    restart_required_label->add_theme_color_override(
        "font_color",
        get_theme_color("warning_color", "Editor")
    );

    // カスタムテーマラベルにプレースホルダ色
    custom_theme_label->add_theme_color_override(
        "font_color",
        get_theme_color("font_placeholder_color", "Editor")
    );
} break;
```

### 11.2 NOTIFICATION_VISIBILITY_CHANGED

```cpp
case NOTIFICATION_VISIBILITY_CHANGED: {
    if (is_visible()) {
        _update_current_values();  // 表示時に現在値を更新
    }
} break;
```

## 12. シグナル

### 12.1 発行シグナル

| シグナル | 発行タイミング |
|---------|---------------|
| restart_required | 再起動が必要な設定変更時 |

### 12.2 シグナル接続

| シグナル送信元 | シグナル名 | ハンドラ |
|---------------|-----------|---------|
| language_option_button | item_selected | _language_selected |
| style_option_button | item_selected | _style_selected |
| theme_option_button | item_selected | _theme_selected |
| scale_option_button | item_selected | _scale_selected |
| network_mode_option_button | item_selected | _network_mode_selected |
| check_for_update_button | item_selected | _check_for_update_selected |
| directory_naming_convention_button | item_selected | _directory_naming_convention_selected |
| open_full_settings | pressed | _show_full_settings |
| restart_required_button | pressed | _request_restart |
| editor_settings_dialog | restart_requested | _request_restart |

## 13. プラットフォーム依存

### 13.1 Android制限

```cpp
#ifndef ANDROID_ENABLED
    // Android以外でのみ言語選択が有効
    // 理由: GH-60353 - 設定が保存されない
    // また、ハードウェアマウスなしではドロップダウンのスクロールが困難
    OptionButton *language_option_button;
    void _language_selected(int p_id);
#endif
```

## 14. コードリーディングガイド

### 14.1 プログラム呼び出し階層図

```
QuickSettingsDialog
    │
    ├── コンストラクタ
    │   ├── set_title("Quick Settings")
    │   ├── set_ok_button_text("Close")
    │   ├── _fetch_setting_values()
    │   │   └── EditorSettings::get_property_list()
    │   ├── 各OptionButton作成
    │   │   └── _add_setting_control()
    │   ├── _update_current_values()
    │   └── restart_required_label作成
    │
    ├── NOTIFICATION_VISIBILITY_CHANGED
    │   └── _update_current_values()
    │       └── EDITOR_GET() で各設定値取得
    │
    ├── 設定選択ハンドラ
    │   ├── _language_selected()
    │   ├── _style_selected()
    │   ├── _theme_selected()
    │   ├── _scale_selected() ← p_restart_required = true
    │   ├── _network_mode_selected()
    │   │   └── check_for_update_button->set_disabled()
    │   ├── _check_for_update_selected()
    │   └── _directory_naming_convention_selected()
    │       └── _set_setting_value()
    │           ├── EditorSettings::set()
    │           ├── EditorSettings::notify_changes()
    │           ├── EditorSettings::save()
    │           └── 再起動UI更新（必要時）
    │
    └── _show_full_settings()
        ├── EditorHelp::generate_doc()
        ├── EditorSettingsDialog生成
        └── popup_edit_settings()
```

### 14.2 データフロー図

```
[EditorSettings]
      │
      ▼
[_fetch_setting_values()]
      │
      ├── editor_languages ──→ language_option_button
      ├── editor_styles ────→ style_option_button
      ├── editor_themes ────→ theme_option_button
      ├── editor_scales ────→ scale_option_button
      ├── editor_network_modes ──→ network_mode_option_button
      ├── editor_check_for_updates ──→ check_for_update_button
      └── editor_directory_naming_conventions ──→ directory_naming_convention_button
                      │
                      ▼
            [_update_current_values()]
                      │
                      ├── EDITOR_GET("interface/editor/editor_language")
                      ├── EDITOR_GET("interface/theme/style")
                      ├── EDITOR_GET("interface/theme/color_preset")
                      ├── EDITOR_GET("interface/editor/display_scale")
                      ├── EDITOR_GET("network/connection/network_mode")
                      ├── EDITOR_GET("network/connection/check_for_updates")
                      └── EDITOR_GET("project_manager/directory_naming_convention")
                                  │
                                  ▼
                          [OptionButton選択状態更新]
                                  │
[ユーザー選択] ─────────────────→│
                                  ▼
                     [_xxx_selected(int p_id)]
                                  │
                                  ▼
                     [_set_setting_value()]
                                  │
                          ┌───────┴───────┐
                          ▼               ▼
                   [EditorSettings]  [再起動UI]
                   set() / save()    (display_scale)
```

### 14.3 関連ファイル一覧

| ファイルパス | 役割 |
|-------------|------|
| editor/project_manager/quick_settings_dialog.h | QuickSettingsDialog クラス定義 |
| editor/project_manager/quick_settings_dialog.cpp | QuickSettingsDialog 実装（約453行） |
| editor/project_manager/project_manager.h/cpp | プロジェクトマネージャー本体 |
| editor/settings/editor_settings.h/cpp | エディタ設定管理 |
| editor/settings/editor_settings_dialog.h/cpp | 完全な設定ダイアログ |
| scene/gui/dialogs.h | AcceptDialog基底クラス |
| scene/gui/option_button.h/cpp | OptionButtonコンポーネント |

### 14.4 読解の推奨順序

1. **quick_settings_dialog.h** - メンバ変数と設定項目リストを確認
2. **コンストラクタ** - UI構築とOptionButton初期化（302-452行目）
3. **_fetch_setting_values()** - EditorSettingsからプロパティ情報取得（46-81行目）
4. **_update_current_values()** - 現在値のUI反映（83-181行目）
5. **各_xxx_selected()** - 設定変更ハンドラ（196-231行目）
6. **_set_setting_value()** - 設定保存と再起動UI（233-250行目）
7. **_show_full_settings()** - EditorSettingsDialog遅延生成（252-269行目）
