# 通知設計書 11-StatusIndicator

## 概要

本ドキュメントは、Godotエンジンにおける `StatusIndicator` ノードの設計と実装について記述する。StatusIndicatorは、システムトレイ（通知エリア）にアイコンを表示し、ユーザーに視覚的なフィードバックを提供するための通知機構である。

### 本通知の処理概要

StatusIndicatorは、アプリケーションがシステムトレイ/通知エリアにアイコンを表示し、ユーザーとの対話を可能にする機能を提供する。このノードは、macOSとWindowsのネイティブAPIを活用して、デスクトップ環境に溶け込んだ通知体験を実現する。

**業務上の目的・背景**：StatusIndicatorは、アプリケーションがバックグラウンドで動作している際にも、ユーザーに状態を通知し、簡易的な操作手段を提供するために必要である。特にゲームのサーバーステータス表示、ダウンロードマネージャーの進捗表示、常駐型アプリケーションのステータス表示など、ウィンドウを表示せずにユーザーとコミュニケーションを取る必要がある場面で活用される。

**通知の送信タイミング**：StatusIndicatorは以下のタイミングで動作する。(1) ノードがシーンツリーに追加された時（NOTIFICATION_ENTER_TREE）にシステムトレイアイコンが作成される。(2) ユーザーがトレイアイコンをクリックした時に `pressed` シグナルが発火する。(3) ノードがシーンツリーから削除される時（NOTIFICATION_EXIT_TREE）にシステムトレイアイコンが削除される。

**通知の受信者**：通知の受信者は、StatusIndicatorノードに接続されたスクリプトおよびシグナルハンドラである。`pressed` シグナルを購読しているオブジェクトがクリックイベントを受信し、適切な処理を実行する。

**通知内容の概要**：`pressed` シグナルには、クリックに使用されたマウスボタン（mouse_button: int）とクリック位置（mouse_position: Vector2i）の情報が含まれる。これにより、右クリック/左クリックの区別や、コンテキストメニューの表示位置の決定が可能となる。

**期待されるアクション**：受信者は、クリックイベントに応じてアプリケーションウィンドウの表示/非表示の切り替え、コンテキストメニューの表示、設定画面の起動などの適切なアクションを実行することが期待される。

## 通知種別

システムトレイ通知（OS/システム通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 中 |
| リトライ | 無 |

### 送信先決定ロジック

シグナル接続による。StatusIndicatorノードの `pressed` シグナルに `connect()` メソッドで接続されたすべてのCallableが受信者となる。

## 通知テンプレート

### シグナル定義

| 項目 | 内容 |
|-----|------|
| シグナル名 | pressed |
| パラメータ1 | mouse_button (int) - クリックされたマウスボタン |
| パラメータ2 | mouse_position (Vector2i) - クリック位置 |

### 使用例

```gdscript
func _ready():
    $StatusIndicator.pressed.connect(_on_status_indicator_pressed)

func _on_status_indicator_pressed(mouse_button: int, mouse_position: Vector2i):
    if mouse_button == MOUSE_BUTTON_LEFT:
        # メインウィンドウを表示
        get_window().visible = true
    elif mouse_button == MOUSE_BUTTON_RIGHT:
        # コンテキストメニューを表示
        $PopupMenu.popup(Rect2i(mouse_position, Vector2i.ZERO))
```

### プロパティ

| プロパティ名 | 型 | デフォルト値 | 説明 |
|------------|-----|------------|------|
| tooltip | String | "" | ツールチップテキスト |
| icon | Texture2D | null | 表示アイコン |
| menu | NodePath | "" | PopupMenuへのパス |
| visible | bool | true | 表示状態 |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| mouse_button | クリックされたマウスボタン | DisplayServerコールバック | Yes |
| mouse_position | クリック位置 | DisplayServerコールバック | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| ユーザー操作 | トレイアイコンクリック | visible == true かつ iid が有効 | ユーザーがシステムトレイアイコンをクリックした時 |
| ライフサイクル | NOTIFICATION_ENTER_TREE | visible == true | ノードがシーンツリーに追加された時にアイコン作成 |
| ライフサイクル | NOTIFICATION_EXIT_TREE | iid が有効 | ノードがシーンツリーから削除される時にアイコン削除 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| visible == false | StatusIndicatorが非表示の場合、シグナルは発火しない |
| is_part_of_edited_scene() | エディタで編集中のシーンの一部である場合は動作しない |
| !has_feature(FEATURE_STATUS_INDICATOR) | プラットフォームがStatusIndicator機能をサポートしていない場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[NOTIFICATION_ENTER_TREE] --> B{visible == true?}
    B -->|Yes| C{FEATURE_STATUS_INDICATOR?}
    B -->|No| D[処理なし]
    C -->|Yes| E[DisplayServer.create_status_indicator]
    C -->|No| D
    E --> F[PopupMenuバインド]
    F --> G[アイコン表示完了]

    H[ユーザークリック] --> I[_callback呼び出し]
    I --> J[emit_signal pressed]
    J --> K[接続されたハンドラ実行]

    L[NOTIFICATION_EXIT_TREE] --> M{iid有効?}
    M -->|Yes| N[PopupMenuアンバインド]
    N --> O[DisplayServer.delete_status_indicator]
    O --> P[クリーンアップ完了]
    M -->|No| P
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（StatusIndicatorはデータベースを使用しない）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 機能未サポート | プラットフォームがFEATURE_STATUS_INDICATORをサポートしていない | 何も起こらない（静かに失敗） |
| 無効なIndicatorID | iidがINVALID_INDICATOR_IDの場合 | 操作をスキップ |
| メインスレッド違反 | メインスレッド以外からの呼び出し | ERR_MAIN_THREAD_GUARDでエラーメッセージ表示 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- StatusIndicatorはメインスレッドからのみ操作可能（ERR_MAIN_THREAD_GUARD）
- エディタで編集中のシーンでは動作しない（is_part_of_edited_scene()チェック）
- ネイティブAPIを通じてOSと通信するため、信頼できるコードからのみ使用すること

## 備考

- macOSとWindowsでのみ実装されている
- PopupMenuをmenuプロパティに設定すると、pressedシグナルは発火せず、ネイティブポップアップメニューが表示される
- get_rect()メソッドでアイコンの画面座標での矩形を取得可能

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

StatusIndicatorの内部状態を保持するメンバ変数を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | status_indicator.h | `scene/main/status_indicator.h` | クラス定義、メンバ変数（icon, tooltip, visible, iid, menu）を確認 |

**読解のコツ**: `iid`（IndicatorID）はDisplayServerから返される識別子で、INVALID_INDICATOR_IDとの比較で有効性を判定する。

#### Step 2: エントリーポイントを理解する

通知ハンドリングの起点となる_notificationメソッドを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | status_indicator.cpp | `scene/main/status_indicator.cpp` | _notification関数、NOTIFICATION_ENTER_TREE/EXIT_TREEの処理フローを理解 |

**主要処理フロー**:
1. **35-71行目**: _notification関数全体
2. **44-55行目**: NOTIFICATION_ENTER_TREEでのアイコン作成
3. **56-68行目**: NOTIFICATION_EXIT_TREEでのアイコン削除

#### Step 3: シグナル発火を理解する

ユーザークリック時のコールバック処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | status_indicator.cpp | `scene/main/status_indicator.cpp` | _callback関数、emit_signalの呼び出しを確認 |

**主要処理フロー**:
- **93-95行目**: _callback関数でpressedシグナルを発火

#### Step 4: DisplayServerインターフェースを理解する

プラットフォーム抽象化層の理解。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | display_server.h | `servers/display/display_server.h` | create_status_indicator, status_indicator_set_*メソッドの宣言 |

**主要処理フロー**:
- **868-874行目**: StatusIndicator関連のvirtualメソッド宣言
- **159行目**: FEATURE_STATUS_INDICATOR定数

### プログラム呼び出し階層図

```
StatusIndicator::_notification(NOTIFICATION_ENTER_TREE)
    │
    ├─ DisplayServer::has_feature(FEATURE_STATUS_INDICATOR)
    │
    ├─ DisplayServer::create_status_indicator()
    │      └─ プラットフォーム固有実装（macOS/Windows）
    │
    └─ PopupMenu::bind_global_menu()
           └─ DisplayServer::status_indicator_set_menu()

StatusIndicator::_callback()
    │
    └─ emit_signal("pressed", mouse_button, mouse_position)
           └─ 接続されたシグナルハンドラ

StatusIndicator::_notification(NOTIFICATION_EXIT_TREE)
    │
    ├─ PopupMenu::unbind_global_menu()
    │
    └─ DisplayServer::delete_status_indicator()
```

### データフロー図

```
[入力]                         [処理]                              [出力]

ノード追加
(add_child)        ───▶ _notification(ENTER_TREE) ───▶ システムトレイアイコン表示
                         │
                         └─▶ DisplayServer::create_status_indicator()

ユーザークリック    ───▶ DisplayServer callback      ───▶ pressedシグナル
                         │
                         └─▶ _callback()
                                │
                                └─▶ emit_signal()      ───▶ シグナルハンドラ実行

ノード削除
(remove_child)     ───▶ _notification(EXIT_TREE)  ───▶ システムトレイアイコン削除
                         │
                         └─▶ DisplayServer::delete_status_indicator()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| status_indicator.h | `scene/main/status_indicator.h` | ヘッダ | StatusIndicatorクラス定義 |
| status_indicator.cpp | `scene/main/status_indicator.cpp` | ソース | StatusIndicatorクラス実装 |
| display_server.h | `servers/display/display_server.h` | ヘッダ | プラットフォーム抽象化インターフェース |
| display_server.cpp | `servers/display/display_server.cpp` | ソース | デフォルト実装 |
| display_server_macos.mm | `platform/macos/display_server_macos.mm` | ソース | macOS固有実装 |
| display_server_windows.cpp | `platform/windows/display_server_windows.cpp` | ソース | Windows固有実装 |
| StatusIndicator.xml | `doc/classes/StatusIndicator.xml` | ドキュメント | APIリファレンス |
