# 通知設計書 30-NOTIFICATION_ENTER_CANVAS

## 概要

本ドキュメントは、Godot EngineのCanvasItemクラスにおける`NOTIFICATION_ENTER_CANVAS`通知の設計仕様を記載する。この通知はノードがキャンバス（2D描画領域）に入った際に発火し、2D描画システムへの登録を示す重要なライフサイクル通知である。

### 本通知の処理概要

NOTIFICATION_ENTER_CANVASは、CanvasItem派生クラス（Control、Node2Dなど）がキャンバス描画システムに登録されたタイミングで発行されるシステム通知である。この通知は、RenderingServerへのキャンバスアイテム登録が完了し、描画が可能になったことを示す。

**業務上の目的・背景**：2Dゲーム開発において、ノードがシーンツリーに追加されることと、実際に描画可能になることは異なるタイミングで発生する場合がある。NOTIFICATION_ENTER_CANVASは、CanvasItemがRenderingServerに正しく登録され、描画パイプラインに組み込まれたことを示す。この通知を使用することで、キャンバス固有の初期化（シェーダーパラメータの設定、テクスチャフィルタの適用など）を適切なタイミングで実行できる。

**通知の送信タイミング**：_enter_canvas()が呼ばれた時。具体的にはNOTIFICATION_ENTER_TREEのハンドラ内で、RenderingServerへのcanvas_item_set_parentの登録が完了した後に送信される。

**通知の受信者**：CanvasItemおよびそのすべての派生クラス（Control、Node2D、Polygon2D、Label、Spriteなど）。

**通知内容の概要**：通知IDは整数値32（NOTIFICATION_ENTER_CANVAS = 32）。追加のパラメータは含まない。

**期待されるアクション**：
- キャンバス固有の初期化処理
- RID（リソースID）を使用した描画設定
- CanvasLayerへの参照取得
- World2Dへの参照取得
- 描画グループへの参加

## 通知種別

エンジン内部通知（Engine Internal Notification）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 高（ライフサイクル通知） |
| リトライ | 無し |

### 送信先決定ロジック

1. NOTIFICATION_ENTER_TREEが処理される
2. _enter_canvas()が呼び出される
3. 親のCanvasItem/CanvasLayer/Viewportを特定
4. RenderingServer::canvas_item_set_parent()でキャンバスに登録
5. 必要に応じてcanvas_groupに追加
6. queue_redraw()を呼び出し
7. NOTIFICATION_ENTER_CANVASを送信

## 通知テンプレート

### システム通知の場合

| 項目 | 内容 |
|-----|------|
| 通知ID | 32 |
| 通知名 | NOTIFICATION_ENTER_CANVAS |
| 定義箇所 | scene/main/canvas_item.h |

### 通知ハンドリング例

```cpp
void MyCanvasItem::_notification(int p_notification) {
    switch (p_notification) {
        case NOTIFICATION_ENTER_CANVAS: {
            // キャンバス登録後の初期化
            _setup_canvas_specific_resources();

            // CanvasLayerの参照を取得
            CanvasLayer *layer = get_canvas_layer_node();
            if (layer) {
                // レイヤー固有の設定
            }
        } break;
    }
}
```

```gdscript
func _notification(what):
    if what == NOTIFICATION_ENTER_CANVAS:
        # キャンバス登録後の初期化
        var canvas_layer = get_canvas_layer_node()
        if canvas_layer:
            # レイヤー固有の設定
            pass
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| p_notification | 通知ID（32） | エンジン内部 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 内部処理 | _enter_canvas() | NOTIFICATION_ENTER_TREE処理中 | シーンツリー追加時 |
| 内部処理 | WORLD_2D_CHANGED処理 | Worldが変更された場合 | World2D変更時の再登録 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | _enter_canvas()呼び出し時は常に送信される |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[NOTIFICATION_ENTER_TREE] --> B[_enter_canvas呼び出し]
    B --> C{親がCanvasItem?}
    C -->|Yes| D[parent_itemのcanvas_layerを使用]
    C -->|No| E[CanvasLayer/Viewportを検索]
    D --> F[RenderingServer\ncanvas_item_set_parent\nparent_item\]
    E --> G[RenderingServer\ncanvas_item_set_parent\ncanvas RID\]
    F --> H[canvas_item_set_visibility_layer]
    G --> I[canvas_groupに追加]
    I --> H
    H --> J[queue_redraw]
    J --> K[notification\nNOTIFICATION_ENTER_CANVAS\]
    K --> L[終了]
```

## データベース参照・更新仕様

### 参照データ一覧

| データ | 用途 | 備考 |
|--------|------|------|
| CanvasItem::canvas_layer | 所属するCanvasLayer | CanvasLayer* |
| CanvasItem::canvas_item | キャンバスアイテムRID | RID |
| CanvasItem::canvas_group | キャンバスグループ名 | StringName |
| CanvasItem::visibility_layer | 可視性レイヤー | uint32_t |

### 更新データ一覧

| データ | 操作 | 概要 |
|--------|------|------|
| CanvasItem::canvas_layer | 設定 | 親のcanvas_layerまたは検索結果 |
| CanvasItem::canvas_group | 設定 | トップレベルの場合にグループ名を設定 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| Viewportなし | 親にViewportが見つからない | ERR_FAIL_NULL(viewport) |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（ゲーム実行中いつでも発生可能）

## セキュリティ考慮事項

- メインスレッドからのみキャンバス登録が行われる
- RIDは内部リソースへの参照であり、不正アクセスは防止されている

## 備考

- NOTIFICATION_EXIT_CANVASとペアで使用される対照的な通知
- NOTIFICATION_ENTER_TREEの処理中に発火する
- トップレベルノード（top_level == true）は独自のキャンバスグループを持つ
- CanvasLayerノードはこの通知を受けない（Node派生であるため）
- World2D変更時も再度発火する（NOTIFICATION_WORLD_2D_CHANGED経由）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

CanvasItemのキャンバス関連データ構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | canvas_item.h | `scene/main/canvas_item.h` | NOTIFICATION_ENTER_CANVAS定数の定義（213行目） |
| 1-2 | canvas_item.h | `scene/main/canvas_item.h` | canvas_item RID、canvas_layer、canvas_groupフィールド（82-85行目） |

**読解のコツ**: canvas_itemはRenderingServerに登録されるRID、canvas_layerは所属するCanvasLayerへのポインタ。

#### Step 2: エントリーポイントを理解する

キャンバス登録処理の起点となる関数を特定。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | canvas_item.cpp | `scene/main/canvas_item.cpp` | _enter_canvas関数（234-285行目） |

**主要処理フロー**:
1. **236行目**: get_parent_item()で親CanvasItemを取得
2. **238行目**: Viewportのcanvas_parent_mark_dirtyを呼び出し
3. **242-245行目**: 親がCanvasItemの場合、そのcanvas_layerを使用しRenderingServerに登録
4. **247-280行目**: トップレベルの場合、CanvasLayer/Viewportを検索してキャンバスを取得
5. **282行目**: queue_redraw()を呼び出し
6. **284行目**: notification(NOTIFICATION_ENTER_CANVAS)を送信

#### Step 3: EXIT_CANVASとの対比を理解する

ペアとなる通知を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | canvas_item.cpp | `scene/main/canvas_item.cpp` | _exit_canvas関数（287-295行目） |

### プログラム呼び出し階層図

```
NOTIFICATION_ENTER_TREE処理
    │
    └─ CanvasItem::_notification(NOTIFICATION_ENTER_TREE)
           │
           └─ _enter_canvas()
                  │
                  ├─ get_parent_item() [親CanvasItem取得]
                  │
                  ├─ viewport->canvas_parent_mark_dirty()
                  │
                  ├─ [親がCanvasItemの場合]
                  │      ├─ canvas_layer = parent_item->canvas_layer
                  │      └─ RenderingServer::canvas_item_set_parent(
                  │             canvas_item, parent_item->get_canvas_item())
                  │
                  ├─ [トップレベルの場合]
                  │      ├─ CanvasLayerを検索
                  │      ├─ canvas取得（CanvasLayerまたはWorld2D）
                  │      ├─ RenderingServer::canvas_item_set_parent(
                  │      │      canvas_item, canvas)
                  │      └─ canvas_groupに追加
                  │
                  ├─ RenderingServer::canvas_item_set_visibility_layer()
                  │
                  ├─ queue_redraw()
                  │
                  └─ notification(NOTIFICATION_ENTER_CANVAS)
                         │
                         └─ [派生クラスの_notification]
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ENTER_TREE ─────────────▶ _enter_canvas() ───────────────▶ キャンバス登録
       │                       │
       │                       ▼
       │              parent_item取得
       │                       │
       │                       ▼
       │              canvas_layer設定
       │                       │
       │                       ▼
       │              RenderingServer登録
       │                       │
       └───────────────────────┴───────────────────────▶ NOTIFICATION_ENTER_CANVAS
                                                              │
                                                              ▼
                                                        描画可能状態
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| canvas_item.h | `scene/main/canvas_item.h` | ヘッダー | CanvasItem基底クラス定義、通知定数 |
| canvas_item.cpp | `scene/main/canvas_item.cpp` | ソース | _enter_canvas、_exit_canvasの実装 |
| canvas_layer.h | `scene/main/canvas_layer.h` | ヘッダー | CanvasLayerクラス定義 |
| canvas_layer.cpp | `scene/main/canvas_layer.cpp` | ソース | CanvasLayerの実装 |
| viewport.h | `scene/main/viewport.h` | ヘッダー | Viewportクラス定義 |
| viewport.cpp | `scene/main/viewport.cpp` | ソース | Viewportの実装、canvas_parent_mark_dirty |
| world_2d.h | `scene/resources/world_2d.h` | ヘッダー | World2Dクラス定義 |
| rendering_server.h | `servers/rendering/rendering_server.h` | ヘッダー | canvas_item_*API定義 |
