# 通知設計書 5-push_warning

## 概要

本ドキュメントは、Godotエンジンの@GlobalScope関数であるpush_warning()の設計仕様を記述する。

### 本通知の処理概要

push_warning()は、GDScriptや他のスクリプト言語からデバッガおよびターミナル（標準エラー出力）に警告メッセージをプッシュするためのグローバル関数である。スタックトレース情報を含む警告出力を生成し、開発者が潜在的な問題箇所を特定できるようにする。

**業務上の目的・背景**：ゲーム開発中にエラーほど深刻ではないが注意が必要な状況（非推奨APIの使用、パフォーマンス上の懸念、想定外だが継続可能な状態など）を報告する必要がある。push_warning()はプログラムの実行を中断せずに警告を記録し、開発者が後から問題を確認・修正できるようにする。push_error()と同様の仕組みだが、重要度が低いことを示す。

**通知の送信タイミング**：スクリプトコード内でpush_warning()関数が明示的に呼び出されたとき。引数として1つ以上のVariant値を受け取り、それらを連結して警告メッセージとして出力する。

**通知の受信者**：デバッガ（エディタデバッガまたはリモートデバッガ）、ターミナル/コンソール（標準エラー出力）、ログファイル（設定されている場合）。エディタ実行中はデバッガパネルに、スタンドアロン実行時はターミナルにメッセージが表示される。

**通知内容の概要**：連結された警告メッセージ文字列、呼び出し元の関数名、ファイルパス、行番号、スクリプトバックトレース情報が含まれる。メッセージのプレフィックスは「WARNING:」となる。

**期待されるアクション**：開発者は警告メッセージを確認し、必要に応じて原因となるコードを改善する。エラーほど緊急ではないため、後回しにすることも可能だが、品質向上のために対処が推奨される。

## 通知種別

デバッグ/ログ通知（標準エラー出力、デバッガ通知、ログファイル出力）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 中（警告レベル） |
| リトライ | 無 |

### 送信先決定ロジック

- OS::print_error()経由でLoggerに警告メッセージが送信される
- 登録されているすべてのErrorHandlerList（エラーハンドラチェーン）に通知される
- CompositeLoggerを通じて複数の出力先（stdout、ログファイル）に同時出力される

## 通知テンプレート

### コンソール/ログ出力の場合

| 項目 | 内容 |
|-----|------|
| プレフィックス | WARNING: |
| 位置情報形式 | at: {function} ({file}:{line}) |
| バックトレース | スクリプト呼び出しスタック |

### 本文テンプレート

```
WARNING: {message}
     at: {function} ({file}:{line})
{script_backtrace}
```

※WARNING:はエラーより長いため、インデントは5スペース

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| message | 警告メッセージ（引数を連結） | push_warning引数 | Yes |
| function | 呼び出し元関数名 | __FUNCTION__マクロ | Yes |
| file | 呼び出し元ファイルパス | __FILE__マクロ | Yes |
| line | 呼び出し元行番号 | __LINE__マクロ | Yes |
| script_backtrace | スクリプトバックトレース | ScriptServer::capture_script_backtraces | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | push_warning(args...) | 常時 | GDScriptからの直接呼び出し |
| C++マクロ | WARN_PRINT(message) | 常時 | C++コードからの警告出力 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| CoreGlobals::print_error_enabled == false | エラー/警告出力がグローバルで無効化されている場合 |
| is_printing_error == true | 再帰呼び出し防止 |
| _stderr_enabled == false | OS標準エラー出力が無効 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[push_warning呼び出し] --> B[引数をStringに連結]
    B --> C[WARN_PRINTマクロ呼び出し]
    C --> D[_err_print_error関数]
    D --> E[ErrorHandlerType=ERR_HANDLER_WARNING設定]
    E --> F{is_printing_error?}
    F -->|Yes| G[フォールバック出力]
    F -->|No| H[is_printing_error = true]
    H --> I{OS初期化済み?}
    I -->|Yes| J[OS::print_error呼び出し]
    I -->|No| K[stderr直接出力]
    J --> L[Logger::log_error with ERR_WARNING]
    L --> M[スクリプトバックトレース取得]
    M --> N[フォーマット出力]
    N --> O[ErrorHandlerList通知]
    O --> P[is_printing_error = false]
    P --> Q[終了]
    G --> Q
    K --> Q
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（データベースは使用しない）

### 更新テーブル一覧

該当なし（データベースは使用しない）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 引数不足 | p_arg_count < 1 | CallError::CALL_ERROR_TOO_FEW_ARGUMENTSを設定 |
| 再帰呼び出し | is_printing_errorがtrue | フォールバック出力に切り替え |
| OS未初期化 | OS::get_singleton()がnull | stderr直接出力 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| レート制限 | なし |
| バッファリング | なし（即時出力） |

### 配信時間帯

制限なし（常時）

## セキュリティ考慮事項

- 警告メッセージはそのまま出力されるため、機密情報を含めないよう注意が必要
- ファイルパスがフルパスで出力される場合があり、システム構成の露出に注意
- 本番環境ではログ出力を制限することを推奨

## 備考

- push_warning()は可変長引数を受け取り、すべてをString変換して連結する
- WARN_PRINTマクロは内部的に同じ処理を実行する
- push_error()との違いはERR_HANDLER_WARNINGタイプが使用されること
- 警告出力後も実行は継続される
- エディタ設定で内部警告のトースト通知の有無を制御可能

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

push_warningの引数処理と内部データ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | variant_utility.h | `core/variant/variant_utility.h` | push_warning関数シグネチャ（141行目） |
| 1-2 | error_macros.h | `core/error/error_macros.h` | WARN_PRINTマクロ定義（713-714行目） |
| 1-3 | error_macros.h | `core/error/error_macros.h` | ERR_HANDLER_WARNING列挙値（44行目） |

**読解のコツ**: push_warningはpush_errorとほぼ同じ構造だが、WARN_PRINTマクロを使用しERR_HANDLER_WARNINGタイプで出力される点が異なる。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | variant_utility.cpp | `core/variant/variant_utility.cpp` | push_warning実装（1028-1036行目） |

**主要処理フロー**:
1. **1028行目**: 関数定義開始
2. **1029-1032行目**: 引数数チェック（最低1つ必要）
3. **1034行目**: WARN_PRINTマクロ呼び出し（join_stringで引数連結）
4. **1035行目**: CallError::CALL_OK設定

#### Step 3: 警告出力処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | error_macros.h | `core/error/error_macros.h` | WARN_PRINTマクロでERR_HANDLER_WARNING指定（714行目） |
| 3-2 | error_macros.cpp | `core/error/error_macros.cpp` | _err_print_error実装（106-135行目） |

**主要処理フロー**:
- **713-714行目**: WARN_PRINTマクロでp_type=ERR_HANDLER_WARNINGを指定
- **117行目**: Logger::ErrorTypeにキャストして渡す

#### Step 4: Logger処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | logger.h | `core/io/logger.h` | ErrorType::ERR_WARNING定義（52行目） |
| 4-2 | logger.h | `core/io/logger.h` | error_type_string関数（57-69行目）で"WARNING"文字列 |
| 4-3 | logger.h | `core/io/logger.h` | error_type_indent関数（71-83行目）で5スペースインデント |

**主要処理フロー**:
- **52行目**: ERR_WARNING列挙値
- **62行目**: ERR_WARNINGの場合"WARNING"を返す
- **76行目**: ERR_WARNINGの場合5スペースインデント（"     "）

### プログラム呼び出し階層図

```
push_warning(args...)  [GDScript]
    │
    └─ VariantUtilityFunctions::push_warning(p_args, p_arg_count)
           │
           ├─ join_string(p_args, p_arg_count)  →  警告メッセージ連結
           │
           └─ WARN_PRINT(joined_message)
                  │
                  └─ _err_print_error(..., false, ERR_HANDLER_WARNING)
                         │
                         ├─ 再帰呼び出しチェック
                         │
                         └─ OS::get_singleton()->print_error(..., ERR_WARNING)
                                │
                                ├─ Logger::log_error(..., ERR_WARNING)
                                │      │
                                │      ├─ logf_error("WARNING: %s")
                                │      │
                                │      └─ スクリプトバックトレース出力
                                │
                                └─ ErrorHandlerList通知
                                       │
                                       └─ 各ハンドラのerrfunc(..., WARNING)
```

### データフロー図

```
[入力]                    [処理]                           [出力]

p_args[0..n]     ───▶  join_string()  ───▶  message(String)
(Variant**)                                      │
                                                 │
                                           WARN_PRINT()
                                                 │
                                           _err_print_error()
                                                 │
                                     p_type=ERR_HANDLER_WARNING
                                                 │
                              ┌──────────────────┼──────────────────┐
                              ▼                  ▼                  ▼
                      OS::print_error()  ErrorHandlerList    フォールバック
                              │                  │                  │
                       Logger::log_error()  各ハンドラ          stderr
                       (ERR_WARNING)             │
                    ┌─────────┼─────────┐        │
                    ▼         ▼         ▼        ▼
               stdout    logfile    stderr   EditorToaster
            "WARNING:"                       (SEVERITY_WARNING)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| variant_utility.h | `core/variant/variant_utility.h` | ソース | push_warning関数宣言 |
| variant_utility.cpp | `core/variant/variant_utility.cpp` | ソース | push_warning実装 |
| error_macros.h | `core/error/error_macros.h` | ソース | WARN_PRINTマクロ、ERR_HANDLER_WARNING定義 |
| error_macros.cpp | `core/error/error_macros.cpp` | ソース | _err_print_error実装 |
| logger.h | `core/io/logger.h` | ソース | Logger基底クラス、ERR_WARNING定義 |
| logger.cpp | `core/io/logger.cpp` | ソース | Logger::log_error実装 |
