# 通知設計書 72-NOTIFICATION_ACCESSIBILITY_INVALIDATE

## 概要

本ドキュメントは、Godot Engineにおける`NOTIFICATION_ACCESSIBILITY_INVALIDATE`通知の設計仕様を記載する。この通知はアクセシビリティ要素が無効化される際にノードに送信され、支援技術との連携に必要なリソースをクリーンアップするための通知である。

### 本通知の処理概要

この通知は、アクセシビリティ要素の無効化が必要な場合にノードへ送信される。受信したノードは、保持しているアクセシビリティ要素（RID）を解放し、DisplayServerに登録されたリソースを削除する。この通知を受けた後は、既存のアクセシビリティ要素への参照は無効となるため、破棄する必要がある。

**業務上の目的・背景**：アプリケーションのライフサイクルにおいて、ノードがシーンツリーから削除される際や、アクセシビリティシステムの再初期化が必要な際に、メモリリークを防ぎ、支援技術に正確な情報を提供し続けるためにリソースのクリーンアップが必要となる。

**通知の送信タイミング**：以下のイベントで通知が送信される：
- ノードがシーンツリーから退出する時（NOTIFICATION_EXIT_TREE処理内）
- アクセシビリティシステムの再初期化時
- ノードの削除時

**通知の受信者**：シーンツリー内のすべてのNodeおよびその派生クラスが受信対象となる。ただし、エディタシーンの一部として編集中のノードは除外される。

**通知内容の概要**：通知自体にはデータは含まれないが、受信時に以下の処理が行われる：
- アクセシビリティ要素（RID）の解放
- DisplayServerからの要素削除
- 内部参照の初期化

**期待されるアクション**：受信したノードは`_notification()`メソッド内で以下を実行する：
1. 保持しているアクセシビリティ要素への参照を破棄
2. 派生クラスで追加のクリーンアップが必要な場合は実行

## 通知種別

アプリ内通知（エンジン内部通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 通知定数 | `NOTIFICATION_ACCESSIBILITY_INVALIDATE` |
| 通知値 | 3001 |
| 送信方式 | 同期 |
| 優先度 | 高（リソース解放のため） |
| リトライ | なし |

### 送信先決定ロジック

通知は以下の条件を満たすノードに送信される：
- アクセシビリティ要素が有効である（`accessibility_element.is_valid() == true`）
- シーンツリーからの退出処理中、または明示的な無効化要求時

## 通知テンプレート

### 通知パラメータ

| 項目 | 内容 |
|-----|------|
| 通知ID | 3001 |
| パラメータ | なし |
| 伝播方向 | なし（個別ノードへの直接送信） |

### 処理テンプレート（GDScript）

```gdscript
func _notification(what):
    if what == NOTIFICATION_ACCESSIBILITY_INVALIDATE:
        # アクセシビリティ要素への参照を保持している場合は破棄
        # 基底クラスの処理でRIDは自動的に解放される
        pass
```

### 処理テンプレート（C++）

```cpp
void MyNode::_notification(int p_notification) {
    switch (p_notification) {
        case NOTIFICATION_ACCESSIBILITY_INVALIDATE: {
            // Nodeクラスの実装では以下が自動的に行われる：
            if (data.accessibility_element.is_valid()) {
                DisplayServer::get_singleton()->accessibility_free_element(data.accessibility_element);
                data.accessibility_element = RID();
            }
            // 派生クラスでの追加クリーンアップ処理
        } break;
    }
}
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| accessibility_element | アクセシビリティ要素のRID | `data.accessibility_element` | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| ライフサイクル | NOTIFICATION_EXIT_TREE | アクセシビリティ要素が有効 | ツリー退出時 |
| システム | アクセシビリティ再初期化 | アクセシビリティ要素が有効 | システム再初期化時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| accessibility_element.is_null() | アクセシビリティ要素が存在しない場合は処理スキップ |
| is_part_of_edited_scene() == true | エディタで編集中のシーンでは通常のクリーンアップ |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[NOTIFICATION_EXIT_TREE受信] --> B{accessibility_element.is_valid?}
    B -->|No| C[処理終了]
    B -->|Yes| D{is_part_of_edited_scene?}
    D -->|Yes| E[通常のクリーンアップ]
    D -->|No| F[accessibility_free_element呼び出し]
    F --> G[accessibility_element = RID]
    G --> H[_accessibility_notify_change呼び出し]
    H --> I[終了]
    E --> I
    C --> I
```

## データベース参照・更新仕様

該当なし（本通知はエンジン内部の通知システムで処理され、データベースは使用しない）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 二重解放 | すでに解放されたRIDへのアクセス | is_valid()チェックで防止 |
| DisplayServer未初期化 | シングルトン未取得 | null チェックで防止 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 制限 | なし（ライフサイクルイベントに連動） |

### 配信時間帯

制限なし（ノードのライフサイクルに依存）

## セキュリティ考慮事項

- リソース解放処理であるため、二重解放によるクラッシュを防止するためのチェックが実装されている
- DisplayServerとの通信はエンジン内部で行われる

## 備考

- この通知はGodot 4.xで追加されたアクセシビリティシステムの一部
- NOTIFICATION_ACCESSIBILITY_UPDATEと対になる通知で、更新と無効化のペアで使用される
- 通知を受け取った後は、既存のアクセシビリティ要素への参照を全て破棄する必要がある

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

通知定数とアクセシビリティ要素の管理構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | node.h | `scene/main/node.h` | NOTIFICATION_ACCESSIBILITY_INVALIDATE定数（値3001）とaccessibility_element RIDメンバ |

**読解のコツ**: `data.accessibility_element`はミュータブルなRIDで、無効化時に`RID()`（空のRID）にリセットされる。

#### Step 2: エントリーポイントを理解する

通知の発火元を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | node.cpp | `scene/main/node.cpp` | _notification()内のNOTIFICATION_EXIT_TREE処理（行186-201） |

**主要処理フロー**:
1. **行190-191**: アクセシビリティ要素の有効性チェック
2. **行192**: `accessibility_free_element()`呼び出し
3. **行193**: `accessibility_element = RID()`で初期化

#### Step 3: 通知処理を理解する

通知受信時の処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | node.cpp | `scene/main/node.cpp` | _notification()内のNOTIFICATION_ACCESSIBILITY_INVALIDATEケース（行62-67） |

**主要処理フロー**:
- **行62**: `case NOTIFICATION_ACCESSIBILITY_INVALIDATE:`開始
- **行63**: `accessibility_element.is_valid()`チェック
- **行64**: `accessibility_free_element()`でリソース解放
- **行65**: `accessibility_element = RID()`で初期化

### プログラム呼び出し階層図

```
Node::_notification(NOTIFICATION_EXIT_TREE)
    │
    ├─ DisplayServer::accessibility_free_element()
    │      └─ RID解放
    │
    └─ SceneTree::_accessibility_notify_change(node, true)
           └─ 親ノードへの通知

Node::_notification(NOTIFICATION_ACCESSIBILITY_INVALIDATE)
    │
    ├─ accessibility_element.is_valid()チェック
    │
    ├─ DisplayServer::accessibility_free_element()
    │
    └─ accessibility_element = RID()
```

### データフロー図

```
[入力]               [処理]                    [出力]

ノード退出         ───▶ NOTIFICATION_EXIT    ───▶ アクセシビリティ
 - remove_child()       _TREE処理                 要素の解放
 - queue_free()              │                          │
                             ▼                          ▼
                    accessibility_free         DisplayServer
                    _element()                 からリソース削除
                             │                          │
                             ▼                          ▼
                    accessibility_element      支援技術への
                    = RID()                    情報削除通知
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| node.h | `scene/main/node.h` | ヘッダ | 通知定数とアクセシビリティ関連メンバ宣言 |
| node.cpp | `scene/main/node.cpp` | ソース | 通知処理とリソース解放実装 |
| display_server.h | `servers/display_server.h` | ヘッダ | accessibility_free_element() API宣言 |
| scene_tree.cpp | `scene/main/scene_tree.cpp` | ソース | _accessibility_notify_change()実装 |
| Node.xml | `doc/classes/Node.xml` | ドキュメント | 公式ドキュメント定義 |
