# 通知設計書 75-tree_entered

## 概要

本ドキュメントは、Godot Engineにおける`tree_entered`シグナルの設計仕様を記載する。このシグナルはNodeクラスで定義され、ノードがSceneTreeに入った際に発火する。ノードのライフサイクル管理において重要な通知の一つである。

### 本通知の処理概要

このシグナルは、ノードがSceneTreeに追加された直後に発火する。`NOTIFICATION_ENTER_TREE`通知の後、`_enter_tree()`仮想メソッドの呼び出し後に発火するため、ノードはすでにツリー内に存在し、親ノードや兄弟ノードへのアクセスが可能な状態となっている。

**業務上の目的・背景**：ゲーム開発において、ノードがシーンに追加された際に初期化処理や他のノードとの接続が必要になることが多い。このシグナルを利用することで、ノードの追加を監視し、必要な初期化やセットアップを行うことができる。通知（NOTIFICATION_ENTER_TREE）と異なり、シグナルは外部オブジェクトからの監視に適している。

**通知の送信タイミング**：以下のイベントでシグナルが発火する：
- `add_child()`でノードが追加された時
- シーンがインスタンス化された時
- `reparent()`でノードが再配置された時

**通知の受信者**：シグナルに接続したすべてのオブジェクト。主な利用者：
- 親ノード（子の追加を監視）
- シーン管理システム
- デバッグ/ロギングシステム

**通知内容の概要**：シグナル自体にはパラメータがない。

**期待されるアクション**：シグナル受信者は以下のような処理を行う：
1. ノードの初期化処理
2. 他のノードとの接続確立
3. リソースのロード開始

## 通知種別

シグナル（Signal）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| シグナル名 | `tree_entered` |
| 定義クラス | Node |
| 送信方式 | 同期（emit_signal） |
| 優先度 | 高（ライフサイクルイベント） |
| リトライ | なし |

### 送信先決定ロジック

シグナルは`_propagate_enter_tree()`メソッド内で以下の順序で発火される：
1. `notification(NOTIFICATION_ENTER_TREE)`送信
2. `GDVIRTUAL_CALL(_enter_tree)`呼び出し
3. `emit_signal(SceneStringName(tree_entered))`発火

## 通知テンプレート

### シグナル定義

```cpp
// node.cpp
ADD_SIGNAL(MethodInfo("tree_entered"));
```

### 処理テンプレート（GDScript）

```gdscript
func _ready():
    # 自身のtree_enteredシグナルに接続（通常は_enter_treeを使用）
    tree_entered.connect(_on_tree_entered)

func _on_tree_entered():
    print("ノードがツリーに入りました")

# 子ノードの追加を監視する場合はchild_entered_treeを使用
```

### 処理テンプレート（C++）

```cpp
// シグナル接続
node->connect("tree_entered", callable_mp(this, &MyClass::_on_node_entered_tree));

void MyClass::_on_node_entered_tree() {
    // ノードがツリーに入った時の処理
    _setup_node_connections();
}
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| なし | シグナルにはパラメータがない | - | - |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 構造変更 | add_child() | ノードがツリーに入る | 子ノード追加 |
| 構造変更 | reparent() | 新しい親がツリー内 | 再配置 |
| シーン | PackedScene.instantiate() | インスタンス化時 | シーンロード |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| is_blocking_signals() | シグナルブロック中 |
| ノードがすでにツリー内 | 再追加は不可 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[add_child呼び出し] --> B[_propagate_enter_tree開始]
    B --> C[ツリー/Viewport設定]
    C --> D[グループへの追加]
    D --> E[notification ENTER_TREE]
    E --> F[GDVIRTUAL_CALL _enter_tree]
    F --> G[emit_signal tree_entered]
    G --> H[SceneTree.node_added通知]
    H --> I{親ノードあり?}
    I -->|Yes| J[親にchild_entered_tree発火]
    I -->|No| K[子ノードの_propagate_enter_tree]
    J --> K
    K --> L[終了]
```

## データベース参照・更新仕様

該当なし（本シグナルはエンジン内部のノードシステムで処理され、データベースは使用しない）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 特になし | - | - |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 制限 | なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- ライフサイクルイベントであるため、特別なセキュリティ考慮は不要
- シグナルハンドラ内での長時間処理はゲームのフレームレートに影響する可能性がある

## 備考

- `NOTIFICATION_ENTER_TREE`通知の後に発火する
- 自身のシグナルを使用するよりも、通常は`_enter_tree()`仮想メソッドのオーバーライドが推奨される
- 外部からの監視には`child_entered_tree`シグナルの方が適している場合がある

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Nodeクラスのツリー管理構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | node.h | `scene/main/node.h` | treeメンバ、depthメンバ、viewportメンバ |
| 1-2 | scene_string_names.h | `scene/scene_string_names.h` | tree_entered文字列名定義 |

**読解のコツ**: `data.tree`がnullでない場合、ノードはツリー内に存在する。`is_inside_tree()`はこのチェックを行う。

#### Step 2: エントリーポイントを理解する

シグナルの発火元を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | node.cpp | `scene/main/node.cpp` | _propagate_enter_tree()メソッド（行335-383） |

**主要処理フロー**:
1. **行338-343**: ツリーとdepthの設定
2. **行345-348**: Viewportの設定
3. **行350-352**: グループへの追加
4. **行354**: `notification(NOTIFICATION_ENTER_TREE)`
5. **行356**: `GDVIRTUAL_CALL(_enter_tree)`
6. **行358**: `emit_signal(SceneStringName(tree_entered))`

#### Step 3: シグナル定義を理解する

シグナルの登録処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | node.cpp | `scene/main/node.cpp` | _bind_methods()内のADD_SIGNAL（行4025） |

### プログラム呼び出し階層図

```
Node::add_child(Node* p_child)
    │
    └─ Node::_propagate_enter_tree()
           │
           ├─ data.tree = parent->data.tree
           ├─ data.depth = parent->depth + 1
           ├─ グループへの追加
           │
           ├─ notification(NOTIFICATION_ENTER_TREE)
           │
           ├─ GDVIRTUAL_CALL(_enter_tree)
           │
           ├─ emit_signal("tree_entered")  ← シグナル発火
           │
           ├─ SceneTree::node_added(this)
           │
           ├─ 親ノードへchild_entered_tree発火
           │
           └─ 子ノードの_propagate_enter_tree()（再帰）
```

### データフロー図

```
[入力]               [処理]                    [出力]

add_child()    ───▶ _propagate_enter      ───▶ tree_entered
                    _tree()                    シグナル発火
                         │                          │
                         ├─ ツリー設定               ▼
                         ├─ 通知送信            シグナル接続先
                         └─ シグナル発火        への通知
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| node.h | `scene/main/node.h` | ヘッダ | Nodeクラス定義 |
| node.cpp | `scene/main/node.cpp` | ソース | _propagate_enter_tree()実装、シグナル定義 |
| scene_string_names.h | `scene/scene_string_names.h` | ヘッダ | tree_entered文字列名 |
| Node.xml | `doc/classes/Node.xml` | ドキュメント | 公式ドキュメント定義（行1143-1147） |
