# 通知設計書 76-tree_exiting

## 概要

本ドキュメントは、Godot Engineにおける`tree_exiting`シグナルの設計仕様を記載する。このシグナルはNodeクラスで定義され、ノードがSceneTreeから退出しようとしている際に発火する。ノードはまだツリー内に存在し、アクセス可能な状態である。

### 本通知の処理概要

このシグナルは、ノードがSceneTreeから退出する処理の途中で発火する。`_exit_tree()`仮想メソッドの呼び出し後、`NOTIFICATION_EXIT_TREE`通知の前に発火するため、ノードはまだアクティブなツリー内に存在している。デイニシャライズ処理や参照の切断に適したタイミングである。

**業務上の目的・背景**：ゲーム開発において、ノードがシーンから削除される前にクリーンアップ処理が必要になることが多い。このシグナルを利用することで、ノードがまだ有効な状態でリソースの解放や接続の切断を行うことができる。

**通知の送信タイミング**：以下のイベントでシグナルが発火する：
- `remove_child()`でノードが削除される時
- `queue_free()`でノードが削除予定になった時
- シーン切り替え時
- 親ノードがツリーから退出する時

**通知の受信者**：シグナルに接続したすべてのオブジェクト。主な利用者：
- 依存関係を持つ他のノード
- リソース管理システム
- デバッグ/ロギングシステム

**通知内容の概要**：シグナル自体にはパラメータがない。

**期待されるアクション**：シグナル受信者は以下のような処理を行う：
1. ノードへの参照をクリア
2. 接続の切断
3. 依存リソースのクリーンアップ準備

## 通知種別

シグナル（Signal）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| シグナル名 | `tree_exiting` |
| 定義クラス | Node |
| 送信方式 | 同期（emit_signal） |
| 優先度 | 高（ライフサイクルイベント） |
| リトライ | なし |

### 送信先決定ロジック

シグナルは`_propagate_exit_tree()`メソッド内で以下の順序で発火される：
1. 子ノードの`_propagate_exit_tree()`（逆順で再帰）
2. `GDVIRTUAL_CALL(_exit_tree)`呼び出し
3. `emit_signal(SceneStringName(tree_exiting))`発火
4. `notification(NOTIFICATION_EXIT_TREE, true)`送信

## 通知テンプレート

### シグナル定義

```cpp
// node.cpp
ADD_SIGNAL(MethodInfo("tree_exiting"));
```

### 処理テンプレート（GDScript）

```gdscript
func _ready():
    tree_exiting.connect(_on_tree_exiting)

func _on_tree_exiting():
    print("ノードがツリーから退出しようとしています")
    # クリーンアップ処理
    _cleanup_connections()
```

### 処理テンプレート（C++）

```cpp
// シグナル接続
node->connect("tree_exiting", callable_mp(this, &MyClass::_on_node_exiting_tree));

void MyClass::_on_node_exiting_tree() {
    // ノードがまだアクティブな間にクリーンアップ
    _node_ref = nullptr;
}
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| なし | シグナルにはパラメータがない | - | - |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 構造変更 | remove_child() | ノードがツリー内 | 子ノード削除 |
| 削除 | queue_free() | ノードがツリー内 | 削除予約 |
| シーン | シーン切り替え | ノードがツリー内 | シーンアンロード |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| is_blocking_signals() | シグナルブロック中 |
| ノードがツリー外 | すでに退出済み |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[remove_child/queue_free] --> B[_propagate_exit_tree開始]
    B --> C[子ノードの_propagate_exit_tree（逆順）]
    C --> D[GDVIRTUAL_CALL _exit_tree]
    D --> E[emit_signal tree_exiting]
    E --> F[notification EXIT_TREE]
    F --> G[SceneTree.node_removed通知]
    G --> H{親ノードあり?}
    H -->|Yes| I[親にchild_exiting_tree発火]
    H -->|No| J[グループからの削除]
    I --> J
    J --> K[ツリー/Viewport/depth クリア]
    K --> L[_propagate_after_exit_tree]
    L --> M[終了]
```

## データベース参照・更新仕様

該当なし（本シグナルはエンジン内部のノードシステムで処理され、データベースは使用しない）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 特になし | - | - |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 制限 | なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- ライフサイクルイベントであるため、特別なセキュリティ考慮は不要
- シグナルハンドラ内でのノードアクセスは安全（まだツリー内）

## 備考

- `_exit_tree()`仮想メソッドの後、`NOTIFICATION_EXIT_TREE`の前に発火
- ノードはまだアクティブなツリー内にあるため、他のノードへのアクセスが可能
- `tree_exited`シグナルはノードが完全にツリーから退出した後に発火する
- 公式ドキュメントでは「デストラクタ」のような用途を推奨

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Nodeクラスの退出処理構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | node.h | `scene/main/node.h` | _propagate_exit_tree()宣言 |
| 1-2 | scene_string_names.h | `scene/scene_string_names.h` | tree_exiting文字列名定義 |

**読解のコツ**: 退出処理は子ノードから先に（逆順で）行われる。これはenter処理と逆の順序。

#### Step 2: エントリーポイントを理解する

シグナルの発火元を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | node.cpp | `scene/main/node.cpp` | _propagate_exit_tree()メソッド（行404-451） |

**主要処理フロー**:
1. **行413-418**: 子ノードの退出処理（逆順）
2. **行421**: `GDVIRTUAL_CALL(_exit_tree)`
3. **行423**: `emit_signal(SceneStringName(tree_exiting))`
4. **行425**: `notification(NOTIFICATION_EXIT_TREE, true)`

#### Step 3: シグナル定義を理解する

シグナルの登録処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | node.cpp | `scene/main/node.cpp` | _bind_methods()内のADD_SIGNAL（行4026） |

### プログラム呼び出し階層図

```
Node::remove_child(Node* p_child)
    │
    └─ Node::_propagate_exit_tree()
           │
           ├─ 子ノードの_propagate_exit_tree()（逆順・再帰）
           │
           ├─ GDVIRTUAL_CALL(_exit_tree)
           │
           ├─ emit_signal("tree_exiting")  ← シグナル発火
           │
           ├─ notification(NOTIFICATION_EXIT_TREE, true)
           │
           ├─ SceneTree::node_removed(this)
           │
           ├─ 親ノードへchild_exiting_tree発火
           │
           ├─ グループからの削除
           │
           └─ data.tree = nullptr（ツリークリア）
```

### データフロー図

```
[入力]               [処理]                    [出力]

remove_child() ───▶ _propagate_exit      ───▶ tree_exiting
                    _tree()                    シグナル発火
                         │                          │
                         ├─ 子ノード処理             ▼
                         ├─ _exit_tree          シグナル接続先
                         ├─ シグナル発火        への通知
                         └─ ツリークリア
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| node.h | `scene/main/node.h` | ヘッダ | Nodeクラス定義 |
| node.cpp | `scene/main/node.cpp` | ソース | _propagate_exit_tree()実装、シグナル定義 |
| scene_string_names.h | `scene/scene_string_names.h` | ヘッダ | tree_exiting文字列名 |
| Node.xml | `doc/classes/Node.xml` | ドキュメント | 公式ドキュメント定義（行1155-1159） |
