# 通知設計書 77-tree_exited

## 概要

本ドキュメントは、Godot Engineにおける`tree_exited`シグナルの設計仕様を記載する。このシグナルはNodeクラスで定義され、ノードがSceneTreeから完全に退出した後に発火する。ノードはもはやアクティブなツリーの一部ではなく、孤立した状態となっている。

### 本通知の処理概要

このシグナルは、ノードがSceneTreeから完全に退出した後、`_propagate_after_exit_tree()`メソッド内で発火する。`NOTIFICATION_EXIT_TREE`通知の後に発火するため、ノードはすでにツリーから切り離されており、`is_inside_tree()`はfalseを返す状態である。

**業務上の目的・背景**：ゲーム開発において、ノードがシーンから完全に退出した後に最終的なクリーンアップ処理が必要になることがある。このシグナルを利用することで、ノードが完全にツリーから離れたことを確認してから、外部システムへの通知や参照のクリアを行うことができる。

**通知の送信タイミング**：以下のイベントでシグナルが発火する：
- `remove_child()`でノードが削除された後
- `queue_free()`でノードが削除された後
- シーン切り替えでノードがアンロードされた後
- 親ノードがツリーから退出した後

**通知の受信者**：シグナルに接続したすべてのオブジェクト。主な利用者：
- 外部の監視システム
- リソース管理システム
- メモリ管理システム

**通知内容の概要**：シグナル自体にはパラメータがない。

**期待されるアクション**：シグナル受信者は以下のような処理を行う：
1. ノードへの参照を完全にクリア
2. 関連リソースの解放
3. カウンターやリストからの削除

## 通知種別

シグナル（Signal）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| シグナル名 | `tree_exited` |
| 定義クラス | Node |
| 送信方式 | 同期（emit_signal） |
| 優先度 | 高（ライフサイクルイベント） |
| リトライ | なし |

### 送信先決定ロジック

シグナルは`_propagate_after_exit_tree()`メソッド内で以下の順序で発火される：
1. オーナーのクリーンアップ
2. 子ノードの`_propagate_after_exit_tree()`（逆順で再帰）
3. `emit_signal(SceneStringName(tree_exited))`発火

## 通知テンプレート

### シグナル定義

```cpp
// node.cpp
ADD_SIGNAL(MethodInfo("tree_exited"));
```

### 処理テンプレート（GDScript）

```gdscript
func _ready():
    # 他のノードの退出を監視
    var target_node = get_node("../TargetNode")
    target_node.tree_exited.connect(_on_target_exited)

func _on_target_exited():
    print("ターゲットノードがツリーから完全に退出しました")
    target_ref = null  # 参照をクリア
```

### 処理テンプレート（C++）

```cpp
// シグナル接続
node->connect("tree_exited", callable_mp(this, &MyClass::_on_node_exited_tree));

void MyClass::_on_node_exited_tree() {
    // ノードが完全にツリーから退出した後の処理
    _tracked_nodes.erase(node);
}
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| なし | シグナルにはパラメータがない | - | - |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 構造変更 | remove_child()完了 | 退出処理完了 | 子ノード削除後 |
| 削除 | queue_free()実行 | 退出処理完了 | 削除実行後 |
| シーン | シーン切り替え完了 | 退出処理完了 | シーンアンロード後 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| is_blocking_signals() | シグナルブロック中 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[_propagate_exit_tree完了] --> B[_propagate_after_exit_tree開始]
    B --> C{オーナーがprune対象外?}
    C -->|Yes| D[オーナーをクリア]
    C -->|No| E[スキップ]
    D --> F[子ノードの_propagate_after_exit_tree]
    E --> F
    F --> G[emit_signal tree_exited]
    G --> H[終了]
```

## データベース参照・更新仕様

該当なし（本シグナルはエンジン内部のノードシステムで処理され、データベースは使用しない）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 特になし | - | - |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 制限 | なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- ライフサイクルイベントであるため、特別なセキュリティ考慮は不要
- シグナルハンドラ内でのノードアクセスは注意が必要（ツリー外）

## 備考

- `NOTIFICATION_EXIT_TREE`通知の後に発火
- ノードはすでにツリーから切り離されている（`is_inside_tree()` = false）
- `tree_exiting`シグナルの後に発火する（tree_exitingはノードがまだツリー内）
- 公式ドキュメントでは「ノードが完全に退出した後」の通知と説明

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

退出後処理の構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | node.h | `scene/main/node.h` | _propagate_after_exit_tree()宣言 |
| 1-2 | scene_string_names.h | `scene/scene_string_names.h` | tree_exited文字列名定義 |

**読解のコツ**: `_propagate_after_exit_tree()`は`_propagate_exit_tree()`の後に呼び出される別の処理。ツリークリア後の追加処理。

#### Step 2: エントリーポイントを理解する

シグナルの発火元を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | node.cpp | `scene/main/node.cpp` | _propagate_after_exit_tree()メソッド（行385-402） |

**主要処理フロー**:
1. **行387-391**: オーナーのクリーンアップ（prune対象外の場合）
2. **行393-398**: 子ノードの処理（逆順）
3. **行401**: `emit_signal(SceneStringName(tree_exited))`

#### Step 3: シグナル定義を理解する

シグナルの登録処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | node.cpp | `scene/main/node.cpp` | _bind_methods()内のADD_SIGNAL（行4027） |

### プログラム呼び出し階層図

```
Node::_propagate_exit_tree() 完了後
    │
    └─ Node::_propagate_after_exit_tree()
           │
           ├─ オーナーのクリーンアップ（条件付き）
           │      └─ _clean_up_owner()
           │
           ├─ 子ノードの_propagate_after_exit_tree()（逆順・再帰）
           │
           └─ emit_signal("tree_exited")  ← シグナル発火
```

### データフロー図

```
[入力]               [処理]                    [出力]

_propagate_exit  ───▶ _propagate_after     ───▶ tree_exited
_tree()完了           _exit_tree()              シグナル発火
                         │                          │
                         ├─ オーナークリア            ▼
                         ├─ 子ノード処理         シグナル接続先
                         └─ シグナル発火         への通知
                                                     │
                                                     ▼
                                               is_inside_tree()
                                               = false
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| node.h | `scene/main/node.h` | ヘッダ | Nodeクラス定義 |
| node.cpp | `scene/main/node.cpp` | ソース | _propagate_after_exit_tree()実装、シグナル定義 |
| scene_string_names.h | `scene/scene_string_names.h` | ヘッダ | tree_exited文字列名 |
| Node.xml | `doc/classes/Node.xml` | ドキュメント | 公式ドキュメント定義（行1149-1153） |
