# 通知設計書 78-child_entered_tree

## 概要

本ドキュメントは、Godot Engineにおける`child_entered_tree`シグナルの設計仕様を記載する。このシグナルはNodeクラスで定義され、子ノードがSceneTreeに入った際に親ノードから発火する。子ノードの追加を監視するための重要なシグナルである。

### 本通知の処理概要

このシグナルは、子ノードがSceneTreeに追加された直後、子ノード自身の`NOTIFICATION_ENTER_TREE`および`tree_entered`シグナルの後に発火する。親ノードが子の追加を検知し、必要な初期化や接続を行うために使用される。

**業務上の目的・背景**：ゲーム開発において、親ノードが動的に追加される子ノードを監視し、適切な初期化やセットアップを行う必要がある場合がある。このシグナルを利用することで、`add_child()`の呼び出し元に依存せず、子ノードの追加を確実に検知できる。

**通知の送信タイミング**：以下のイベントでシグナルが発火する：
- 子ノードの`_propagate_enter_tree()`完了直後
- 親ノードがすでにツリー内にある状態で`add_child()`が呼ばれた時
- シーンインスタンス化時に子ノードがツリーに入る時

**通知の受信者**：シグナルに接続したすべてのオブジェクト。主な利用者：
- 親ノード自身
- 子ノード管理システム
- UIコンテナ

**通知内容の概要**：追加された子ノードへの参照がパラメータとして渡される。

**期待されるアクション**：シグナル受信者は以下のような処理を行う：
1. 子ノードの初期化
2. 子ノードへのシグナル接続
3. 子ノードリストの更新

## 通知種別

シグナル（Signal）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| シグナル名 | `child_entered_tree` |
| 定義クラス | Node |
| 送信方式 | 同期（emit_signalp） |
| 優先度 | 高（ライフサイクルイベント） |
| リトライ | なし |

### 送信先決定ロジック

シグナルは`_propagate_enter_tree()`メソッド内で以下の条件で発火される：
1. 親ノードが存在する（`data.parent != nullptr`）
2. 子ノードの`tree_entered`シグナル発火後

## 通知テンプレート

### シグナル定義

```cpp
// node.cpp
ADD_SIGNAL(MethodInfo("child_entered_tree", PropertyInfo(Variant::OBJECT, "node", PROPERTY_HINT_NONE, "", PROPERTY_USAGE_DEFAULT, "Node")));
```

### 処理テンプレート（GDScript）

```gdscript
func _ready():
    child_entered_tree.connect(_on_child_entered)

func _on_child_entered(node: Node):
    print("子ノード追加: ", node.name)
    # 子ノードの初期化
    if node is MyCustomNode:
        _setup_custom_node(node)
```

### 処理テンプレート（C++）

```cpp
// シグナル接続
connect("child_entered_tree", callable_mp(this, &MyContainer::_on_child_entered));

void MyContainer::_on_child_entered(Node *p_node) {
    // 子ノードが追加された時の処理
    _child_list.push_back(p_node);
    _refresh_layout();
}
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| node | 追加された子ノード | シグナルパラメータ | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 構造変更 | add_child() | 親がツリー内 | 子ノード追加 |
| シーン | PackedScene.instantiate() | 親がツリー内 | シーンインスタンス化 |
| 構造変更 | reparent() | 新しい親がツリー内 | 再配置 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| is_blocking_signals() | シグナルブロック中 |
| 親ノードがツリー外 | 親がツリーに入っていない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[子ノード._propagate_enter_tree] --> B[子ノードの初期化完了]
    B --> C[子ノードのtree_entered発火]
    C --> D[SceneTree.node_added]
    D --> E{親ノードあり?}
    E -->|Yes| F[親ノードへchild_entered_tree発火]
    E -->|No| G[処理続行]
    F --> G
    G --> H[子ノードの子の_propagate_enter_tree]
    H --> I[終了]
```

## データベース参照・更新仕様

該当なし（本シグナルはエンジン内部のノードシステムで処理され、データベースは使用しない）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 特になし | - | - |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 制限 | なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- ライフサイクルイベントであるため、特別なセキュリティ考慮は不要
- シグナルハンドラ内での子ノードアクセスは安全（すでにツリー内）

## 備考

- 子ノードの`NOTIFICATION_ENTER_TREE`および`tree_entered`の後に発火
- 子ノードはすでにアクティブなツリー内にある
- パラメータとして子ノードへの参照が渡される
- 公式ドキュメントでは`add_child()`やシーン変更時の発火を説明

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

親子関係の構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | node.h | `scene/main/node.h` | data.parentメンバ |

**読解のコツ**: `data.parent`は親ノードへの直接ポインタ。ツリー内でのみ有効な親子関係を示す。

#### Step 2: エントリーポイントを理解する

シグナルの発火元を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | node.cpp | `scene/main/node.cpp` | _propagate_enter_tree()メソッド（行362-366） |

**主要処理フロー**:
1. **行362**: `if (data.parent)`チェック
2. **行363-364**: Variantへの変換
3. **行365**: `data.parent->emit_signalp(SNAME("child_entered_tree"), &cptr, 1)`

#### Step 3: シグナル定義を理解する

シグナルの登録処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | node.cpp | `scene/main/node.cpp` | _bind_methods()内のADD_SIGNAL（行4028） |

### プログラム呼び出し階層図

```
子ノード::_propagate_enter_tree()
    │
    ├─ notification(NOTIFICATION_ENTER_TREE)
    │
    ├─ GDVIRTUAL_CALL(_enter_tree)
    │
    ├─ emit_signal("tree_entered")
    │
    ├─ SceneTree::node_added(this)
    │
    ├─ if (data.parent) {
    │      Variant c = this;
    │      data.parent->emit_signalp("child_entered_tree", &c, 1);
    │  }  ← シグナル発火
    │
    └─ 子ノードの子の_propagate_enter_tree()（再帰）
```

### データフロー図

```
[入力]               [処理]                    [出力]

add_child()    ───▶ 子._propagate_enter   ───▶ child_entered
                    _tree()                    _tree シグナル
                         │                          │
                         ├─ 子の初期化               ▼
                         ├─ tree_entered         パラメータ:
                         └─ 親へシグナル         Node* 子ノード
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| node.h | `scene/main/node.h` | ヘッダ | Nodeクラス定義 |
| node.cpp | `scene/main/node.cpp` | ソース | _propagate_enter_tree()実装、シグナル定義 |
| Node.xml | `doc/classes/Node.xml` | ドキュメント | 公式ドキュメント定義（行1096-1101） |
