# 通知設計書 79-child_exiting_tree

## 概要

本ドキュメントは、Godot Engineにおける`child_exiting_tree`シグナルの設計仕様を記載する。このシグナルはNodeクラスで定義され、子ノードがSceneTreeから退出しようとしている際に親ノードから発火する。子ノードの削除を監視するための重要なシグナルである。

### 本通知の処理概要

このシグナルは、子ノードがSceneTreeから退出する処理の途中で発火する。子ノード自身の`tree_exiting`シグナルおよび`NOTIFICATION_EXIT_TREE`通知の後に発火するが、子ノードはまだアクティブなツリー内に存在している状態である。

**業務上の目的・背景**：ゲーム開発において、親ノードが子ノードの削除を検知し、適切なクリーンアップやリソース解放を行う必要がある場合がある。このシグナルを利用することで、子ノードがまだ有効な状態で参照のクリアや依存関係の切断を行うことができる。

**通知の送信タイミング**：以下のイベントでシグナルが発火する：
- 子ノードの`_propagate_exit_tree()`処理中
- `remove_child()`で子ノードが削除される時
- `queue_free()`で子ノードが削除予定になった時
- シーン切り替え時

**通知の受信者**：シグナルに接続したすべてのオブジェクト。主な利用者：
- 親ノード自身
- 子ノード管理システム
- リソース管理システム

**通知内容の概要**：退出しようとしている子ノードへの参照がパラメータとして渡される。

**期待されるアクション**：シグナル受信者は以下のような処理を行う：
1. 子ノードへの参照をクリア
2. 子ノード関連のリソース解放
3. 子ノードリストの更新

## 通知種別

シグナル（Signal）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| シグナル名 | `child_exiting_tree` |
| 定義クラス | Node |
| 送信方式 | 同期（emit_signalp） |
| 優先度 | 高（ライフサイクルイベント） |
| リトライ | なし |

### 送信先決定ロジック

シグナルは`_propagate_exit_tree()`メソッド内で以下の条件で発火される：
1. 親ノードが存在する（`data.parent != nullptr`）
2. 子ノードの`tree_exiting`シグナルおよび`NOTIFICATION_EXIT_TREE`の後

## 通知テンプレート

### シグナル定義

```cpp
// node.cpp
ADD_SIGNAL(MethodInfo("child_exiting_tree", PropertyInfo(Variant::OBJECT, "node", PROPERTY_HINT_NONE, "", PROPERTY_USAGE_DEFAULT, "Node")));
```

### 処理テンプレート（GDScript）

```gdscript
func _ready():
    child_exiting_tree.connect(_on_child_exiting)

func _on_child_exiting(node: Node):
    print("子ノード退出中: ", node.name)
    # 子ノードはまだアクセス可能
    _cleanup_child_references(node)
```

### 処理テンプレート（C++）

```cpp
// シグナル接続
connect("child_exiting_tree", callable_mp(this, &MyContainer::_on_child_exiting));

void MyContainer::_on_child_exiting(Node *p_node) {
    // 子ノードがまだアクティブな間にクリーンアップ
    _child_list.erase(p_node);
    _refresh_layout();
}
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| node | 退出中の子ノード | シグナルパラメータ | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 構造変更 | remove_child() | 親がツリー内 | 子ノード削除 |
| 削除 | queue_free() | 親がツリー内 | 削除予約 |
| シーン | シーン切り替え | 親がツリー内 | シーンアンロード |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| is_blocking_signals() | シグナルブロック中 |
| 親ノードがツリー外 | 親がすでにツリーから退出 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[子ノード._propagate_exit_tree開始] --> B[子の子ノードの退出処理（逆順）]
    B --> C[GDVIRTUAL_CALL _exit_tree]
    C --> D[emit_signal tree_exiting]
    D --> E[notification EXIT_TREE]
    E --> F[SceneTree.node_removed]
    F --> G{親ノードあり?}
    G -->|Yes| H[親ノードへchild_exiting_tree発火]
    G -->|No| I[グループからの削除]
    H --> I
    I --> J[ツリークリア]
    J --> K[終了]
```

## データベース参照・更新仕様

該当なし（本シグナルはエンジン内部のノードシステムで処理され、データベースは使用しない）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 特になし | - | - |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 制限 | なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- ライフサイクルイベントであるため、特別なセキュリティ考慮は不要
- シグナルハンドラ内での子ノードアクセスは安全（まだツリー内）

## 備考

- 子ノードの`tree_exiting`シグナルおよび`NOTIFICATION_EXIT_TREE`の後に発火
- 子ノードはまだアクティブなツリー内にある（アクセス可能）
- パラメータとして子ノードへの参照が渡される
- 公式ドキュメントでは「子ノードがツリーから退出しようとしている時」と説明

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

退出処理の親子関係を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | node.h | `scene/main/node.h` | data.parentメンバ |

**読解のコツ**: 子ノードの退出処理中、`data.parent`はまだ有効な参照を保持している。

#### Step 2: エントリーポイントを理解する

シグナルの発火元を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | node.cpp | `scene/main/node.cpp` | _propagate_exit_tree()メソッド（行430-434） |

**主要処理フロー**:
1. **行430**: `if (data.parent)`チェック
2. **行431-432**: Variantへの変換
3. **行433**: `data.parent->emit_signalp(SNAME("child_exiting_tree"), &cptr, 1)`

#### Step 3: シグナル定義を理解する

シグナルの登録処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | node.cpp | `scene/main/node.cpp` | _bind_methods()内のADD_SIGNAL（行4029） |

### プログラム呼び出し階層図

```
子ノード::_propagate_exit_tree()
    │
    ├─ 子の子ノードの_propagate_exit_tree()（逆順・再帰）
    │
    ├─ GDVIRTUAL_CALL(_exit_tree)
    │
    ├─ emit_signal("tree_exiting")
    │
    ├─ notification(NOTIFICATION_EXIT_TREE, true)
    │
    ├─ SceneTree::node_removed(this)
    │
    ├─ if (data.parent) {
    │      Variant c = this;
    │      data.parent->emit_signalp("child_exiting_tree", &c, 1);
    │  }  ← シグナル発火
    │
    ├─ グループからの削除
    │
    └─ ツリークリア
```

### データフロー図

```
[入力]               [処理]                    [出力]

remove_child() ───▶ 子._propagate_exit    ───▶ child_exiting
                    _tree()                    _tree シグナル
                         │                          │
                         ├─ 子の処理                 ▼
                         ├─ tree_exiting         パラメータ:
                         ├─ EXIT_TREE通知        Node* 子ノード
                         └─ 親へシグナル
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| node.h | `scene/main/node.h` | ヘッダ | Nodeクラス定義 |
| node.cpp | `scene/main/node.cpp` | ソース | _propagate_exit_tree()実装、シグナル定義 |
| Node.xml | `doc/classes/Node.xml` | ドキュメント | 公式ドキュメント定義（行1103-1108） |
