# API設計書

## 概要

本ドキュメントは、Horseフレームワーク（Delphi/Lazarus向けWebフレームワーク）が提供するHTTP API構築機能の設計仕様を記載する。HorseはExpress.jsにインスパイアされた軽量Webフレームワークであり、RESTful APIの構築を容易にする。本書では、フレームワークが提供するルーティング、リクエスト/レスポンス処理、ミドルウェア機能を中心に解説する。

## 共通仕様

### ベースURL

```
http://{host}:{port}
```

- デフォルトホスト: `0.0.0.0`
- デフォルトポート: `9000`

### 認証方式

Horseフレームワーク自体は認証機能を内蔵していないが、以下の公式ミドルウェアにより認証を実現可能：

- **horse/basic-auth**: Basic認証
- **horse/jwt**: JWT（JSON Web Token）認証
- **horse-bearer-auth**: Bearer Token認証（サードパーティ）

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Content-Type | - | リクエストボディの形式（application/json等） |
| Accept | - | レスポンスの期待形式 |
| Authorization | - | 認証トークン（認証ミドルウェア使用時） |

### 共通エラーレスポンス

Horseフレームワークは`EHorseException`クラスによる構造化エラーレスポンスをサポート：

| ステータスコード | 説明 |
| --- | --- |
| 100 | Continue |
| 200 | OK - 正常終了 |
| 201 | Created - リソース作成成功 |
| 204 | No Content - コンテンツなし |
| 301 | Moved Permanently - 恒久的リダイレクト |
| 302 | Found - 一時的リダイレクト |
| 303 | See Other - リダイレクト |
| 400 | Bad Request - リクエストパラメータ不正 |
| 401 | Unauthorized - 認証エラー |
| 403 | Forbidden - 権限エラー |
| 404 | Not Found - リソース未検出 |
| 405 | Method Not Allowed - 許可されていないメソッド |
| 422 | Unprocessable Entity - バリデーションエラー |
| 500 | Internal Server Error - サーバーエラー |
| 502 | Bad Gateway - ゲートウェイエラー |
| 503 | Service Unavailable - サービス利用不可 |

### サポートMIMEタイプ

| タイプ | Content-Type |
| --- | --- |
| MultiPartFormData | multipart/form-data |
| ApplicationXWWWFormURLEncoded | application/x-www-form-urlencoded |
| ApplicationJSON | application/json |
| ApplicationOctetStream | application/octet-stream |
| ApplicationXML | application/xml |
| ApplicationJavaScript | application/javascript |
| ApplicationPDF | application/pdf |
| ApplicationTypeScript | application/typescript |
| ApplicationZIP | application/zip |
| TextPlain | text/plain |
| TextCSS | text/css |
| TextCSV | text/csv |
| TextHTML | text/html |
| ImageJPEG | image/jpeg |
| ImagePNG | image/png |
| ImageGIF | image/gif |

## API一覧

HorseはREST APIを構築するためのフレームワークであり、以下のHTTPメソッドに対応したルーティング登録が可能：

| カテゴリ | メソッド | 説明 |
| --- | --- | --- |
| ルーティング | GET | リソースの取得 |
| ルーティング | POST | リソースの作成 |
| ルーティング | PUT | リソースの更新（全体置換） |
| ルーティング | PATCH | リソースの部分更新 |
| ルーティング | DELETE | リソースの削除 |
| ルーティング | HEAD | ヘッダー情報の取得 |
| ルーティング | ALL | 全HTTPメソッドへの対応 |
| ミドルウェア | Use | ミドルウェアの登録 |

## フレームワークAPI定義

### ルーティング

#### 1. GET - リソース取得

GETメソッドによるエンドポイント登録

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 登録メソッド | `THorse.Get(APath, ACallback)` |
| HTTPメソッド | GET |

**シグネチャ**

```delphi
class function Get(const APath: string; const ACallback: THorseCallback): THorseCore;
class function Get(const APath: string; const ACallback: THorseCallbackRequestResponse): THorseCore;
class function Get(const APath: string; const ACallback: THorseCallbackRequest): THorseCore;
class function Get(const APath: string; const ACallback: THorseCallbackResponse): THorseCore;
```

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| APath | string | ○ | エンドポイントパス（例: `/users`, `/users/:id`） |
| ACallback | THorseCallback / THorseCallbackRequestResponse / THorseCallbackRequest / THorseCallbackResponse | ○ | リクエスト処理コールバック |

**使用例**

```delphi
THorse.Get('/ping',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.Send('pong');
  end);

THorse.Get('/users/:id',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.Send('User ID: ' + Req.Params['id']);
  end);
```

---

#### 2. POST - リソース作成

POSTメソッドによるエンドポイント登録

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 登録メソッド | `THorse.Post(APath, ACallback)` |
| HTTPメソッド | POST |

**シグネチャ**

```delphi
class function Post(const APath: string; const ACallback: THorseCallback): THorseCore;
class function Post(const APath: string; const ACallback: THorseCallbackRequestResponse): THorseCore;
class function Post(const APath: string; const ACallback: THorseCallbackRequest): THorseCore;
class function Post(const APath: string; const ACallback: THorseCallbackResponse): THorseCore;
```

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| APath | string | ○ | エンドポイントパス |
| ACallback | THorseCallback系 | ○ | リクエスト処理コールバック |

**使用例**

```delphi
THorse.Post('/users',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    // Req.Body でリクエストボディを取得
    Res.Status(THTTPStatus.Created).Send('User created');
  end);
```

---

#### 3. PUT - リソース更新（全体置換）

PUTメソッドによるエンドポイント登録

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 登録メソッド | `THorse.Put(APath, ACallback)` |
| HTTPメソッド | PUT |

**シグネチャ**

```delphi
class function Put(const APath: string; const ACallback: THorseCallback): THorseCore;
class function Put(const APath: string; const ACallback: THorseCallbackRequestResponse): THorseCore;
class function Put(const APath: string; const ACallback: THorseCallbackRequest): THorseCore;
class function Put(const APath: string; const ACallback: THorseCallbackResponse): THorseCore;
```

---

#### 4. PATCH - リソース部分更新

PATCHメソッドによるエンドポイント登録

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 登録メソッド | `THorse.Patch(APath, ACallback)` |
| HTTPメソッド | PATCH |
| 対応コンパイラ | Delphi XE8以降 / FPC |

**シグネチャ**

```delphi
class function Patch(const APath: string; const ACallback: THorseCallback): THorseCore;
class function Patch(const APath: string; const ACallback: THorseCallbackRequestResponse): THorseCore;
class function Patch(const APath: string; const ACallback: THorseCallbackRequest): THorseCore;
class function Patch(const APath: string; const ACallback: THorseCallbackResponse): THorseCore;
```

---

#### 5. DELETE - リソース削除

DELETEメソッドによるエンドポイント登録

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 登録メソッド | `THorse.Delete(APath, ACallback)` |
| HTTPメソッド | DELETE |
| 対応コンパイラ | Delphi XE8以降 / FPC |

**シグネチャ**

```delphi
class function Delete(const APath: string; const ACallback: THorseCallback): THorseCore;
class function Delete(const APath: string; const ACallback: THorseCallbackRequestResponse): THorseCore;
class function Delete(const APath: string; const ACallback: THorseCallbackRequest): THorseCore;
class function Delete(const APath: string; const ACallback: THorseCallbackResponse): THorseCore;
```

---

#### 6. HEAD - ヘッダー取得

HEADメソッドによるエンドポイント登録

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 登録メソッド | `THorse.Head(APath, ACallback)` |
| HTTPメソッド | HEAD |

**シグネチャ**

```delphi
class function Head(const APath: string; const ACallback: THorseCallback): THorseCore;
class function Head(const APath: string; const ACallback: THorseCallbackRequestResponse): THorseCore;
class function Head(const APath: string; const ACallback: THorseCallbackRequest): THorseCore;
class function Head(const APath: string; const ACallback: THorseCallbackResponse): THorseCore;
```

---

#### 7. ALL - 全メソッド対応

全HTTPメソッドに対応するエンドポイント登録

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 登録メソッド | `THorse.All(APath, ACallback)` |
| HTTPメソッド | GET, POST, PUT, PATCH, DELETE, HEAD |

**シグネチャ**

```delphi
class function All(const APath: string; const ACallback: THorseCallback): THorseCore;
class function All(const APath: string; const ACallback: THorseCallbackRequestResponse): THorseCore;
class function All(const APath: string; const ACallback: THorseCallbackRequest): THorseCore;
class function All(const APath: string; const ACallback: THorseCallbackResponse): THorseCore;
```

---

### ミドルウェア

#### 8. Use - ミドルウェア登録

グローバルまたはパス指定のミドルウェア登録

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 登録メソッド | `THorse.Use(ACallback)` / `THorse.Use(APath, ACallback)` |
| 用途 | 認証、ログ、CORS等の共通処理 |

**シグネチャ**

```delphi
class function Use(const ACallback: THorseCallback): THorseCore;
class function Use(const APath: string; const ACallback: THorseCallback): THorseCore;
class function Use(const ACallbacks: array of THorseCallback): THorseCore;
class function Use(const APath: string; const ACallbacks: array of THorseCallback): THorseCore;
```

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| APath | string | - | ミドルウェアを適用するパス（省略時は全パス） |
| ACallback | THorseCallback | ○ | ミドルウェア処理コールバック |
| ACallbacks | array of THorseCallback | - | 複数ミドルウェアの配列 |

**使用例**

```delphi
// グローバルミドルウェア
THorse.Use(
  procedure(Req: THorseRequest; Res: THorseResponse; Next: TProc)
  begin
    // 前処理
    Next;  // 次のミドルウェア/ハンドラへ
    // 後処理
  end);

// パス指定ミドルウェア
THorse.Use('/api',
  procedure(Req: THorseRequest; Res: THorseResponse; Next: TProc)
  begin
    // /api以下のリクエストにのみ適用
    Next;
  end);
```

---

### リクエスト処理

#### THorseRequest クラス

HTTP リクエスト情報へのアクセスを提供

**主要メソッド/プロパティ**

| メソッド/プロパティ | 戻り型 | 説明 |
| --- | --- | --- |
| Body | string | リクエストボディ（文字列） |
| Body<T> | T | リクエストボディ（オブジェクト） |
| Headers | THorseCoreParam | HTTPヘッダー |
| Query | THorseCoreParam | クエリパラメータ（?key=value） |
| Params | THorseCoreParam | パスパラメータ（:id等） |
| Cookie | THorseCoreParam | Cookie値 |
| ContentFields | THorseCoreParam | フォームフィールド |
| Session<T> | T | セッションオブジェクト |
| MethodType | TMethodType | HTTPメソッド種別 |
| ContentType | string | Content-Typeヘッダー |
| Host | string | ホスト名 |
| PathInfo | string | リクエストパス |
| RawWebRequest | TWebRequest | 生のWebRequestオブジェクト |

**パラメータアクセス例**

```delphi
// パスパラメータ: /users/:id
LUserId := Req.Params['id'];

// クエリパラメータ: /users?page=1&limit=10
LPage := Req.Query['page'];
LLimit := Req.Query['limit'];

// ヘッダー
LAuthToken := Req.Headers['Authorization'];

// ボディ
LBodyText := Req.Body;
```

---

### レスポンス処理

#### THorseResponse クラス

HTTPレスポンスの構築を提供

**主要メソッド**

| メソッド | 戻り型 | 説明 |
| --- | --- | --- |
| Send(AContent: string) | THorseResponse | テキストレスポンス送信 |
| Send<T>(AContent: T) | THorseResponse | オブジェクトレスポンス送信 |
| Status(AStatus: THTTPStatus) | THorseResponse | ステータスコード設定 |
| Status(AStatus: Integer) | THorseResponse | ステータスコード設定（数値） |
| RedirectTo(ALocation: string) | THorseResponse | リダイレクト（303） |
| RedirectTo(ALocation, AStatus) | THorseResponse | リダイレクト（ステータス指定） |
| SendFile(AFileName: string) | THorseResponse | ファイル送信（インライン表示） |
| Download(AFileName: string) | THorseResponse | ファイルダウンロード |
| Render(AFileName: string) | THorseResponse | HTMLファイルレンダリング |
| AddHeader(AName, AValue: string) | THorseResponse | ヘッダー追加 |
| RemoveHeader(AName: string) | THorseResponse | ヘッダー削除 |
| ContentType(AContentType: string) | THorseResponse | Content-Type設定 |
| Content(AContent: TObject) | THorseResponse | コンテンツオブジェクト設定 |

**レスポンス例**

```delphi
// テキストレスポンス
Res.Send('Hello, World!');

// JSONレスポンス（要horse/jsonミドルウェア）
Res.Send<TJSONObject>(LJsonObj);

// ステータスコード付きレスポンス
Res.Status(THTTPStatus.Created).Send('Created');

// ファイルダウンロード
Res.Download('/path/to/file.pdf');

// リダイレクト
Res.RedirectTo('/new-location');
```

---

### 例外処理

#### EHorseException クラス

構造化エラーレスポンスの生成

**主要メソッド**

| メソッド | 型 | 説明 |
| --- | --- | --- |
| Error(AValue: string) | EHorseException | エラーメッセージ設定 |
| Status(AValue: THTTPStatus) | EHorseException | HTTPステータス設定 |
| Type(AValue: TMessageType) | EHorseException | メッセージタイプ設定 |
| Title(AValue: string) | EHorseException | タイトル設定 |
| Code(AValue: Integer) | EHorseException | エラーコード設定 |
| Hint(AValue: string) | EHorseException | ヒント設定 |
| Unit(AValue: string) | EHorseException | ユニット名設定 |
| Detail(AValue: string) | EHorseException | 詳細設定 |
| ToJSON | string | JSON文字列に変換 |
| ToJSONObject | TJSONObject | JSONオブジェクトに変換 |

**使用例**

```delphi
raise EHorseException.New
  .Error('User not found')
  .Status(THTTPStatus.NotFound)
  .Detail('The requested user ID does not exist');
```

**エラーレスポンス形式**

```json
{
  "type": "Error",
  "title": "Not Found",
  "code": 404,
  "error": "User not found",
  "hint": "Check the user ID",
  "detail": "The requested user ID does not exist"
}
```

---

### サーバー起動

#### Listen メソッド

HTTPサーバーの起動

**シグネチャ**

```delphi
class procedure Listen;
class procedure Listen(const APort: Integer; const AHost: string = '0.0.0.0';
                       const ACallbackListen: TProc = nil;
                       const ACallbackStopListen: TProc = nil);
class procedure Listen(const APort: Integer; const ACallbackListen: TProc;
                       const ACallbackStopListen: TProc = nil);
class procedure Listen(const AHost: string; const ACallbackListen: TProc = nil;
                       const ACallbackStopListen: TProc = nil);
class procedure Listen(const ACallbackListen: TProc;
                       const ACallbackStopListen: TProc = nil);
```

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| APort | Integer | - | 9000 | リッスンポート番号 |
| AHost | string | - | 0.0.0.0 | バインドホスト |
| ACallbackListen | TProc | - | nil | サーバー起動時コールバック |
| ACallbackStopListen | TProc | - | nil | サーバー停止時コールバック |

**使用例**

```delphi
THorse.Listen(9000);

THorse.Listen(8080, '127.0.0.1',
  procedure
  begin
    Writeln('Server started on port 8080');
  end);
```

---

### サーバー設定

#### プロバイダプロパティ

| プロパティ | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| Host | string | 0.0.0.0 | バインドホスト |
| Port | Integer | 9000 | リッスンポート |
| MaxConnections | Integer | - | 最大同時接続数 |
| ListenQueue | Integer | - | リッスンキューサイズ |
| KeepConnectionAlive | Boolean | True | Keep-Alive有効 |
| IOHandleSSL | IHorseProviderIOHandleSSL | - | SSL/TLS設定 |

---

### ルートグループ

#### Route / Group メソッド

ルートのグループ化とプレフィックス設定

**シグネチャ**

```delphi
class function Route(const APath: string): IHorseCoreRoute<THorseCore>;
class function Group: IHorseCoreGroup<THorseCore>;
```

**使用例**

```delphi
// Fluent API によるルート定義
THorse.Route('/api/users')
  .Get(procedure(Req: THorseRequest; Res: THorseResponse)
    begin
      Res.Send('Get users');
    end)
  .Post(procedure(Req: THorseRequest; Res: THorseResponse)
    begin
      Res.Send('Create user');
    end)
  .&End;
```

---

## パスパラメータ

Horseは動的パスパラメータをサポート：

| パターン | 説明 | 例 |
| --- | --- | --- |
| `:param` | 名前付きパラメータ | `/users/:id` -> `Req.Params['id']` |
| `*` | ワイルドカード | `/*` -> 全パスにマッチ |
| `(regex)` | 正規表現パターン | `/users/([0-9]+)` |

## 備考

### バージョン情報

- フレームワークバージョン: 3.1.9
- 対応Delphiバージョン: XE7, XE8, 10 Seattle, 10.1 Berlin, 10.2 Tokyo, 10.3 Rio, 10.4 Sydney, 11 Alexandria, 12 Athens, 13 Florence
- 対応Lazarus/FPC: FPC 3.x以降

### プロバイダ種別

| プロバイダ | コンパイルディレクティブ | 説明 |
| --- | --- | --- |
| Console | (デフォルト) | コンソールアプリケーション |
| VCL | HORSE_VCL | VCLフォームアプリケーション |
| ISAPI | HORSE_ISAPI | IIS用ISAPIモジュール |
| Apache | HORSE_APACHE | Apache用モジュール |
| CGI | HORSE_CGI | CGIアプリケーション |
| FastCGI | HORSE_FCGI | FastCGIアプリケーション |
| Daemon | HORSE_DAEMON | Windowsサービス/デーモン |
| LCL | HORSE_LCL | Lazarus LCLアプリケーション |

### 公式ミドルウェア

| ミドルウェア | 機能 |
| --- | --- |
| horse/json (jhonson) | JSON解析・シリアライズ |
| horse/basic-auth | Basic認証 |
| horse/cors | CORS対応 |
| horse/stream | ストリーミング |
| horse/jwt | JWT認証 |
| horse/exception (handle-exception) | 例外ハンドリング |
| horse/logger | リクエストログ |
| horse/compression | レスポンス圧縮 |
