# インフラ設計書

## 概要

本ドキュメントは、Horse Webフレームワークのインフラストラクチャとデプロイメントオプションについてまとめた設計書です。HorseはDelphi/Lazarus向けのExpress風軽量Webフレームワークであり、複数のデプロイメント形態をサポートしています。

## システム構成

### 全体構成

Horseフレームワークは、以下の複数のデプロイメントプロバイダーをサポートするマルチプラットフォームWebフレームワークです：

1. **スタンドアロンHTTPサーバー（Console/VCL）**: Indy HTTPサーバーをベースとした独立稼働型
2. **Windowsサービス（WinSvc/Daemon）**: バックグラウンドサービスとして稼働
3. **CGI/FastCGI**: Apache/nginx等のWebサーバーと連携
4. **ISAPI**: IISとの連携（Windows専用）
5. **Apacheモジュール**: Apache HTTP Serverのネイティブモジュール
6. **LCL**: Lazarus GUIアプリケーション内での稼働

### サーバー構成

| サーバー種別 | 役割 | スペック | 台数 |
| --- | --- | --- | --- |
| アプリケーションサーバー | Horse HTTPサーバー（Console/VCL）として稼働 | デフォルトポート: 9000、デフォルトホスト: 0.0.0.0 | アプリケーション要件に依存 |
| Webサーバー（オプション） | Apache/IIS/nginx - リバースプロキシまたはCGI/ISAPI/モジュール連携 | 標準Webサーバー構成 | 1台以上 |
| DBサーバー（オプション） | アプリケーションが使用するデータベース | アプリケーション要件に依存 | アプリケーション要件に依存 |

### ネットワーク構成

#### スタンドアロン構成（Console/VCL）

```
クライアント → Horse HTTPサーバー（ポート9000）
```

- ホスト: `0.0.0.0`（全インターフェースでリッスン）または指定IP
- ポート: デフォルト `9000`（設定可能）
- SSL/TLS: OpenSSL経由でHTTPS対応可能

#### リバースプロキシ構成

```
クライアント → Webサーバー（Apache/nginx/IIS）→ Horse HTTPサーバー
```

#### CGI/ISAPI/Apacheモジュール構成

```
クライアント → Webサーバー（Apache/IIS）→ Horse CGI/ISAPI/モジュール
```

## ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Indy (TIdHTTPWebBrokerBridge) | Delphi同梱版 | スタンドアロンHTTPサーバー基盤 |
| OpenSSL | システム依存 | SSL/TLS暗号化通信 |
| Apache HTTP Server | 2.4系推奨 | Apacheモジュール/CGI連携時 |
| IIS | Windows Server版 | ISAPI連携時（Windows専用） |
| nginx | 最新安定版 | リバースプロキシ/FastCGI連携時 |

### 公式ミドルウェアライブラリ

| ミドルウェア | 対応環境 | 用途 |
| --- | --- | --- |
| horse/json (jhonson) | Delphi/Lazarus | JSONリクエスト/レスポンス処理 |
| horse/basic-auth | Delphi/Lazarus | Basic認証 |
| horse/cors | Delphi/Lazarus | CORS対応 |
| horse/stream | Delphi/Lazarus | ストリーミング対応 |
| horse/jwt | Delphi/Lazarus | JWT認証 |
| horse/exception | Delphi/Lazarus | 例外ハンドリング |
| horse/logger | Delphi/Lazarus | ロギング |
| horse/compression | Delphi/Lazarus | レスポンス圧縮 |

## 外部サービス連携

| サービス名 | 用途 | 接続方式 |
| --- | --- | --- |
| Apache HTTP Server | Webサーバー連携（モジュール/CGI） | `LoadModule`/CGI実行 |
| IIS | Webサーバー連携（ISAPI） | ISAPI DLL |
| OpenSSL | SSL/TLS暗号化 | ライブラリリンク |
| 外部データベース | データ永続化（アプリケーション依存） | アプリケーション実装による |

## スケーリング設計

### 水平スケーリング

Horseアプリケーションの水平スケーリングは以下の方法で実現可能：

1. **ロードバランサー経由**: 複数のHorseインスタンスをロードバランサー（nginx/HAProxy等）の背後に配置
2. **リバースプロキシ**: Apache/nginxのリバースプロキシ機能を使用

### 垂直スケーリング

- **MaxConnections**: 最大同時接続数の設定（`THorse.MaxConnections`プロパティ）
- **ListenQueue**: 接続待ちキューサイズの設定（`THorse.ListenQueue`プロパティ）
- **KeepConnectionAlive**: HTTP Keep-Alive設定（デフォルト: True）

### 設定パラメータ

```pascal
// 接続設定例
THorse.MaxConnections := 100;
THorse.ListenQueue := 50;
THorse.KeepConnectionAlive := True;
```

## 可用性設計

### 冗長構成

1. **アクティブ-アクティブ**: 複数のHorseインスタンスをロードバランサーで分散
2. **アクティブ-スタンバイ**: Windowsサービス構成時はサービス監視による自動再起動

### フェイルオーバー方式

- Windowsサービス（WinSvc）構成時: サービスコントローラーによる自動再起動
- Linuxデーモン構成時: systemdによるプロセス監視と自動再起動
- コンテナ構成時: Kubernetes/Docker Swarmのヘルスチェックと自動リスタート

### サービス監視エンドポイント

```pascal
// ヘルスチェックエンドポイント例
THorse.Get('/health',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.Status(THTTPStatus.OK).Send('OK');
  end);
```

## バックアップ設計

| 対象 | 方式 | 頻度 | 保持期間 |
| --- | --- | --- | --- |
| アプリケーションバイナリ | ファイルバックアップ/バージョン管理 | デプロイ時 | 世代管理（推奨3世代以上） |
| 設定ファイル | ファイルバックアップ/バージョン管理 | 変更時 | 世代管理（推奨5世代以上） |
| SSL証明書 | 暗号化バックアップ | 更新時 | 有効期限まで |
| ログファイル | ログローテーション/アーカイブ | 日次 | アプリケーション要件による |
| データベース（使用時） | データベースバックアップツール | 日次/時間単位 | アプリケーション要件による |

## 監視設計

### 監視項目

| 監視対象 | 監視方法 | 閾値/条件 |
| --- | --- | --- |
| プロセス稼働状態 | プロセス監視 | プロセス存在確認 |
| HTTPレスポンス | ヘルスチェックエンドポイント | HTTP 200レスポンス |
| ポート疎通 | ポート監視 | TCP接続確立可能 |
| リソース使用率 | OS監視ツール | CPU/メモリ閾値超過 |
| エラーログ | ログ監視 | エラーレベルログ検出 |

### 監視ツール連携

- **Prometheus Metrics**: サードパーティミドルウェア `horse-prometheus-metrics` で対応可能
- **カスタムロギング**: `horse/logger` ミドルウェアまたは `horse-datalogger` で実装

### ログ出力

```pascal
// Syslog対応（Linux/POSIX）
uses ThirdParty.Posix.Syslog;

// 環境変数取得
uses Horse.EnvironmentVariables;
var
  LEnvValue: string;
begin
  LEnvValue := THorseEnvironmentVariables.GetEnvironmentVariable('LOG_LEVEL');
end;
```

## デプロイメント構成

### コンパイルディレクティブ

| ディレクティブ | デプロイメント形態 | 対応プラットフォーム |
| --- | --- | --- |
| （未指定） | Consoleスタンドアロン | Delphi（Windows/Linux） |
| `HORSE_VCL` | VCL GUIアプリケーション | Delphi（Windows） |
| `HORSE_ISAPI` | IIS ISAPI | Delphi（Windows） |
| `HORSE_CGI` | CGI | Delphi/Lazarus |
| `HORSE_FCGI` | FastCGI | Lazarus |
| `HORSE_APACHE` | Apacheモジュール | Delphi/Lazarus |
| `HORSE_DAEMON` | デーモン/バックグラウンドサービス | Delphi/Lazarus |
| `HORSE_LCL` | LCL GUIアプリケーション | Lazarus |

### Apache連携設定例

```apache
# httpd.conf
LoadModule apache_horse_module modules/Apache.dll
<Location /apache_horse>
   SetHandler apache_horse_module-handle
</Location>
```

### SSL/TLS設定

```pascal
// SSL設定例
THorse.IOHandleSSL
  .CertFile('server.crt')
  .KeyFile('server.key')
  .RootCertFile('ca.crt')
  .Active(True);

THorse.Listen(443);
```

## CI/CD構成

### GitHub Actions（テスト）

```yaml
# .github/workflows/tests.yml
name: tests
on:
  push:
    branches: [ master ]
  pull_request:
    branches: [ master ]

jobs:
  test:
    runs-on: windows-latest
    steps:
    - uses: actions/checkout@v2
    - name: Launch Console Tests
      run: ./tests/Console.exe
    - name: Launch VCL Tests
      run: ./tests/VCL.exe
```

### パッケージ管理

- **Boss**: Delphiパッケージマネージャー
- インストール: `boss install horse`

## 対応プラットフォーム

### Delphi

- Delphi 13 Florence
- Delphi 12 Athens
- Delphi 11 Alexandria
- Delphi 10.4 Sydney
- Delphi 10.3 Rio
- Delphi 10.2 Tokyo
- Delphi 10.1 Berlin
- Delphi 10 Seattle
- Delphi XE8
- Delphi XE7

### Lazarus/Free Pascal

- Free Pascal Compiler対応
- Windows/Linux/macOS

## 備考

- Horseフレームワークバージョン: 3.1.9
- ライセンス: MIT License
- デフォルト設定はホスト `0.0.0.0`、ポート `9000`
- SSL/TLSにはOpenSSLライブラリが必要
- 本設計書はフレームワークのコード解析に基づいて作成されており、特定のインフラ構成ファイル（Docker、Kubernetes、Terraform等）は本プロジェクトには含まれていません
