# Horse コードリーディングガイドライン

## はじめに

このガイドラインは、Horseのコードベースを効率的に理解するための手引きです。
Object Pascal（Delphi/Lazarus）に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、Object Pascal（Delphi/Lazarus）の基本構文と概念を解説します。

### 1.1 プログラム構造

Object Pascalのプログラムは「ユニット（Unit）」という単位で構成されます。各ユニットは以下の構造を持ちます。

```pascal
// ファイル: src/Horse.Core.pas:1-9
unit Horse.Core;

{$IF DEFINED(FPC)}
{$MODE DELPHI}{$H+}
{$ENDIF}

interface

uses
```

**構成要素:**
- `unit ユニット名;` - ユニットの宣言（ファイル名と一致）
- `{$IF DEFINED(FPC)}` - コンパイラディレクティブ（条件付きコンパイル）
- `interface` - 公開部分（他のユニットからアクセス可能）
- `implementation` - 実装部分（非公開）

### 1.2 データ型と変数

Object Pascalは強い型付け言語です。変数宣言は以下のパターンで行います。

```pascal
// ファイル: src/Horse.Core.pas:42-46
class var FRoutes: THorseRouterTree;
class var FCallbacks: TList<THorseCallback>;
class var FDefaultHorse: THorseCore;
```

**基本的なデータ型:**
- `string` - 文字列
- `Integer` - 整数
- `Boolean` - 真偽値
- `TObject` - すべてのクラスの基底クラス
- `TList<T>` - ジェネリックリスト
- `TDictionary<K, V>` - ジェネリック辞書

**変数宣言のパターン:**
- `var 変数名: 型;` - ローカル変数
- `class var 変数名: 型;` - クラス変数（静的変数）

### 1.3 制御構造

条件分岐とループの例を示します。

```pascal
// ファイル: src/Horse.Commons.pas:144-161
function StringCommandToMethodType(const ACommand: string): TMethodType;
begin
  Result := TMethodType.mtAny;
  case AnsiIndexText(ACommand, ['DELETE', 'GET', 'HEAD', 'PATCH', 'POST', 'PUT']) of
    0:
      Result := TMethodType.mtDelete;
    1:
      Result := TMethodType.mtGet;
    2:
      Result := TMethodType.mtHead;
    3:
      Result := TMethodType.mtPatch;
    4:
      Result := TMethodType.mtPost;
    5:
      Result := TMethodType.mtPut;
  end;
end;
```

**制御構造:**
- `if ... then ... else` - 条件分岐
- `case ... of ... end` - 複数分岐（switch文相当）
- `for ... in ... do` - 拡張forループ
- `while ... do` - whileループ

### 1.4 関数/メソッド定義

関数とプロシージャの違いを理解することが重要です。

```pascal
// ファイル: src/Horse.Core.pas:205-212
class function THorseCore.RegisterRoute(const AHTTPType: TMethodType; const APath: string; const ACallback: THorseCallback): THorseCore;
var
  LDefaultHorse: THorseCore;
begin
  LDefaultHorse := GetInstance;
  Result := LDefaultHorse;
  LDefaultHorse.GetRoutes.RegisterRoute(AHTTPType, TrimPath(APath), ACallback);
end;
```

**関数とプロシージャ:**
- `function` - 戻り値あり（`Result`で返す）
- `procedure` - 戻り値なし
- `class function` - 静的メソッド（インスタンス不要）
- `const` 引数 - 読み取り専用引数

### 1.5 モジュール/インポート

`uses`節でユニットをインポートします。

```pascal
// ファイル: src/Horse.pas:9-47
uses
{$IF DEFINED(FPC)}
  SysUtils,
  Horse.Provider.FPC.HTTPApplication,
  {$IF DEFINED(HORSE_APACHE)}
  Horse.Provider.FPC.Apache,
  {$ELSEIF DEFINED(HORSE_CGI)}
  Horse.Provider.FPC.CGI,
  {$ENDIF}
{$ELSE}
  System.SysUtils,
  Horse.Provider.Console,
  Horse.Provider.VCL,
{$ENDIF}
  Horse.Core,
  Horse.Request,
  Horse.Response;
```

**インポートのパターン:**
- `interface`セクションの`uses` - 公開インターフェースで使用
- `implementation`セクションの`uses` - 実装でのみ使用
- 条件付きコンパイル - コンパイラ/プラットフォームによる分岐

---

## 2. プロジェクト固有の概念

> このセクションでは、Horseプロジェクト特有の概念を解説します。

### 2.1 フレームワーク固有の概念

Horseは**Express.js風のWebフレームワーク**です。主要概念は以下の通りです。

#### ルーティング

```pascal
// ファイル: samples/delphi/console/Console.dpr:16-20
THorse.Get('/ping',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.Send('pong');
  end);
```

- HTTPメソッド（Get, Post, Put, Delete, Patch）をクラスメソッドとして呼び出し
- パスとコールバックを引数として渡す
- 匿名手続き（ラムダ式相当）を使用

#### ミドルウェア

```pascal
// ファイル: src/Horse.Core.pas:393-397
class function THorseCore.Use(const ACallback: THorseCallback): THorseCore;
begin
  Result := GetInstance;
  Result.Routes.RegisterMiddleware('/', ACallback);
end;
```

- `Use`メソッドでミドルウェアを登録
- すべてのリクエストに対して実行される共通処理

#### プロバイダー

```pascal
// ファイル: src/Horse.pas:100-112
{$ELSEIF DEFINED(HORSE_VCL)}
  THorseProvider = Horse.Provider.VCL.THorseProvider;
{$ELSE}
  THorseProvider =
  {$IF DEFINED(FPC)}
    Horse.Provider.FPC.HTTPApplication.THorseProvider;
  {$ELSE}
    Horse.Provider.Console.THorseProvider;
  {$ENDIF}
{$ENDIF}

  THorse = class(THorseProvider);
```

- プロバイダーはHTTPサーバーの実装を抽象化
- Console, VCL, ISAPI, CGI, Apache, Daemonなど複数の実行環境に対応

### 2.2 プロジェクト独自のパターン

#### シングルトンパターン

```pascal
// ファイル: src/Horse.Core.pas:169-174
class function THorseCore.GetInstance: THorseCore;
begin
  if FDefaultHorse = nil then
    FDefaultHorse := THorseCore.Create;
  Result := FDefaultHorse;
end;
```

- `THorseCore`はシングルトンとして実装
- すべてのルート登録は同一インスタンスに対して行われる

#### Fluent Interface（メソッドチェーン）

```pascal
// ファイル: src/Horse.Response.pas:119-123
function THorseResponse.RedirectTo(const ALocation: string): THorseResponse;
begin
  FWebResponse.SetCustomHeader('Location', ALocation);
  Result := Status(THTTPStatus.SeeOther);
end;
```

- メソッドが`Self`を返すことでメソッドチェーンを実現
- `Res.Status(200).Send('OK')`のような記述が可能

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `Horse.*.pas` | Horseフレームワークのコアファイル | `Horse.Core.pas`, `Horse.Request.pas` |
| `Horse.Provider.*.pas` | 各実行環境用のプロバイダー | `Horse.Provider.Console.pas`, `Horse.Provider.VCL.pas` |
| `Horse.Core.*.pas` | コア機能の詳細実装 | `Horse.Core.RouterTree.pas`, `Horse.Core.Param.pas` |
| `Horse.Provider.FPC.*.pas` | Free Pascal Compiler用のプロバイダー | `Horse.Provider.FPC.CGI.pas` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `THorse*` | Horseフレームワークのクラス | `THorseCore`, `THorseRequest`, `THorseResponse` |
| `EHorse*` | Horseフレームワークの例外クラス | `EHorseException`, `EHorseCallbackInterrupted` |
| `IHorse*` | Horseフレームワークのインターフェース | `IHorseCoreGroup`, `IHorseCoreRoute` |
| `PHorse*` | ポインタ型 | `PHorseCore`, `PHorseModule` |
| `F*` | プライベートフィールド | `FPort`, `FHost`, `FRoutes` |
| `L*` | ローカル変数 | `LCallback`, `LDefaultHorse` |
| `A*` | 引数（パラメータ） | `APath`, `ACallback`, `ARequest` |

### 3.3 プログラム分類一覧

| 分類 | ファイルパターン | 役割 |
|-----|-----------------|------|
| エントリーポイント | `*.dpr`, `*.lpr` | アプリケーションのメインファイル |
| ユニット | `*.pas` | 機能モジュール |
| フォーム定義 | `*.dfm`, `*.lfm` | GUI画面定義 |
| プロジェクト設定 | `*.dproj`, `*.lpi` | IDE用プロジェクト設定 |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
horse-master/
├── src/                          # フレームワークのソースコード
│   ├── Horse.pas                 # メインユニット（ファサード）
│   ├── Horse.Core.pas            # コアロジック
│   ├── Horse.Request.pas         # リクエスト処理
│   ├── Horse.Response.pas        # レスポンス処理
│   ├── Horse.Core.RouterTree.pas # ルーティング機能
│   ├── Horse.Provider.*.pas      # 各種プロバイダー
│   └── ...
├── samples/                      # サンプルアプリケーション
│   ├── delphi/                   # Delphi用サンプル
│   │   ├── console/              # コンソールアプリケーション
│   │   ├── vcl/                  # VCL GUIアプリケーション
│   │   ├── cgi/                  # CGI
│   │   ├── isapi/                # ISAPI（IIS用）
│   │   ├── apache/               # Apache module
│   │   ├── daemon/               # デーモン
│   │   └── winsvc/               # Windows Service
│   └── lazarus/                  # Lazarus用サンプル
│       ├── console/
│       ├── lcl/
│       └── ...
├── tests/                        # テストコード
│   └── src/
│       ├── tests/                # テストケース
│       └── controllers/          # テスト用コントローラー
├── boss.json                     # Boss（パッケージマネージャー）設定
└── README.md                     # プロジェクト説明
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/` | フレームワーク本体のソースコード | `Horse.pas`, `Horse.Core.pas` |
| `samples/delphi/` | Delphi向けサンプルプロジェクト | `Console.dpr`, `VCL.dpr` |
| `samples/lazarus/` | Lazarus（Free Pascal）向けサンプル | `Console.lpr` |
| `tests/` | ユニットテスト | `Tests.*.pas` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

Horseはミドルウェアベースのルーティングアーキテクチャを採用しています。

```
┌─────────────────────────────────────────────────────────────────────┐
│                        THorse (Facade)                               │
│                          Horse.pas                                   │
└───────────────────────────────┬─────────────────────────────────────┘
                                │
    ┌───────────────────────────┼───────────────────────────────┐
    │                           │                               │
    ▼                           ▼                               ▼
┌─────────────┐       ┌─────────────────┐           ┌──────────────────┐
│ THorseCore  │       │ THorseProvider  │           │ THorseRouterTree │
│ (Singleton) │       │   (Abstract)    │           │   (Routing)      │
└──────┬──────┘       └────────┬────────┘           └────────┬─────────┘
       │                       │                              │
       │              ┌────────┴────────┐                     │
       │              │                 │                     │
       ▼              ▼                 ▼                     ▼
┌──────────────┐ ┌──────────┐ ┌──────────────┐ ┌─────────────────────┐
│THorseRequest │ │ Console  │ │    VCL       │ │ THorseCallback      │
│THorseResponse│ │ Provider │ │  Provider    │ │ (Middleware/Handler)│
└──────────────┘ └──────────┘ └──────────────┘ └─────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| Facade | 外部向けAPI（ユーザーが直接使用） | `Horse.pas` |
| Core | ルーティングとリクエスト処理のコアロジック | `Horse.Core.pas`, `Horse.Core.RouterTree.pas` |
| Request/Response | HTTPリクエスト/レスポンスの抽象化 | `Horse.Request.pas`, `Horse.Response.pas` |
| Provider | 各実行環境用のHTTPサーバー実装 | `Horse.Provider.Console.pas`, `Horse.Provider.VCL.pas` |
| Utilities | 共通機能（MIME型、例外処理等） | `Horse.Commons.pas`, `Horse.Exception.pas` |

### 5.3 データフロー

1. **HTTPリクエスト受信**: Providerが生のHTTPリクエストを受信
2. **リクエストオブジェクト生成**: `THorseRequest`でリクエストをラップ
3. **ルーティング**: `THorseRouterTree`がパスとメソッドからハンドラーを特定
4. **ミドルウェア実行**: 登録されたミドルウェアを順次実行
5. **ハンドラー実行**: 最終的なルートハンドラーを実行
6. **レスポンス送信**: `THorseResponse`の内容をクライアントに返却

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

アプリケーションのエントリーポイントは`.dpr`（Delphi）または`.lpr`（Lazarus）ファイルです。

```pascal
// ファイル: samples/delphi/console/Console.dpr:1-28
program Console;

{$APPTYPE CONSOLE}
{$R *.res}

uses
  Horse,
  System.SysUtils;

begin
  THorse.Get('/ping',
    procedure(Req: THorseRequest; Res: THorseResponse)
    begin
      Res.Send('pong');
    end);

  THorse.Listen(9000,
    procedure
    begin
      Writeln(Format('Server is runing on %s:%d', [THorse.Host, THorse.Port]));
      Readln;
    end);
end.
```

**解説:**
- `THorse.Get`でルートを登録
- `THorse.Listen`でサーバーを起動
- コールバック内でリクエスト処理を実装

### 6.2 ビジネスロジック

ルートハンドラー（コールバック）がビジネスロジックの実装場所です。

```pascal
// ファイル: src/Horse.Callback.pas:33-36
THorseCallbackRequest = reference to procedure(AReq: THorseRequest);
THorseCallbackResponse = reference to procedure(ARes: THorseResponse);
THorseCallbackRequestResponse = reference to procedure(AReq: THorseRequest; ARes: THorseResponse);
THorseCallback = reference to procedure(AReq: THorseRequest; ARes: THorseResponse; ANext: TNextProc);
```

**コールバック型:**
- `THorseCallbackRequestResponse` - リクエストとレスポンス両方を受け取る（最も一般的）
- `THorseCallback` - ミドルウェア用（Next関数を呼び出して次の処理へ）

### 6.3 データアクセス

Horseはデータアクセス層を持たないフレームワークですが、リクエストパラメータへのアクセスは`THorseCoreParam`で行います。

```pascal
// ファイル: src/Horse.Core.Param.pas:25-54
THorseCoreParam = class
private
  FParams: THorseList;
  FFiles: TDictionary<String, TStream>;
  FFields: TDictionary<string, THorseCoreParamField>;
public
  function Field(const AKey: string): THorseCoreParamField;
  function ContainsKey(const AKey: string): Boolean;
  function TryGetValue(const AKey: string; var AValue: string): Boolean;
  property Items[const AKey: string]: string read GetItem; default;
  property Dictionary: THorseList read GetDictionary;
end;
```

**使用例:**
- `Req.Query['name']` - クエリパラメータの取得
- `Req.Params['id']` - パスパラメータの取得
- `Req.Body` - リクエストボディの取得

### 6.4 ユーティリティ/共通機能

#### HTTPステータスコード

```pascal
// ファイル: src/Horse.Commons.pas:24-87
THTTPStatus = (
  Continue = 100,
  OK = 200,
  Created = 201,
  NoContent = 204,
  BadRequest = 400,
  Unauthorized = 401,
  Forbidden = 403,
  NotFound = 404,
  InternalServerError = 500,
  // ... その他多数
);
```

#### MIMEタイプ

```pascal
// ファイル: src/Horse.Commons.pas:89-106
TMimeTypes = (
  MultiPartFormData,
  ApplicationJSON,
  ApplicationXML,
  TextPlain,
  TextHTML,
  // ... その他
);
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| シングルトン | クラスのインスタンスを1つに制限 | 高 | `Horse.Core.pas` |
| Fluent Interface | メソッドチェーンを可能にする設計 | 高 | `Horse.Response.pas` |
| コールバック | 匿名手続きによるイベント処理 | 高 | `Horse.Callback.pas` |
| ファクトリーメソッド | 条件に応じたオブジェクト生成 | 中 | `Horse.pas` |
| 条件付きコンパイル | コンパイラ/環境に応じたコード分岐 | 高 | すべてのファイル |

### 各パターンの詳細

#### パターン1: シングルトン

**目的:** アプリケーション全体で単一のルーターインスタンスを共有する

```pascal
// ファイル: src/Horse.Core.pas:169-174
class function THorseCore.GetInstance: THorseCore;
begin
  if FDefaultHorse = nil then
    FDefaultHorse := THorseCore.Create;
  Result := FDefaultHorse;
end;
```

**解説:** `GetInstance`は常に同じ`THorseCore`インスタンスを返します。これにより、どこからルートを登録しても同じルーターに追加されます。

#### パターン2: Fluent Interface

**目的:** メソッドを連鎖的に呼び出せるようにし、可読性を向上させる

```pascal
// ファイル: src/Horse.Response.pas:107-111
function THorseResponse.Send(const AContent: string): THorseResponse;
begin
  FWebResponse.Content := AContent;
  Result := Self;
end;
```

**解説:** メソッドが`Self`を返すことで、`Res.Status(200).ContentType('application/json').Send('{}')`のような記述が可能になります。

#### パターン3: コールバック

**目的:** 処理の流れを柔軟にカスタマイズ可能にする

```pascal
// ファイル: src/Horse.Callback.pas:36
THorseCallback = reference to procedure(AReq: THorseRequest; ARes: THorseResponse; ANext: TNextProc);
```

**解説:** `reference to procedure`は匿名手続き型を定義します。これにより、ルートハンドラーやミドルウェアを関数引数として渡せます。

#### パターン4: 条件付きコンパイル

**目的:** DelphiとFree Pascal、異なる実行環境に対応

```pascal
// ファイル: src/Horse.pas:69-112
{$IF DEFINED(HORSE_ISAPI)}
  THorseProvider = Horse.Provider.ISAPI.THorseProvider;
{$ELSEIF DEFINED(HORSE_VCL)}
  THorseProvider = Horse.Provider.VCL.THorseProvider;
{$ELSE}
  THorseProvider = Horse.Provider.Console.THorseProvider;
{$ENDIF}
```

**解説:** コンパイル時のシンボル定義により、適切なプロバイダーが選択されます。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: HTTPリクエストの処理フロー

**概要:** GETリクエスト `/ping` が処理される流れを追跡

**処理フロー:**
```
Listen開始 → リクエスト受信 → RouterTree.Execute → Callback実行 → レスポンス送信
```

**詳細な追跡:**

1. **サーバー起動** (`src/Horse.Provider.Console.pas:174-238`)
   ```pascal
   class procedure THorseProvider.InternalListen;
   begin
     LIdHTTPWebBrokerBridge := GetDefaultHTTPWebBroker;
     LIdHTTPWebBrokerBridge.Active := True;
     LIdHTTPWebBrokerBridge.StartListening;
     FRunning := True;
   end;
   ```

2. **ルート登録** (`src/Horse.Core.pas:300-304`)
   ```pascal
   class function THorseCore.Get(const APath: string; const ACallback: THorseCallback): THorseCore;
   begin
     Result := RegisterCallbacksRoute(mtGet, APath);
     RegisterRoute(mtGet, APath, ACallback);
   end;
   ```

3. **リクエスト受信とルーティング** (`src/Horse.Core.RouterTree.pas:139-166`)
   ```pascal
   function THorseRouterTree.Execute(const ARequest: THorseRequest; const AResponse: THorseResponse): Boolean;
   var
     LQueue: TQueue<string>;
   begin
     LPathInfo := ARequest.RawWebRequest.RawPathInfo;
     LQueue := GetQueuePath(LPathInfo, False);
     Result := ExecuteInternal(LQueue, LMethodType, ARequest, AResponse);
   end;
   ```

4. **コールバック実行** (`src/Horse.Core.RouterTree.NextCaller.pas` 内)
   - ミドルウェアチェーンを順次実行
   - 最終的にルートハンドラーを呼び出し

5. **レスポンス送信** (`src/Horse.Response.pas:107-111`)
   ```pascal
   function THorseResponse.Send(const AContent: string): THorseResponse;
   begin
     FWebResponse.Content := AContent;
     Result := Self;
   end;
   ```

### 8.3 フロー追跡チェックリスト

- [x] エントリーポイントを特定したか
- [x] 呼び出し関係を把握したか
- [x] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [x] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. README.md
2. アーキテクチャ設計書
3. 機能一覧

#### 特定機能を理解したい場合
1. 機能一覧
2. 機能設計書（対象機能）
3. 関連ソースコード（`src/`ディレクトリ）

#### 改修作業を行う場合
1. 対象機能の機能設計書
2. API設計書（ルーティング関連の場合）
3. 単体テストケース一覧

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| README.md | プロジェクト概要、クイックスタート | 最初に参照 |
| アーキテクチャ設計書 | 全体構造の理解 | 初期学習時 |
| 機能一覧 | 全機能の一覧と概要 | 機能把握時 |
| 機能設計書 | 各機能の詳細仕様 | 実装/改修時 |
| API設計書 | ルーティングとハンドラーの仕様 | API実装時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `{$IF DEFINED(FPC)}`とは何ですか？
A: Free Pascal Compiler（FPC）でコンパイルする場合のみ有効なコードブロックを定義する条件付きコンパイルディレクティブです。Delphiでは`{$ELSE}`以降のコードが使用されます。

#### Q: `class function`と`function`の違いは？
A: `class function`はインスタンスなしで呼び出せる静的メソッドです。`THorse.Get(...)`のようにクラス名から直接呼び出します。

#### Q: `Result := Self`の意味は？
A: Object Pascalでは`Result`変数に戻り値を代入します。`Self`は現在のインスタンスを指し、これを返すことでFluent Interfaceパターンを実現します。

#### Q: `THorseList`とは何ですか？
A: `TDictionary<string, string>`のエイリアス（別名）です。キーと値が両方文字列の辞書型です（`src/Horse.Core.Param.pas:23`）。

### デバッグのヒント

1. **ルーティングの問題**
   - `Horse.Core.RouterTree.pas`の`Execute`メソッドにブレークポイントを設定
   - パスのキュー化（`GetQueuePath`）を確認

2. **コールバックが呼ばれない**
   - ルート登録時のHTTPメソッドを確認
   - パスの正規化（`TrimPath`）を確認

3. **プロバイダーの問題**
   - コンパイルシンボル（`HORSE_VCL`等）の設定を確認
   - プロバイダーの`InternalListen`を追跡

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Unit | Object Pascalのモジュール単位。1ファイル=1ユニット |
| Provider | HTTPサーバーの実装を抽象化したクラス |
| Callback | リクエスト処理を行う匿名手続き |
| Middleware | リクエスト処理の前後に実行される共通処理 |
| Router Tree | パスをツリー構造で管理するルーティング機構 |
| FPC | Free Pascal Compiler（オープンソースのPascalコンパイラ） |
| Lazarus | FPCベースのIDEでDelphiのオープンソース版に相当 |
| Boss | Delphiのパッケージマネージャー |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `src/Horse.pas` | メインファサード | 外部公開API、型エイリアス定義 |
| `src/Horse.Core.pas` | コアロジック | シングルトン、ルート登録、メソッド定義 |
| `src/Horse.Request.pas` | リクエスト処理 | クエリ、パラメータ、ボディの取得 |
| `src/Horse.Response.pas` | レスポンス処理 | Send、Status、ファイル送信 |
| `src/Horse.Core.RouterTree.pas` | ルーティング | パスマッチング、コールバック実行 |
| `src/Horse.Provider.*.pas` | プロバイダー | 各実行環境用のHTTPサーバー実装 |
| `src/Horse.Commons.pas` | 共通定義 | HTTPステータス、MIMEタイプ |
| `src/Horse.Exception.pas` | 例外処理 | カスタム例外クラス |
| `src/Horse.Callback.pas` | コールバック定義 | コールバック型の定義 |

### C. 参考資料

- [Delphi公式ドキュメント](https://docwiki.embarcadero.com/)
- [Free Pascal公式ドキュメント](https://www.freepascal.org/docs.html)
- [Horse GitHub リポジトリ](https://github.com/HashLoad/horse)
- [Boss パッケージマネージャー](https://github.com/HashLoad/boss)
- [Express.js（Horseの設計インスピレーション元）](https://expressjs.com/)
