# セキュリティ設計書

## 概要

本ドキュメントは、Horse Webフレームワークにおけるセキュリティアーキテクチャとセキュリティ機能の設計について記述する。Horseは Delphi/Lazarus 向けの軽量Webフレームワークであり、セキュリティ機能は主にミドルウェアを通じた拡張ポイントとして提供される設計となっている。

## 認証設計

### 認証方式

Horseフレームワーク本体には認証機構は組み込まれていない。認証機能はミドルウェアを通じて提供される設計であり、以下の公式/サードパーティミドルウェアによる実装が推奨される。

**公式認証ミドルウェア**:
- `horse/basic-auth`: HTTP Basic認証
- `horse/jwt`: JWT（JSON Web Token）認証

**サードパーティ認証ミドルウェア**:
- `andre-djsystem/horse-bearer-auth`: Bearer認証

Providerレベルでは `OnParseAuthentication` イベントが実装されており、認証ヘッダーの解析フックポイントが提供されている（`Horse.Provider.Console.pas` / `Horse.Provider.VCL.pas`）。

```pascal
class procedure THorseProvider.OnAuthentication(AContext: TIdContext;
  const AAuthType, AAuthData: String;
  var VUsername, VPassword: String;
  var VHandled: Boolean);
begin
  VHandled := True;
end;
```

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | 利用者側で設定 | フレームワークとしてはセッションストアを提供するのみ |
| セッション固定化対策 | 未実装 | ミドルウェアによる実装が必要 |

Horseは `THorseSessions` クラスによるセッションストア機構を提供している（`Horse.Session.pas`）。セッションクラスの型安全な管理が可能であり、`TObjectDictionary<TSessionClass, TSession>` による管理を行う。

```pascal
THorseSessions = class
  private
    FSessions: TObjectDictionary<TSessionClass, TSession>;
  public
    function TryGetSession<T: class>(out ASession: T): Boolean;
    function Contains(const ASessionClass: TSessionClass): Boolean;
    function SetSession(const ASessionClass: TSessionClass;
      const AInstance: TSession): THorseSessions;
end;
```

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| 未定義 | N/A | フレームワークレベルでは権限体系は定義されていない |

Horseフレームワークは認可機能を内蔵していない。アプリケーション開発者がミドルウェアまたはルートハンドラ内で独自に実装する必要がある。

### アクセス制御

アクセス制御はミドルウェアチェーンを通じて実装される設計となっている。`THorseCore.Use` メソッドによりミドルウェアを登録し、リクエスト処理パイプラインにアクセス制御ロジックを組み込むことができる。

```pascal
class function THorseCore.Use(const ACallback: THorseCallback): THorseCore;
begin
  Result := GetInstance;
  Result.Routes.RegisterMiddleware('/', ACallback);
end;
```

ルーティングは `THorseRouterTree` によるパターンマッチングで行われ、パスパラメータ（`:param`）や正規表現パターンをサポートする。

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | SSL/TLS対応（OpenSSL経由） |
| HSTS | 未実装（アプリケーション側でヘッダー設定が必要） |

### SSL/TLS設定

`Horse.Provider.IOHandleSSL.pas` により SSL/TLS 設定が提供されている。

```pascal
THorseProviderIOHandleSSL = class(TInterfacedObject, IHorseProviderIOHandleSSL)
private
  FKeyFile: string;          // 秘密鍵ファイル
  FRootCertFile: string;     // ルート証明書ファイル
  FCertFile: string;         // サーバー証明書ファイル
  FDHParamsFile: string;     // DHパラメータファイル
  FCipherList: string;       // 暗号スイート設定
  FMethod: TIdSSLVersion;    // SSL/TLSバージョン
  FSSLVersions: TIdSSLVersions;
  FOnGetPassword: TPasswordEvent;
```

**対応SSL/TLSバージョン**: OpenSSLライブラリがサポートするバージョン（`TIdSSLVersion` / `TIdSSLVersions`）

**設定項目**:
- 証明書ファイル（CertFile）
- 秘密鍵ファイル（KeyFile）
- ルート証明書（RootCertFile）
- DHパラメータファイル（DHParamsFile）
- 暗号スイート（CipherList）
- パスワードコールバック（OnGetPassword）

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | TLS/SSL（OpenSSL経由、暗号スイート設定可能） |
| 保存データ | 未実装（アプリケーション側で実装が必要） |

### 機密情報管理

フレームワークレベルでの機密情報管理機能は提供されていない。アプリケーション開発者が環境変数や外部設定ファイルを通じて管理することが推奨される。

`Horse.EnvironmentVariables.pas` が存在するが、基本的な環境変数アクセス機能のみを提供する。

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | 未実装（フレームワークレベルでの自動エスケープなし） |
| SQLインジェクション | 未実装（データベースアクセスはスコープ外） |
| CSRF | 未実装（`horse/cors` ミドルウェアによるCORS設定は利用可能） |

### 入力バリデーション

`Horse.Core.Param.Field.pas` により型安全なパラメータ取得と基本的なバリデーションが提供されている。

**サポートされる型変換**:
- `AsString`: 文字列として取得
- `AsInteger` / `AsInt64`: 整数として取得
- `AsFloat` / `AsExtended` / `AsCurrency`: 数値として取得
- `AsBoolean`: 真偽値として取得
- `AsDate` / `AsTime` / `AsDateTime` / `AsISO8601DateTime`: 日時として取得
- `AsStream`: ファイルストリームとして取得
- `AsList<T>`: リストとして取得

**バリデーション機能**:
- `Required`: 必須チェック
- `RequiredMessage`: 必須エラーメッセージのカスタマイズ
- `InvalidFormatMessage`: 不正フォーマットエラーメッセージのカスタマイズ
- `DateFormat` / `TimeFormat`: 日時フォーマット指定

```pascal
function THorseCoreParamField.RaiseHorseException(const AMessage: string);
var
  LException: EHorseException;
begin
  LException := EHorseException.New.Status(THTTPStatus.BadRequest).Error(AMessage);
  LException.Message := AMessage;
  raise LException;
end;
```

型変換エラーが発生した場合、`EHorseException` が `THTTPStatus.BadRequest`（400）ステータスで発生し、エラー情報がJSON形式でレスポンスされる。

### エラーハンドリング

`Horse.Exception.pas` により構造化されたエラーレスポンスが提供されている。

```pascal
EHorseException = class(Exception)
  FError: string;      // エラーメッセージ
  FStatus: THTTPStatus; // HTTPステータスコード
  FType: TMessageType; // メッセージタイプ
  FTitle: string;      // タイトル
  FCode: Integer;      // アプリケーションエラーコード
  FHint: string;       // ヒント情報
  FUnit: string;       // 発生ユニット
  FDetail: string;     // 詳細情報
```

**注意**: デフォルトでは詳細なエラー情報（Unit名など）がレスポンスに含まれる可能性があり、本番環境では情報漏洩リスクがある。`horse/exception` ミドルウェアによる適切なエラーハンドリングが推奨される。

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| アクセスログ | 未実装（`horse/logger` ミドルウェアで対応可能） | N/A |
| エラーログ | 未実装（`horse/exception` / `arvanus/horse-exception-logger` で対応可能） | N/A |
| 認証ログ | 未実装 | N/A |

フレームワーク本体にはログ機能は実装されていない。以下のミドルウェアによる対応が推奨される:

- `horse/logger`: HTTPアクセスログ
- `arvanus/horse-exception-logger`: 例外ログ
- `dliocode/horse-datalogger`: データログ

## 備考

### ミドルウェアによるセキュリティ拡張

Horseはミニマリストな設計思想に基づき、セキュリティ機能の多くはミドルウェアを通じて提供される。以下の公式ミドルウェアがセキュリティに関連する。

| ミドルウェア | 用途 |
| --- | --- |
| horse/basic-auth | HTTP Basic認証 |
| horse/jwt | JWT認証 |
| horse/cors | CORS設定 |
| horse/exception | 例外ハンドリング |
| horse/logger | アクセスログ |

### セキュリティ上の推奨事項

1. **本番環境では必ずHTTPSを使用すること**: `IOHandleSSL` プロパティを適切に設定
2. **認証ミドルウェアを導入すること**: `horse/basic-auth` または `horse/jwt` を使用
3. **CORSを適切に設定すること**: `horse/cors` ミドルウェアを使用
4. **例外ハンドリングを導入すること**: `horse/exception` ミドルウェアにより詳細エラー情報の漏洩を防止
5. **レートリミットの導入を検討すること**: `dliocode/horse-ratelimit` や `dliocode/horse-slowdown` を使用
6. **入力バリデーションを適切に実装すること**: `THorseCoreParam.Field` による型チェックとバリデーションを活用
7. **セッション管理はアプリケーション層で実装すること**: `THorseSessions` はストレージのみ提供

### フレームワークとしての制限事項

- CSRFトークン生成/検証機能は未実装
- XSS対策（出力エスケープ）は未実装
- セキュリティヘッダー（CSP、X-Frame-Options等）の自動設定は未実装
- 監査ログ機能は未実装
- 暗号化ユーティリティは未実装

これらの機能が必要な場合は、アプリケーション側での実装またはサードパーティミドルウェアの導入が必要となる。
