# 機能設計書 10-リクエストボディ取得

## 概要

本ドキュメントは、Horseフレームワークにおける HTTP リクエストのボディデータを取得する機能の設計を記述する。

### 本機能の処理概要

HTTPリクエストに含まれるボディ（本文）データを取得する機能を提供する。文字列形式での生データ取得と、オブジェクト形式での型付きデータ取得の両方をサポートする。

**業務上の目的・背景**：WebアプリケーションやRESTful APIにおいて、POST、PUT、PATCHリクエストではボディにデータを含めて送信することが一般的である。リクエストボディ取得機能は、クライアントから送信されたJSON、フォームデータ、その他のコンテンツを取得し、ビジネスロジックで利用できるようにする基盤を提供する。これにより、ユーザー登録情報、更新データ、検索条件などをサーバー側で受け取り処理できる。

**機能の利用シーン**：
- JSONデータの受信と解析
- フォームデータの取得
- ファイルアップロードのデータ取得
- GraphQLクエリの受信
- WebhookペイロードのI処理

**主要な処理内容**：
1. `THorseRequest.Body` メソッド（引数なし）を呼び出し、生のリクエストボディ文字列を取得
2. `THorseRequest.Body<T>` メソッド（ジェネリック版）でオブジェクト形式のボディを取得
3. `THorseRequest.Body(ABody: TObject)` メソッドでボディオブジェクトを設定
4. 内部的には `FWebRequest.Content` プロパティからデータを取得

**関連システム・外部連携**：本機能は単独で動作するリクエスト処理機能であり、外部システムとの直接的な連携はない。ただし、取得したボディデータをJSON解析ライブラリ等で処理することは一般的である。

**権限による制御**：本機能自体には権限制御は含まれない。リクエストボディの内容に基づく権限チェックはアプリケーション層で実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 7 | Webモジュール | 主機能 | HandlerActionでTHorseRequestを生成しリクエストボディを処理 |
| 8 | Webモジュール | 主機能 | DoOnRequestでTHorseRequestを生成しリクエストボディを処理 |

## 機能種別

リクエスト処理 / データ取得

## 入力仕様

### 入力パラメータ（Body設定時）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ABody | TObject | Yes | 設定するボディオブジェクト | - |

### Bodyメソッドのオーバーロード

| シグネチャ | 説明 |
|-----------|------|
| `Body: string` | 生のリクエストボディ文字列を取得 |
| `Body<T>: T` | 設定されたボディオブジェクトを型Tとして取得 |
| `Body(ABody: TObject): THorseRequest` | ボディオブジェクトを設定（メソッドチェーン用） |

### 入力データソース

- HTTPリクエストのContent（本文）
- ミドルウェア等で設定されたボディオブジェクト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result（文字列版） | string | リクエストボディの生文字列（Content） |
| Result（ジェネリック版） | T | 設定されたボディオブジェクト（キャスト） |
| Result（設定版） | THorseRequest | メソッドチェーン用のSelf |

### 出力先

呼び出し元（コールバック関数内）

## 処理フロー

### 処理シーケンス

```
1. Body: string（文字列取得版）
   └─ FWebRequest.Content を返却

2. Body<T>: T（ジェネリック版）
   └─ FBody を型Tにキャストして返却

3. Body(ABody: TObject): THorseRequest（設定版）
   └─ 既存のFBodyがあれば解放
   └─ FBody := ABody で設定
   └─ Self を返却（メソッドチェーン用）
```

### フローチャート

```mermaid
flowchart TD
    A[Body メソッド呼び出し] --> B{呼び出し形式}
    B -->|引数なし<br/>string取得| C[FWebRequest.Content を返却]
    B -->|ジェネリック<br/>Body<T>| D[T(FBody) を返却]
    B -->|オブジェクト設定<br/>Body(ABody)| E{既存FBodyあり?}
    E -->|Yes| F[FBody.Free]
    E -->|No| G[FBody := ABody]
    F --> G
    G --> H[Self を返却]
    C --> I[終了]
    D --> I
    H --> I
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 生データ取得 | Bodyメソッド（引数なし）はFWebRequest.Contentを返す | 文字列形式のボディ取得時 |
| BR-02 | オブジェクト解放 | ボディオブジェクト設定時、既存オブジェクトは解放される | Body(ABody)呼び出し時 |
| BR-03 | 型キャスト | Body<T>は設定されたオブジェクトを指定型にキャストする | ジェネリック版Body呼び出し時 |
| BR-04 | 自動解放 | THorseRequestのデストラクタでFBodyが解放される | THorseRequest破棄時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 型キャストエラー | Body<T>で不正な型を指定 | 正しい型を指定するか、事前にオブジェクトが設定されているか確認 |

### リトライ仕様

リトライは不要（同期的なデータ取得処理）

## トランザクション仕様

トランザクション管理は不要（メモリ上のデータ取得のみ）

## パフォーマンス要件

- ボディ取得はリクエスト毎に発生するため、効率的に動作
- 大容量ボディの場合はメモリ使用量に注意

## セキュリティ考慮事項

- 受信したボディデータは信頼できないため、必ず検証すること
- JSONインジェクション等の攻撃に注意
- 大容量ボディによるDoS攻撃を防ぐため、Content-Lengthの制限を検討

## 備考

- Body<T>はミドルウェアでJSONデシリアライズしたオブジェクトを取得する際に使用
- 一般的なパターン：ミドルウェアでJSON→オブジェクト変換し、Body(obj)で設定、ルートコールバックでBody<TMyClass>で取得
- FPC版ではBody<T>のclass制約がない点に注意

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

THorseRequestクラスのボディ関連フィールドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Request.pas | `src/Horse.Request.pas` | FBody: TObject フィールドの定義を確認 |
| 1-2 | Horse.Request.pas | `src/Horse.Request.pas` | FWebRequest フィールド（プラットフォーム依存）を確認 |

**読解のコツ**: FWebRequestはDelphiではTWebRequest、FPCではTRequestとなる条件コンパイルに注意。

#### Step 2: エントリーポイントを理解する

`THorseRequest.Body` メソッドがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Request.pas | `src/Horse.Request.pas` | THorseRequestクラスのBodyメソッド群を確認 |

**主要処理フロー**:
1. **45-47行目**: Body メソッドの宣言（3種類のオーバーロード）
2. **75-78行目**: Body: string 実装（FWebRequest.Content を返却）
3. **80-86行目**: Body(ABody: TObject) 実装（オブジェクト設定）
4. **88-91行目**: Body<T>: T 実装（型付きオブジェクト取得）

#### Step 3: デストラクタを理解する

ボディオブジェクトの解放処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Request.pas | `src/Horse.Request.pas` | Destroyデストラクタ |

**主要処理フロー**:
- **118-135行目**: Destroy デストラクタ実装
- **130-131行目**: FBodyの解放処理
```pascal
if Assigned(FBody) then
  FBody.Free;
```

#### Step 4: リクエスト生成を理解する

THorseRequestの生成フローを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Request.pas | `src/Horse.Request.pas` | Createコンストラクタ |
| 4-2 | Horse.WebModule.pas | `src/Horse.WebModule.pas` | THorseRequestの生成箇所 |

**主要処理フロー**:
- **112-116行目（Horse.Request.pas）**: Create コンストラクタ
```pascal
constructor THorseRequest.Create(const AWebRequest: ...);
begin
  FWebRequest := AWebRequest;
  FSessions := THorseSessions.Create;
end;
```

### プログラム呼び出し階層図

```
THorseRequest.Body（引数なし、string版）
    │
    └─ return FWebRequest.Content

THorseRequest.Body<T>（ジェネリック版）
    │
    └─ return T(FBody)

THorseRequest.Body(ABody: TObject)（設定版）
    │
    ├─ if Assigned(FBody) then
    │      └─ FBody.Free
    │
    ├─ FBody := ABody
    │
    └─ return Self

--- THorseRequest破棄時 ---

THorseRequest.Destroy
    │
    ├─ (他のフィールド解放)
    │
    └─ if Assigned(FBody) then
           └─ FBody.Free
```

### データフロー図

```
[HTTPリクエスト]
クライアント ──────► HTTPサーバー ──────► TWebRequest/TRequest
                                             │
                                             ▼
                                        FWebRequest.Content
                                        (リクエストボディ生データ)
                                             │
                                             ▼
                                    THorseRequest.Body: string
                                        (生データ取得)

[ミドルウェアでのオブジェクト設定パターン]
THorseRequest.Body: string ──► JSONデシリアライズ ──► TMyClass インスタンス
                                                          │
                                                          ▼
                                              THorseRequest.Body(obj)
                                              (FBody := obj)
                                                          │
                                                          ▼
[ルートコールバックでの取得]                            FBody
THorseRequest.Body<TMyClass> ◄────────────────── T(FBody)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | THorseRequestクラス - Bodyメソッドの実装 |
| Horse.WebModule.pas | `src/Horse.WebModule.pas` | ソース | THorseRequestの生成とリクエスト処理 |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponseクラス - レスポンス処理のラッパー |
| Horse.Callback.pas | `src/Horse.Callback.pas` | ソース | コールバック型定義（THorseCallback等） |
| Horse.Session.pas | `src/Horse.Session.pas` | ソース | THorseSessionsクラス - セッション管理 |
