# 機能設計書 16-セッション管理

## 概要

本ドキュメントは、Horse Webフレームワークにおけるリクエストに紐づくセッションデータを管理する機能の設計を記述する。THorseRequest.Session/Sessionsメソッドによる実装を対象とする。

### 本機能の処理概要

この機能は、HTTPリクエストにセッションオブジェクトを紐づけて管理する。ミドルウェア間でのデータ共有や、認証状態の保持に使用される。

**業務上の目的・背景**：ステートレスなHTTPプロトコルにおいて、リクエスト間で状態を保持する必要がある場面は多い。この機能により、認証済みユーザー情報、一時的なデータ、ミドルウェアで生成した情報などをリクエストコンテキストとして保持できる。JWT認証のデコード結果、データベース接続、トランザクションコンテキストなど、ミドルウェアチェーン全体でアクセス可能なデータストアとして機能する。

**機能の利用シーン**：JWT/OAuth認証ミドルウェアでデコードしたユーザー情報の格納、データベーストランザクションの伝播、リクエストスコープの依存性注入、ロギングミドルウェアでのコンテキスト情報共有など。

**主要な処理内容**：
1. 単一セッションの管理（Session<T>/Session(ASession)）
2. 複数セッションの型別管理（Sessions.SetSession/Session[]）
3. セッションの存在確認（Sessions.Contains）
4. 型安全なセッション取得（Sessions.TryGetSession<T>）

**関連システム・外部連携**：本機能はフレームワーク内部で完結する。外部のセッションストア（Redis、データベース等）との連携は、ミドルウェア側で実装する。

**権限による制御**：本機能自体はセッションの格納・取得のみを行う。認証・認可のロジックはアプリケーション側のミドルウェアで実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はAPI機能であり、直接関連する画面はない |

## 機能種別

データ管理処理 / セッション管理 / 依存性注入

## 入力仕様

### 入力パラメータ

#### 単一セッション（Session）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ASession | TObject | Yes | 格納するセッションオブジェクト | なし |

#### 複数セッション（Sessions）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ASessionClass | TSessionClass | Yes | セッションのクラス型 | TSessionを継承している必要がある |
| AInstance | TSession | Yes | 格納するセッションインスタンス | ASessionClassを継承している必要がある |

### 入力データソース

ミドルウェアまたはハンドラ内で生成されたオブジェクト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Session<T> | T | 格納された単一セッションオブジェクト（型キャスト済み） |
| Sessions.Session[class] | TSession | クラス型で取得するセッションオブジェクト |
| Sessions.Contains(class) | Boolean | 指定クラスのセッションが存在するか |

### 出力先

呼び出し元のコールバック関数に返却

## 処理フロー

### 処理シーケンス

```
[単一セッションの使用]
1. ミドルウェアでセッションオブジェクトを生成
2. Req.Session(ASession) でセッションを格納
3. ハンドラまたは後続ミドルウェアで Req.Session<T> で取得

[複数セッションの使用]
1. ミドルウェアでセッションオブジェクトを生成
2. Req.Sessions.SetSession(TMySession, MyInstance) でセッションを登録
3. ハンドラで Req.Sessions.Session[TMySession] で取得
   または Req.Sessions.TryGetSession<TMySession>(out Session) で安全に取得
```

### フローチャート

```mermaid
flowchart TD
    subgraph 単一セッション
    A1[ミドルウェア] --> B1[Session オブジェクト生成]
    B1 --> C1["Req.Session(obj)"]
    C1 --> D1[ハンドラ]
    D1 --> E1["Req.Session<T>"]
    end

    subgraph 複数セッション
    A2[ミドルウェア] --> B2[TSession 継承クラス生成]
    B2 --> C2["Sessions.SetSession(Class, Instance)"]
    C2 --> D2[ハンドラ]
    D2 --> E2{存在確認?}
    E2 -->|Yes| F2["Sessions.Session[Class]"]
    E2 -->|安全取得| G2["TryGetSession<T>"]
    end
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-16-01 | 単一セッション上書き | Session(ASession)を複数回呼ぶと前のセッションは解放される | Session設定時 |
| BR-16-02 | クラス型一意性 | Sessionsでは同一クラス型のセッションは1つのみ保持 | SetSession呼び出し時 |
| BR-16-03 | 継承チェック | SetSessionはインスタンスがSessionClassを継承しているかチェック | SetSession呼び出し時 |
| BR-16-04 | 自動解放 | Sessionsに登録されたセッションはTHorseRequest破棄時に自動解放 | doOwnsValues指定 |
| BR-16-05 | 型安全 | Session<T>とSessions.Session[Class]は型キャスト済みで返却 | セッション取得時 |

### 計算ロジック

継承チェックのロジック：
```pascal
if not ASessionClass.InheritsFrom(AInstance.ClassType) then
  raise Exception.CreateFmt('SessionClass differs from of instance[%s].', [AInstance.ClassType.ClassName]);
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | SetSessionでインスタンスがSessionClassを継承していない | 正しいクラス型を指定 |
| - | EListError | 存在しないクラスでSessions.Session[Class]にアクセス | Containsで存在確認、またはTryGetSession使用 |

### リトライ仕様

リトライ処理は不要（即座に結果を返す同期処理）

## トランザクション仕様

トランザクション管理は不要（メモリ内処理のみ）

## パフォーマンス要件

- セッション登録・取得は O(1) の計算量（Dictionary使用）
- doOwnsValuesにより自動メモリ管理

## セキュリティ考慮事項

- セッションデータはサーバーサイドのメモリに保持される
- 機密情報を格納する場合は、リクエスト終了時に適切にクリアすることを推奨
- セッションIDとセッションデータの関連付けは、ミドルウェアで実装する必要がある

## 備考

本機能は「リクエストスコープの依存性注入」として使用できる。ミドルウェアで生成したオブジェクト（認証ユーザー、DBコネクション等）をハンドラで使用するパターンに適している。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

セッション管理のデータ構造の理解が最初のステップである。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Session.pas | `src/Horse.Session.pas` | TSession基底クラス、TSessionClass型、THorseSessionsクラス |

**読解のコツ**: THorseSessionsは`TObjectDictionary<TSessionClass, TSession>`を使用。doOwnsValuesにより値の自動解放が行われる。

#### Step 2: 単一セッション管理を理解する

THorseRequestの単一セッション機能を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Request.pas | `src/Horse.Request.pas` | Session<T>、Session(ASession) メソッド |

**主要処理フロー**:
1. **319-323行目**: Session(ASession) - FSessionにオブジェクトを格納
2. **325-328行目**: Session<T> - FSessionを型キャストして返却

#### Step 3: 複数セッション管理を理解する

THorseSessionsクラスの機能を詳細に理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Session.pas | `src/Horse.Session.pas` | SetSession、GetSession、Contains、TryGetSession メソッド |

**主要処理フロー**:
- **46-49行目**: Create - TObjectDictionary生成（doOwnsValues）
- **67-73行目**: SetSession - 継承チェック後にAddOrSetValue
- **75-78行目**: Contains - ContainsKeyで存在確認
- **80-83行目**: TryGetSession<T> - TryGetValueで安全な取得

#### Step 4: リクエストとの統合を理解する

THorseRequestでのセッション初期化と破棄を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Request.pas | `src/Horse.Request.pas` | Create、Destroy、Sessions メソッド |

**主要処理フロー**:
- **112-116行目**: Create - FSessions := THorseSessions.Create
- **118-135行目**: Destroy - FSessions.Free（doOwnsValuesにより内部セッションも解放）
- **330-333行目**: Sessions - FSessionsを返却

### プログラム呼び出し階層図

```
THorseRequest
    │
    ├─ FSession (TObject)
    │      ├─ Session(ASession) : 格納
    │      └─ Session<T> : 型キャスト取得
    │
    └─ FSessions (THorseSessions)
           │
           ├─ SetSession(Class, Instance)
           │      └─ 継承チェック
           │      └─ AddOrSetValue
           │
           ├─ Session[Class]
           │      └─ FSessions.Items[Class]
           │
           ├─ Contains(Class)
           │      └─ ContainsKey
           │
           └─ TryGetSession<T>(out Session)
                  └─ TryGetValue
```

### データフロー図

```
[ミドルウェア]                    [セッション管理]                [ハンドラ]

認証ミドルウェア
  │
  ├─ JWTデコード
  │      │
  │      └─ TUserSession生成 ──▶ Req.Sessions.SetSession
  │                                    │
  │                                    └─ THorseSessions
  │                                          └─ Dictionary[TUserSession]
  │
  └─ Next() ─────────────────────────────────────────▶ ハンドラ
                                                          │
                                               Req.Sessions.Session[TUserSession]
                                                          │
                                                          └─ TUserSession 取得
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | THorseRequest クラス、Session/Sessions メソッド |
| Horse.Session.pas | `src/Horse.Session.pas` | ソース | TSession、TSessionClass、THorseSessions クラス |

### 使用例

```pascal
// セッションクラスの定義
type
  TUserSession = class(TSession)
  public
    UserId: Integer;
    UserName: string;
  end;

// ミドルウェアでのセッション設定
procedure AuthMiddleware(Req: THorseRequest; Res: THorseResponse; Next: TProc);
var
  LUser: TUserSession;
begin
  // JWT検証など
  LUser := TUserSession.Create;
  LUser.UserId := 123;
  LUser.UserName := 'John';
  Req.Sessions.SetSession(TUserSession, LUser);
  Next();
end;

// ハンドラでのセッション取得
procedure GetProfile(Req: THorseRequest; Res: THorseResponse);
var
  LUser: TUserSession;
begin
  if Req.Sessions.TryGetSession<TUserSession>(LUser) then
    Res.Send(Format('User: %s', [LUser.UserName]))
  else
    Res.Status(401).Send('Unauthorized');
end;
```
