# 機能設計書 18-ステータスコード設定

## 概要

本ドキュメントは、Horse WebフレームワークにおけるHTTPステータスコードを設定する機能の設計を記述する。THorseResponse.Statusメソッドによる実装を対象とする。

### 本機能の処理概要

この機能は、HTTPレスポンスのステータスコードを設定する。200 OK、201 Created、400 Bad Request、404 Not Found など、標準的なHTTPステータスコードをレスポンスに設定できる。

**業務上の目的・背景**：HTTPプロトコルにおいて、ステータスコードはクライアントに処理結果を伝える重要な要素である。この機能により、リクエストの成功/失敗、リソースの状態、エラーの種類などを標準的な方法でクライアントに通知できる。REST APIの設計において、適切なステータスコードの返却はAPI品質の基本要件である。

**機能の利用シーン**：リソース作成成功時の201 Created、バリデーションエラー時の400 Bad Request、認証失敗時の401 Unauthorized、リソース未存在時の404 Not Found、サーバーエラー時の500 Internal Server Error など、あらゆるHTTPレスポンスで使用される。

**主要な処理内容**：
1. 整数値またはTHTTPStatus列挙型でステータスコードを受け取る
2. WebResponseのStatusCode（Delphi）またはCode（FPC）プロパティに設定
3. メソッドチェーン用に自身（THorseResponse）を返却

**関連システム・外部連携**：Delphiの標準WebライブラリであるTWebResponse.StatusCode（Delphi）またはTResponse.Code（Free Pascal）にステータスコードを設定する。

**権限による制御**：本機能自体は認証・認可の制御を行わない。ステータスコードの選択は呼び出し元のハンドラの責任である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はAPI機能であり、直接関連する画面はない |

## 機能種別

レスポンス処理 / ステータスコード設定

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AStatus | Integer | Yes | HTTPステータスコード（整数値） | なし |
| AStatus | THTTPStatus | Yes | HTTPステータスコード（列挙型） | なし |

### 入力データソース

ハンドラ内で決定されるステータスコード

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseResponse | メソッドチェーン用の自身への参照 |
| FWebResponse.StatusCode/Code | Integer | 設定されたHTTPステータスコード |

### 出力先

HTTPレスポンスヘッダーのステータスラインに反映

## 処理フロー

### 処理シーケンス

```
1. Status メソッドの呼び出し
   └─ AStatus（Integer または THTTPStatus）を受け取る
2. THTTPStatus の場合は ToInteger で整数に変換
3. FWebResponse のステータスコードプロパティに代入
   └─ Delphi: FWebResponse.StatusCode
   └─ FPC: FWebResponse.Code
4. Self（THorseResponse）を返却
   └─ メソッドチェーンのサポート
```

### フローチャート

```mermaid
flowchart TD
    A[Status メソッド呼び出し] --> B{引数の型}
    B -->|Integer| C[そのまま使用]
    B -->|THTTPStatus| D[ToInteger で変換]
    C --> E[FWebResponse に設定]
    D --> E
    E --> F[Self を返却]
    F --> G{メソッドチェーン?}
    G -->|Yes| H[次のメソッド呼び出し]
    G -->|No| I[ハンドラ終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-18-01 | メソッドチェーン | Statusは自身を返却し、Send等との連鎖呼び出しが可能 | 常時 |
| BR-18-02 | デフォルト値 | 明示的に設定しない場合は200 OKがデフォルト | Create時に設定 |
| BR-18-03 | 複数回呼び出し | Statusを複数回呼ぶと最後の値が使用される | 常時 |
| BR-18-04 | 列挙型変換 | THTTPStatus列挙型はOrd()で整数値に変換 | THTTPStatus使用時 |

### 計算ロジック

THTTPStatus.ToInteger の実装：
```pascal
function THTTPStatusHelper.ToInteger: Integer;
begin
  Result := Ord(Self);
end;
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 例外なし | 通常の使用では例外は発生しない | - |

### リトライ仕様

リトライ処理は不要（即座に結果を返す同期処理）

## トランザクション仕様

トランザクション管理は不要（メモリ内処理のみ）

## パフォーマンス要件

- ステータスコード設定は O(1) の計算量

## セキュリティ考慮事項

- エラー時に詳細すぎる情報を返さないよう注意
- 500系エラーでは内部情報を漏洩しない

## 備考

THTTPStatus列挙型には標準的なHTTPステータスコードが定義されている。整数値で直接指定することも可能だが、可読性のため列挙型の使用を推奨する。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

THTTPStatus列挙型の定義を理解することが最初のステップである。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Commons.pas | `src/Horse.Commons.pas` | THTTPStatus 列挙型の定義、THTTPStatusHelper |

**読解のコツ**: THTTPStatusは{$SCOPEDENUMS ON}で定義されているため、`THTTPStatus.OK`のように完全修飾名で参照する。各値は実際のHTTPステータスコード（100, 200, 404等）に対応している。

#### Step 2: エントリーポイントを理解する

Statusメソッドの実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Response.pas | `src/Horse.Response.pas` | Status(Integer)、Status(THTTPStatus) メソッド |

**主要処理フロー**:
1. **141-145行目**: Status(THTTPStatus) - AStatus.ToInteger()で整数変換してStatusCode/Codeに設定
2. **251-255行目**: Status(Integer) - 直接StatusCode/Codeに設定
3. **246-249行目**: Status() getter - 現在のステータスコードを取得

#### Step 3: THTTPStatusHelperを理解する

列挙型から整数への変換ヘルパーを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Commons.pas | `src/Horse.Commons.pas` | THTTPStatusHelper.ToInteger メソッド |

**主要処理フロー**:
- **249-254行目**: ToInteger - Ord(Self)で列挙型の値を整数に変換

#### Step 4: デフォルト値の設定を理解する

THorseResponseの生成時にデフォルトステータスが設定される処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Response.pas | `src/Horse.Response.pas` | Create コンストラクタ |

**主要処理フロー**:
- **89行目**: THTTPStatus.Ok.ToInteger でデフォルト200を設定

### プログラム呼び出し階層図

```
THorseResponse.Status(AStatus: THTTPStatus)
    │
    ├─ AStatus.ToInteger()
    │      └─ THTTPStatusHelper.ToInteger
    │             └─ Result := Ord(Self)
    │
    ├─ FWebResponse.StatusCode := 200（Delphi）
    │  または
    │  FWebResponse.Code := 200（FPC）
    │
    └─ Result := Self

THorseResponse.Status(AStatus: Integer)
    │
    ├─ FWebResponse.StatusCode := AStatus（Delphi）
    │  または
    │  FWebResponse.Code := AStatus（FPC）
    │
    └─ Result := Self
```

### データフロー図

```
[入力]                      [処理]                         [出力]

THTTPStatus.NotFound
        │
        └─────▶ Status(THTTPStatus)             ──▶ HTTP Response
                     │
                     ├─ ToInteger() → 404
                     │
                     └─ FWebResponse.StatusCode := 404
                             │
                             └─ HTTP/1.1 404 Not Found
                                ...
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponse クラス、Status メソッドの実装 |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | THTTPStatus 列挙型、THTTPStatusHelper |

### THTTPStatus 列挙型の主要な値

```pascal
THTTPStatus = (
  // 1xx Informational
  Continue = 100,
  SwitchingProtocols = 101,

  // 2xx Success
  OK = 200,
  Created = 201,
  Accepted = 202,
  NoContent = 204,

  // 3xx Redirection
  MovedPermanently = 301,
  Found = 302,
  SeeOther = 303,
  NotModified = 304,

  // 4xx Client Error
  BadRequest = 400,
  Unauthorized = 401,
  Forbidden = 403,
  NotFound = 404,
  MethodNotAllowed = 405,
  Conflict = 409,
  UnprocessableEntity = 422,
  TooManyRequests = 429,

  // 5xx Server Error
  InternalServerError = 500,
  NotImplemented = 501,
  BadGateway = 502,
  ServiceUnavailable = 503
);
```

### 使用例

```pascal
// 列挙型を使用（推奨）
Res.Status(THTTPStatus.Created).Send('Resource created');

// 整数値を使用
Res.Status(201).Send('Resource created');

// エラーレスポンス
Res.Status(THTTPStatus.NotFound).Send('Resource not found');

// バリデーションエラー
Res.Status(THTTPStatus.UnprocessableEntity).Send('Validation failed');

// メソッドチェーン
Res
  .Status(THTTPStatus.OK)
  .ContentType('application/json')
  .Send('{"message":"success"}');
```
