# 機能設計書 19-ファイル送信

## 概要

本ドキュメントは、Horse Webフレームワークにおけるファイルをインライン表示用に送信する機能の設計を記述する。THorseResponse.SendFileメソッドによる実装を対象とする。

### 本機能の処理概要

この機能は、サーバー上のファイルをクライアントに送信し、ブラウザでインライン表示（直接表示）させる。画像、PDF、テキストファイルなどの表示に使用される。

**業務上の目的・背景**：Webアプリケーションでは、画像やドキュメントをブラウザ内で直接表示する必要がある場面が多い。この機能により、サーバー上のファイルをストリームとしてクライアントに送信し、ブラウザがファイルの内容をインラインで表示できる。Content-Disposition: inline ヘッダーにより、ダウンロードではなく表示が促される。

**機能の利用シーン**：ユーザーアバター画像の表示、PDFドキュメントのプレビュー、テキストファイルの表示、ログファイルの閲覧、静的アセット（CSS、JS、画像）の配信など。

**主要な処理内容**：
1. ファイルパスまたはストリームを受け取る
2. ファイル拡張子からMIMEタイプを自動判定（指定がない場合）
3. Content-Disposition: inline ヘッダーを設定
4. ファイルコンテンツをレスポンスストリームに設定
5. レスポンスを送信

**関連システム・外部連携**：THorseCoreFileクラスでファイルの読み込みとストリーム化を行い、THorseMimeTypesでMIMEタイプを判定する。

**権限による制御**：本機能自体は認証・認可の制御を行わない。ファイルアクセス権限はアプリケーション側のミドルウェアで実装する必要がある。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はAPI機能であり、直接関連する画面はない |

## 機能種別

レスポンス処理 / ファイル送信 / ストリーミング

## 入力仕様

### 入力パラメータ

#### ストリームからの送信

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AFileStream | TStream | Yes | 送信するファイルストリーム | Position=0にリセットされる |
| AFileName | string | No | ファイル名（Content-Dispositionで使用） | デフォルト: '' |
| AContentType | string | No | MIMEタイプ | デフォルト: 自動判定 |

#### ファイルパスからの送信

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AFileName | string | Yes | ファイルのフルパス | ファイルの存在確認 |
| AContentType | string | No | MIMEタイプ | デフォルト: 自動判定 |

### 入力データソース

サーバー上のファイルシステム、または生成されたストリーム

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseResponse | メソッドチェーン用の自身への参照 |
| Content-Disposition | Header | inline; filename="..." |
| Content-Type | Header | 自動判定または指定されたMIMEタイプ |
| Content-Length | Header | ファイルサイズ |
| ContentStream | TStream | ファイルコンテンツ |

### 出力先

HTTPレスポンスとしてクライアントに送信

## 処理フロー

### 処理シーケンス

```
[ストリームからの送信]
1. SendFile(AFileStream, AFileName, AContentType) の呼び出し
2. AFileStream.Position := 0（ストリーム位置リセット）
3. FreeContentStream := False（外部ストリームのため解放しない）
4. ContentLength := AFileStream.Size
5. ContentStream := AFileStream
6. Content-Disposition: inline; filename="..." を設定
7. Content-Type を設定（指定なしの場合は拡張子から自動判定）
8. SendContent/SendResponse でレスポンス送信

[ファイルパスからの送信]
1. SendFile(AFileName, AContentType) の呼び出し
2. THorseCoreFile.Create(AFileName) でファイルオブジェクト生成
3. FreeContentStream := True（ファイルオブジェクト側で管理）
4. ストリーム版SendFile を呼び出し
5. ファイルオブジェクトを解放
```

### フローチャート

```mermaid
flowchart TD
    A[SendFile 呼び出し] --> B{引数の型}
    B -->|TStream| C[ストリーム版処理]
    B -->|string path| D[ファイルパス版処理]

    C --> E[Position := 0]
    E --> F[FreeContentStream := False]
    F --> G[ContentLength/ContentStream 設定]

    D --> H[THorseCoreFile.Create]
    H --> I[FreeContentStream := True]
    I --> J[ストリーム版SendFile呼び出し]
    J --> K[THorseCoreFile.Free]

    G --> L[Content-Disposition: inline 設定]
    K --> L
    L --> M{ContentType指定あり?}
    M -->|Yes| N[指定値を使用]
    M -->|No| O[拡張子から自動判定]
    N --> P[SendContent/SendResponse]
    O --> P
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-19-01 | インライン表示 | Content-Disposition: inline でブラウザ内表示を指示 | 常時 |
| BR-19-02 | MIMEタイプ自動判定 | ContentTypeが空の場合はファイル拡張子から自動判定 | ContentType未指定時 |
| BR-19-03 | ストリーム位置リセット | 送信前にストリームのPositionを0にリセット | 常時 |
| BR-19-04 | ファイル存在確認 | ファイルパス版は存在確認、なければException | ファイルパス指定時 |
| BR-19-05 | デフォルトMIME | 判定できない場合は application/octet-stream | 拡張子不明時 |

### 計算ロジック

MIMEタイプ判定のロジック：
```pascal
if (AContentType = EmptyStr) then
  FWebResponse.ContentType := THorseMimeTypes.GetFileType(LFileName);
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | ファイルパスが空 | 有効なパスを指定 |
| - | Exception | ファイルが存在しない | ファイルの存在を確認 |

### リトライ仕様

リトライ処理は不要（ファイルI/Oエラーは再送で解決しない）

## トランザクション仕様

トランザクション管理は不要（ファイル読み込みのみ）

## パフォーマンス要件

- 大きなファイルはストリーミング送信（メモリ効率的）
- ContentLengthを設定することでクライアントが進捗を把握可能

## セキュリティ考慮事項

- パストラバーサル攻撃への対策が必要
- アクセス可能なディレクトリを制限することを推奨
- 機密ファイルへのアクセスには認証・認可を実装

## 備考

SendFileはインライン表示用、Downloadはダウンロード用。違いはContent-Dispositionヘッダーの値（inline vs attachment）のみ。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ファイル処理に関連するクラスの理解が最初のステップである。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | THorseCoreFile クラスの構造 |
| 1-2 | Horse.Mime.pas | `src/Horse.Mime.pas` | THorseMimeTypes.GetFileType メソッド |

**読解のコツ**: THorseCoreFileはファイルパスからストリームを生成し、MIMEタイプも自動判定する。FreeContentStreamプロパティで解放責任を制御。

#### Step 2: エントリーポイントを理解する

SendFileメソッドの実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Response.pas | `src/Horse.Response.pas` | SendFile(TStream)、SendFile(string) メソッド |

**主要処理フロー**:
1. **147-169行目**: SendFile(TStream) - ストリームからの送信処理
2. **171-188行目**: SendFile(string) - ファイルパスからの送信処理

#### Step 3: ストリーム版SendFileの詳細を理解する

ストリームからの送信処理を詳細に理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Response.pas | `src/Horse.Response.pas` | SendFile(TStream) の実装詳細 |

**主要処理フロー**:
- **152行目**: AFileStream.Position := 0
- **155-157行目**: FreeContentStream, ContentLength, ContentStream 設定
- **158行目**: Content-Disposition: inline 設定
- **160-162行目**: ContentType設定（自動判定含む）
- **164-168行目**: SendContent/SendResponse でレスポンス送信

#### Step 4: THorseCoreFileを理解する

ファイルオブジェクトの生成と管理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | Create、ContentStream、ContentType メソッド |

**主要処理フロー**:
- **44-56行目**: Create - ファイル存在確認、名前抽出、MIMEタイプ判定
- **70-75行目**: ContentStream - TFileStreamの遅延生成
- **65-68行目**: ContentType - 判定済みMIMEタイプを返却

### プログラム呼び出し階層図

```
THorseResponse.SendFile(AFileName: string)
    │
    ├─ THorseCoreFile.Create(AFileName)
    │      ├─ ファイル存在確認
    │      ├─ ExtractFileName
    │      └─ THorseMimeTypes.GetFileType
    │
    ├─ SendFile(LFile.ContentStream, LFile.Name, LContentType)
    │      │
    │      ├─ AFileStream.Position := 0
    │      ├─ FreeContentStream := False
    │      ├─ ContentLength := AFileStream.Size
    │      ├─ ContentStream := AFileStream
    │      ├─ Content-Disposition: inline
    │      ├─ ContentType設定
    │      └─ SendContent/SendResponse
    │
    └─ LFile.Free

THorseResponse.SendFile(AFileStream: TStream, ...)
    │
    ├─ AFileStream.Position := 0
    ├─ FreeContentStream := False
    ├─ ContentLength := AFileStream.Size
    ├─ ContentStream := AFileStream
    ├─ SetCustomHeader('Content-Disposition', 'inline; filename="..."')
    ├─ ContentType（自動判定または指定値）
    └─ SendContent（FPC）/ SendResponse（Delphi）
```

### データフロー図

```
[入力]                         [処理]                         [出力]

ファイルパス
  /path/to/image.png      ──▶ THorseCoreFile.Create      ──▶ TFileStream
                                   │
                                   ├─ ファイル存在確認
                                   ├─ 名前抽出: "image.png"
                                   └─ MIME判定: "image/png"
                                          │
                                          └─────▶ SendFile(TStream)
                                                      │
                                                      ├─ Position := 0
                                                      ├─ ContentLength設定
                                                      ├─ Content-Disposition: inline
                                                      └─ HTTP Response
                                                              │
                                                              HTTP/1.1 200 OK
                                                              Content-Type: image/png
                                                              Content-Disposition: inline; filename="image.png"
                                                              Content-Length: 12345

                                                              [binary data]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponse クラス、SendFile メソッドの実装 |
| Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | ソース | THorseCoreFile クラス、ファイル処理 |
| Horse.Mime.pas | `src/Horse.Mime.pas` | ソース | THorseMimeTypes クラス、MIMEタイプ判定 |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | TMimeTypes 列挙型 |

### 使用例

```pascal
// ファイルパスから送信
Res.SendFile('/var/www/images/logo.png');

// Content-Typeを明示的に指定
Res.SendFile('/var/www/files/document.pdf', 'application/pdf');

// ストリームから送信
var
  LStream: TMemoryStream;
begin
  LStream := TMemoryStream.Create;
  // ... ストリームにデータを書き込み ...
  Res.SendFile(LStream, 'generated.txt', 'text/plain');
  LStream.Free;
end;

// ステータスコードと組み合わせ
if FileExists(LPath) then
  Res.SendFile(LPath)
else
  Res.Status(THTTPStatus.NotFound).Send('File not found');
```
