# 機能設計書 20-ファイルダウンロード

## 概要

本ドキュメントは、Horse Webフレームワークにおけるファイルをダウンロード用に送信する機能の設計を記述する。THorseResponse.Downloadメソッドによる実装を対象とする。

### 本機能の処理概要

この機能は、サーバー上のファイルをクライアントに送信し、ブラウザでダウンロードダイアログを表示させる。ファイルのダウンロード提供に使用される。

**業務上の目的・背景**：Webアプリケーションでは、ユーザーにファイルをダウンロードさせる機能が必要な場面が多い。この機能により、サーバー上のファイルをクライアントに送信し、ブラウザが自動的にダウンロードダイアログを表示する。Content-Disposition: attachment ヘッダーにより、インライン表示ではなくダウンロードが促される。

**機能の利用シーン**：帳票PDFのダウンロード、CSVエクスポート、Excelファイルのダウンロード、バックアップファイルの取得、ログファイルのダウンロード、ユーザーがアップロードしたファイルの再ダウンロードなど。

**主要な処理内容**：
1. ファイルパスまたはストリームを受け取る
2. ファイル拡張子からMIMEタイプを自動判定（指定がない場合）
3. Content-Disposition: attachment ヘッダーを設定
4. ファイルコンテンツをレスポンスストリームに設定
5. レスポンスを送信

**関連システム・外部連携**：THorseCoreFileクラスでファイルの読み込みとストリーム化を行い、THorseMimeTypesでMIMEタイプを判定する。

**権限による制御**：本機能自体は認証・認可の制御を行わない。ファイルアクセス権限はアプリケーション側のミドルウェアで実装する必要がある。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はAPI機能であり、直接関連する画面はない |

## 機能種別

レスポンス処理 / ファイル送信 / ダウンロード

## 入力仕様

### 入力パラメータ

#### ストリームからの送信

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AFileStream | TStream | Yes | 送信するファイルストリーム | Position=0にリセットされる |
| AFileName | string | Yes | ファイル名（ダウンロード時のファイル名） | 必須 |
| AContentType | string | No | MIMEタイプ | デフォルト: 自動判定 |

#### ファイルパスからの送信

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AFileName | string | Yes | ファイルのフルパス | ファイルの存在確認 |
| AContentType | string | No | MIMEタイプ | デフォルト: 自動判定 |

### 入力データソース

サーバー上のファイルシステム、または生成されたストリーム

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseResponse | メソッドチェーン用の自身への参照 |
| Content-Disposition | Header | attachment; filename="..." |
| Content-Type | Header | 自動判定または指定されたMIMEタイプ |
| Content-Length | Header | ファイルサイズ |
| ContentStream | TStream | ファイルコンテンツ |

### 出力先

HTTPレスポンスとしてクライアントに送信（ダウンロードダイアログ表示）

## 処理フロー

### 処理シーケンス

```
[ストリームからの送信]
1. Download(AFileStream, AFileName, AContentType) の呼び出し
2. AFileStream.Position := 0（ストリーム位置リセット）
3. FreeContentStream := False（外部ストリームのため解放しない）
4. ContentLength := AFileStream.Size
5. ContentStream := AFileStream
6. Content-Disposition: attachment; filename="..." を設定
7. Content-Type を設定（指定なしの場合は拡張子から自動判定）
8. SendContent/SendResponse でレスポンス送信

[ファイルパスからの送信]
1. Download(AFileName, AContentType) の呼び出し
2. THorseCoreFile.Create(AFileName) でファイルオブジェクト生成
3. FreeContentStream := True（ファイルオブジェクト側で管理）
4. ストリーム版Download を呼び出し
5. ファイルオブジェクトを解放
```

### フローチャート

```mermaid
flowchart TD
    A[Download 呼び出し] --> B{引数の型}
    B -->|TStream| C[ストリーム版処理]
    B -->|string path| D[ファイルパス版処理]

    C --> E[Position := 0]
    E --> F[FreeContentStream := False]
    F --> G[ContentLength/ContentStream 設定]

    D --> H[THorseCoreFile.Create]
    H --> I[FreeContentStream := True]
    I --> J[ストリーム版Download呼び出し]
    J --> K[THorseCoreFile.Free]

    G --> L[Content-Disposition: attachment 設定]
    K --> L
    L --> M{ContentType指定あり?}
    M -->|Yes| N[指定値を使用]
    M -->|No| O[拡張子から自動判定]
    N --> P[SendContent/SendResponse]
    O --> P
    P --> Q[ブラウザがダウンロードダイアログ表示]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-20-01 | ダウンロード指示 | Content-Disposition: attachment でダウンロードを指示 | 常時 |
| BR-20-02 | ファイル名指定 | attachment; filename="..." でダウンロード時のファイル名を指定 | 常時 |
| BR-20-03 | MIMEタイプ自動判定 | ContentTypeが空の場合はファイル拡張子から自動判定 | ContentType未指定時 |
| BR-20-04 | ストリーム位置リセット | 送信前にストリームのPositionを0にリセット | 常時 |
| BR-20-05 | ファイル存在確認 | ファイルパス版は存在確認、なければException | ファイルパス指定時 |
| BR-20-06 | デフォルトMIME | 判定できない場合は application/octet-stream | 拡張子不明時 |

### 計算ロジック

Content-Dispositionヘッダーの生成：
```pascal
FWebResponse.SetCustomHeader('Content-Disposition', Format('attachment; filename="%s"', [LFileName]));
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | ファイルパスが空 | 有効なパスを指定 |
| - | Exception | ファイルが存在しない | ファイルの存在を確認 |

### リトライ仕様

リトライ処理は不要（ファイルI/Oエラーは再送で解決しない）

## トランザクション仕様

トランザクション管理は不要（ファイル読み込みのみ）

## パフォーマンス要件

- 大きなファイルはストリーミング送信（メモリ効率的）
- ContentLengthを設定することでクライアントがダウンロード進捗を把握可能

## セキュリティ考慮事項

- パストラバーサル攻撃への対策が必要
- アクセス可能なディレクトリを制限することを推奨
- 機密ファイルへのアクセスには認証・認可を実装
- ファイル名に不正な文字が含まれないよう検証

## 備考

DownloadはSendFileと実装がほぼ同じで、違いはContent-Dispositionヘッダーの値のみ（attachment vs inline）。用途に応じて使い分ける。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ファイル処理に関連するクラスの理解が最初のステップである。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | THorseCoreFile クラスの構造 |
| 1-2 | Horse.Mime.pas | `src/Horse.Mime.pas` | THorseMimeTypes.GetFileType メソッド |

**読解のコツ**: DownloadはSendFileとほぼ同じ実装。Content-Dispositionヘッダーの値（inline vs attachment）が唯一の違い。

#### Step 2: エントリーポイントを理解する

Downloadメソッドの実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Response.pas | `src/Horse.Response.pas` | Download(TStream)、Download(string) メソッド |

**主要処理フロー**:
1. **190-212行目**: Download(TStream) - ストリームからのダウンロード処理
2. **214-231行目**: Download(string) - ファイルパスからのダウンロード処理

#### Step 3: ストリーム版Downloadの詳細を理解する

ストリームからのダウンロード処理を詳細に理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Response.pas | `src/Horse.Response.pas` | Download(TStream) の実装詳細 |

**主要処理フロー**:
- **195行目**: AFileStream.Position := 0
- **198-200行目**: FreeContentStream, ContentLength, ContentStream 設定
- **201行目**: Content-Disposition: attachment 設定
- **203-205行目**: ContentType設定（自動判定含む）
- **207-211行目**: SendContent/SendResponse でレスポンス送信

#### Step 4: SendFileとの比較を理解する

SendFileとDownloadの違いを明確に理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Response.pas | `src/Horse.Response.pas` | SendFile と Download の比較 |

**主要な違い**:
- **SendFile 158行目**: `'inline; filename="%s"'` - ブラウザ内表示
- **Download 201行目**: `'attachment; filename="%s"'` - ダウンロードダイアログ表示

### プログラム呼び出し階層図

```
THorseResponse.Download(AFileName: string)
    │
    ├─ THorseCoreFile.Create(AFileName)
    │      ├─ ファイル存在確認
    │      ├─ ExtractFileName
    │      └─ THorseMimeTypes.GetFileType
    │
    ├─ Download(LFile.ContentStream, LFile.Name, LContentType)
    │      │
    │      ├─ AFileStream.Position := 0
    │      ├─ FreeContentStream := False
    │      ├─ ContentLength := AFileStream.Size
    │      ├─ ContentStream := AFileStream
    │      ├─ Content-Disposition: attachment  ← SendFileとの違い
    │      ├─ ContentType設定
    │      └─ SendContent/SendResponse
    │
    └─ LFile.Free

THorseResponse.Download(AFileStream: TStream, AFileName, ...)
    │
    ├─ AFileStream.Position := 0
    ├─ FreeContentStream := False
    ├─ ContentLength := AFileStream.Size
    ├─ ContentStream := AFileStream
    ├─ SetCustomHeader('Content-Disposition', 'attachment; filename="..."')
    ├─ ContentType（自動判定または指定値）
    └─ SendContent（FPC）/ SendResponse（Delphi）
```

### データフロー図

```
[入力]                         [処理]                         [出力]

ファイルパス
  /path/to/report.pdf     ──▶ THorseCoreFile.Create      ──▶ TFileStream
                                   │
                                   ├─ ファイル存在確認
                                   ├─ 名前抽出: "report.pdf"
                                   └─ MIME判定: "application/pdf"
                                          │
                                          └─────▶ Download(TStream)
                                                      │
                                                      ├─ Position := 0
                                                      ├─ ContentLength設定
                                                      ├─ Content-Disposition: attachment
                                                      └─ HTTP Response
                                                              │
                                                              HTTP/1.1 200 OK
                                                              Content-Type: application/pdf
                                                              Content-Disposition: attachment; filename="report.pdf"
                                                              Content-Length: 54321

                                                              [binary data]
                                                                    │
                                                                    └─ ブラウザがダウンロードダイアログを表示
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponse クラス、Download メソッドの実装 |
| Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | ソース | THorseCoreFile クラス、ファイル処理 |
| Horse.Mime.pas | `src/Horse.Mime.pas` | ソース | THorseMimeTypes クラス、MIMEタイプ判定 |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | TMimeTypes 列挙型 |

### 使用例

```pascal
// ファイルパスから送信
Res.Download('/var/www/reports/monthly_report.pdf');

// ダウンロード時のファイル名を指定（ストリームから）
var
  LStream: TMemoryStream;
begin
  LStream := TMemoryStream.Create;
  try
    // ... CSVデータを生成 ...
    Res.Download(LStream, 'export_2024.csv', 'text/csv');
  finally
    LStream.Free;
  end;
end;

// Content-Typeを明示的に指定
Res.Download('/var/www/files/data.xlsx', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');

// 動的に生成したファイルをダウンロード
procedure ExportUsers(Req: THorseRequest; Res: THorseResponse);
var
  LStream: TStringStream;
begin
  LStream := TStringStream.Create('id,name,email'#13#10'1,John,john@example.com');
  try
    Res.Download(LStream, 'users.csv', 'text/csv');
  finally
    LStream.Free;
  end;
end;

// エラー処理と組み合わせ
if FileExists(LPath) then
  Res.Download(LPath)
else
  Res.Status(THTTPStatus.NotFound).Send('File not found');
```

### SendFile と Download の使い分け

| メソッド | Content-Disposition | 用途 |
|---------|---------------------|------|
| SendFile | inline | 画像表示、PDFプレビュー、テキスト表示 |
| Download | attachment | ファイルダウンロード、エクスポート、バックアップ |
