# 機能設計書 21-HTMLレンダリング

## 概要

本ドキュメントは、HorseフレームワークにおけるHTMLレンダリング機能の設計を記述する。THorseResponse.Renderメソッドを使用して、HTMLファイルをクライアントに送信する機能について詳述する。

### 本機能の処理概要

**業務上の目的・背景**：WebアプリケーションにおいてHTMLコンテンツをクライアントに返却することは基本的な要件である。動的Webページや静的HTMLファイルをブラウザに表示するために、Content-Typeを適切に設定してHTMLファイルを送信する機能が必要となる。本機能により、開発者はHTMLファイルを簡単にレンダリングしてクライアントに返却できる。

**機能の利用シーン**：Webアプリケーションでブラウザにページを表示する際、APIのレスポンスとしてHTMLを返却する際、サーバーサイドで生成したHTMLをクライアントに送信する際に利用される。特にWebアプリケーションのフロントエンド部分を構築する場合に頻繁に使用される。

**主要な処理内容**：
1. HTMLファイルのパスを受け取り、ファイルの存在を確認する
2. ファイルをストリームとして読み込む
3. Content-Typeを`text/html`に設定する
4. HTTPレスポンスとしてHTMLコンテンツを送信する

**関連システム・外部連携**：本機能はHTTPレスポンス送信の基盤となるSendFile機能を内部的に利用する。Delphi/FPC環境のWebサーバー機能と連携してレスポンスを送信する。

**権限による制御**：本機能自体に権限制御機能は含まれない。アプリケーション層で別途認証・認可を実装する必要がある。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（APIレスポンス機能） |

## 機能種別

データ配信機能 / ファイル送信処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AFileName | string | Yes | レンダリングするHTMLファイルのパス | ファイル存在チェック |
| AFileStream | TStream | No | HTMLコンテンツを含むストリーム（オーバーロード版） | nil以外であること |

### 入力データソース

- ファイルシステム上のHTMLファイル
- メモリ上のTStreamオブジェクト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| HTTP Response | TWebResponse/TResponse | Content-Typeがtext/htmlに設定されたHTMLコンテンツ |
| Content-Disposition | string | inline形式でファイル名を含む |
| Content-Length | Integer | 送信するファイルサイズ |

### 出力先

HTTPレスポンスとしてクライアント（ブラウザ）に送信

## 処理フロー

### 処理シーケンス

```
1. Renderメソッド呼び出し
   └─ ファイルパスまたはストリームを受け取る
2. ファイル版の場合：THorseCoreFileでファイルを読み込み
   └─ ファイル存在確認とストリーム化
3. SendFileメソッドを呼び出し
   └─ Content-Typeを'text/html'で固定して送信
4. HTTPレスポンスの送信完了
   └─ クライアントにHTMLが返却される
```

### フローチャート

```mermaid
flowchart TD
    A[Render呼び出し] --> B{入力タイプ}
    B -->|ファイルパス| C[THorseCoreFile生成]
    B -->|ストリーム| D[直接SendFile呼び出し]
    C --> E[ファイル存在確認]
    E -->|存在しない| F[例外発生]
    E -->|存在する| G[ストリーム生成]
    G --> D
    D --> H[Content-Type: text/html設定]
    H --> I[HTTPレスポンス送信]
    I --> J[終了]
    F --> K[エラー終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-21-01 | Content-Type固定 | Render時のContent-Typeは常にtext/htmlとなる | Renderメソッド使用時 |
| BR-21-02 | ファイル存在必須 | 指定されたファイルが存在しない場合は例外を発生させる | ファイルパス指定時 |
| BR-21-03 | インライン表示 | Content-Dispositionはinline形式で設定される | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能ではデータベース操作は発生しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | ファイルが存在しない場合 | 正しいファイルパスを指定する |
| - | Exception | ファイル名が空の場合 | 有効なファイル名を指定する |
| - | Exception | ファイル読み取り権限がない場合 | ファイルの権限を確認する |

### リトライ仕様

リトライ機能は実装されていない。エラー発生時は呼び出し元で適切に処理する必要がある。

## トランザクション仕様

本機能ではトランザクション処理は発生しない。

## パフォーマンス要件

- ファイル読み込みはストリーミング方式で行われ、大容量ファイルでもメモリ効率が良い
- ファイルサイズに応じたレスポンス時間となる

## セキュリティ考慮事項

- パストラバーサル攻撃に対して、入力されるファイルパスの検証をアプリケーション層で実施すべき
- 機密性の高いHTMLファイルへのアクセス制御はアプリケーション層で実装が必要

## 備考

- SendFileメソッドを内部的に利用し、Content-Typeのみをtext/htmlに固定している
- Delphi版とFree Pascal版で同一のインターフェースを提供

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、レスポンス処理に関わる基本的な型とインターフェースを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Commons.pas | `src/Horse.Commons.pas` | TMimeTypes列挙型（89-106行目）でMIMEタイプの定義を確認。TextHTML = 102が重要 |
| 1-2 | Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | THorseCoreFileクラス（17-32行目）でファイル処理の構造を理解 |

**読解のコツ**: DelphiのクラスヘルパーであるTMimeTypesHelper（119-122行目）がToStringメソッドでMIMEタイプを文字列変換している点に注目。

#### Step 2: エントリーポイントを理解する

Render機能の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Response.pas | `src/Horse.Response.pas` | Renderメソッドの宣言（39-40行目）とシグネチャを確認 |

**主要処理フロー**:
1. **39行目**: `function Render(const AFileStream: TStream; const AFileName: string): THorseResponse;` - ストリーム版のシグネチャ
2. **40行目**: `function Render(const AFileName: string): THorseResponse;` - ファイルパス版のシグネチャ

#### Step 3: Render実装の詳細を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Response.pas | `src/Horse.Response.pas` | Renderメソッドの実装部分（233-244行目） |

**主要処理フロー**:
- **233-238行目**: ストリーム版Render実装 - SendFileを呼び出しContent-Typeにtext/htmlを指定
- **240-244行目**: ファイルパス版Render実装 - 同様にSendFileを呼び出し

```pascal
function THorseResponse.Render(const AFileStream: TStream;
  const AFileName: string): THorseResponse;
begin
  Result := Self;
  SendFile(AFileStream, AFileName, Horse.Commons.TMimeTypes.TextHTML.ToString);
end;

function THorseResponse.Render(const AFileName: string): THorseResponse;
begin
  Result := Self;
  SendFile(AFileName, Horse.Commons.TMimeTypes.TextHTML.ToString);
end;
```

#### Step 4: SendFile処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Response.pas | `src/Horse.Response.pas` | SendFileメソッドの実装（147-188行目） |

**主要処理フロー**:
- **147-169行目**: ストリーム版SendFile - FWebResponseにContent-Disposition、ContentLength、ContentStreamを設定
- **152行目**: `AFileStream.Position := 0` - ストリーム位置を先頭に戻す
- **156-157行目**: ContentLengthとContentStreamの設定
- **158行目**: Content-Disposition: inlineヘッダーの設定

#### Step 5: ファイル読み込み処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | THorseCoreFileクラスの実装（44-81行目） |

**主要処理フロー**:
- **44-56行目**: コンストラクタ - ファイル存在確認とメタデータ設定
- **49-50行目**: ファイル存在チェックで例外発生
- **70-75行目**: ContentStreamメソッド - TFileStreamを遅延生成

### プログラム呼び出し階層図

```
THorseResponse.Render(FileName)
    │
    └─ THorseResponse.SendFile(FileName, ContentType)
           │
           ├─ THorseCoreFile.Create(FileName)
           │      ├─ FileExists チェック
           │      └─ ExtractFileName
           │
           ├─ THorseCoreFile.ContentStream
           │      └─ TFileStream.Create
           │
           └─ THorseResponse.SendFile(Stream, FileName, ContentType)
                  │
                  ├─ FWebResponse.ContentLength 設定
                  ├─ FWebResponse.ContentStream 設定
                  ├─ FWebResponse.SetCustomHeader (Content-Disposition)
                  └─ FWebResponse.SendResponse/SendContent
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTMLファイルパス ───▶ THorseCoreFile ───▶ TFileStream
     │                                          │
     │                                          ▼
     │                              THorseResponse.SendFile
     │                                          │
     │                                          ▼
     │                              FWebResponse設定
     │                              - Content-Type: text/html
     │                              - Content-Disposition: inline
     │                              - Content-Length: ファイルサイズ
     │                                          │
     └──────────────────────────────────────▶ HTTPレスポンス送信
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | Renderメソッドの実装 |
| Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | ソース | ファイル読み込みとストリーム化 |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | MIMEタイプの定義と変換 |
| Horse.Mime.pas | `src/Horse.Mime.pas` | ソース | MIMEタイプ判定（Render時は直接使用しない） |
