# 機能設計書 22-リダイレクト

## 概要

本ドキュメントは、HorseフレームワークにおけるHTTPリダイレクト機能の設計を記述する。THorseResponse.RedirectToメソッドを使用して、クライアントを別のURLへリダイレクトさせる機能について詳述する。

### 本機能の処理概要

**業務上の目的・背景**：Webアプリケーションにおいて、ユーザーを別のページや外部サイトへ誘導する必要がある場面は多い。ログイン後のダッシュボードへの遷移、古いURLから新しいURLへの移行、認証失敗時のログインページへの誘導など、HTTPリダイレクトは基本的なWeb機能として不可欠である。本機能により、開発者はHTTPリダイレクトを簡潔に実装できる。

**機能の利用シーン**：ログイン処理後のページ遷移、フォーム送信後の結果ページへの遷移（POST-Redirect-GETパターン）、URLの恒久的な変更通知、条件に応じた動的なページ遷移、認証エラー時のログインページへの誘導などで使用される。

**主要な処理内容**：
1. リダイレクト先URLを受け取る
2. HTTPレスポンスのLocationヘッダーにURLを設定する
3. HTTPステータスコードを303 See Other（デフォルト）または指定されたステータスに設定する
4. クライアントにリダイレクトレスポンスを返却する

**関連システム・外部連携**：本機能はHTTPプロトコルの標準的なリダイレクト仕様に従う。ブラウザやHTTPクライアントはLocationヘッダーを解釈して自動的に新しいURLへアクセスする。

**権限による制御**：本機能自体に権限制御機能は含まれない。リダイレクト先の決定ロジックにおいて権限チェックを実装することは可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（APIレスポンス機能） |

## 機能種別

HTTP制御機能 / リダイレクト処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ALocation | string | Yes | リダイレクト先のURL | - |
| AStatus | THTTPStatus | No | HTTPステータスコード（デフォルト: SeeOther=303） | 有効なリダイレクトステータス |

### 入力データソース

アプリケーションコードからの直接指定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Location Header | string | リダイレクト先URL |
| HTTP Status Code | Integer | 303 See Other（デフォルト）または指定されたステータス |

### 出力先

HTTPレスポンスヘッダーとしてクライアントに送信

## 処理フロー

### 処理シーケンス

```
1. RedirectToメソッド呼び出し
   └─ リダイレクト先URLを受け取る
2. Locationヘッダーの設定
   └─ SetCustomHeaderでLocationヘッダーを追加
3. ステータスコードの設定
   └─ デフォルトは303 See Other、またはカスタムステータス
4. HTTPレスポンス返却
   └─ クライアントにリダイレクトレスポンスを送信
```

### フローチャート

```mermaid
flowchart TD
    A[RedirectTo呼び出し] --> B{ステータス指定}
    B -->|指定あり| C[カスタムステータス使用]
    B -->|指定なし| D[303 See Other使用]
    C --> E[Locationヘッダー設定]
    D --> E
    E --> F[Statusメソッドでステータス設定]
    F --> G[HTTPレスポンス返却]
    G --> H[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-22-01 | デフォルトステータス | ステータス未指定時は303 See Otherを使用 | ステータス引数なしのオーバーロード使用時 |
| BR-22-02 | Locationヘッダー必須 | リダイレクト時は必ずLocationヘッダーを設定 | 常時 |
| BR-22-03 | フルーエントインターフェース | メソッドは自身のインスタンスを返却しメソッドチェーンを可能にする | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能ではデータベース操作は発生しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 本機能では明示的なエラー処理なし | - |

### リトライ仕様

リトライ機能は実装されていない。

## トランザクション仕様

本機能ではトランザクション処理は発生しない。

## パフォーマンス要件

- 単純なヘッダー設定のみのため、処理は即座に完了する
- 追加のI/O処理は発生しない

## セキュリティ考慮事項

- オープンリダイレクト脆弱性に注意：ユーザー入力をそのままリダイレクト先として使用しないこと
- リダイレクト先URLの検証をアプリケーション層で実装すべき
- 信頼できるドメインのホワイトリストを使用することを推奨

## 備考

- 303 See Otherは主にPOSTリクエスト後のリダイレクトで使用される（POST-Redirect-GETパターン）
- 301 Moved Permanentlyや302 Foundなど他のリダイレクトステータスも指定可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

リダイレクトに関わるHTTPステータスコードの定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Commons.pas | `src/Horse.Commons.pas` | THTTPStatus列挙型（24-87行目）でリダイレクト関連ステータスを確認 |

**読解のコツ**: THTTPStatusの中で特にリダイレクト関連のステータスに注目する。
- `SeeOther = 303` （デフォルトで使用される）
- `MovedPermanently = 301`
- `Found = 302`
- `TemporaryRedirect = 307`
- `PermanentRedirect = 308`

#### Step 2: エントリーポイントを理解する

RedirectTo機能の宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Response.pas | `src/Horse.Response.pas` | RedirectToメソッドの宣言（31-32行目） |

**主要処理フロー**:
1. **31行目**: `function RedirectTo(const ALocation: string): THorseResponse;` - URL指定のみのオーバーロード
2. **32行目**: `function RedirectTo(const ALocation: string; const AStatus: THTTPStatus): THorseResponse;` - URL+ステータス指定

#### Step 3: RedirectTo実装の詳細を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Response.pas | `src/Horse.Response.pas` | RedirectToメソッドの実装部分（119-129行目） |

**主要処理フロー**:
- **119-123行目**: URL指定のみのオーバーロード実装

```pascal
function THorseResponse.RedirectTo(const ALocation: string): THorseResponse;
begin
  FWebResponse.SetCustomHeader('Location', ALocation);
  Result := Status(THTTPStatus.SeeOther);
end;
```

- **125-129行目**: URL+ステータス指定のオーバーロード実装

```pascal
function THorseResponse.RedirectTo(const ALocation: string; const AStatus: THTTPStatus): THorseResponse;
begin
  FWebResponse.SetCustomHeader('Location', ALocation);
  Result := Status(AStatus);
end;
```

#### Step 4: Statusメソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Response.pas | `src/Horse.Response.pas` | Statusメソッドの実装（141-145行目） |

**主要処理フロー**:
- **141-145行目**: THTTPStatus版のStatus設定

```pascal
function THorseResponse.Status(const AStatus: THTTPStatus): THorseResponse;
begin
{$IF DEFINED(FPC)}FWebResponse.Code{$ELSE}FWebResponse.StatusCode{$ENDIF} := AStatus.ToInteger;
  Result := Self;
end;
```

#### Step 5: HTTPステータスのInteger変換を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | Horse.Commons.pas | `src/Horse.Commons.pas` | THTTPStatusHelper.ToInteger（251-254行目） |

**主要処理フロー**:
- **251-254行目**: 列挙型をInteger値に変換

```pascal
function THTTPStatusHelper.ToInteger: Integer;
begin
  Result := Ord(Self);
end;
```

### プログラム呼び出し階層図

```
THorseResponse.RedirectTo(Location)
    │
    ├─ FWebResponse.SetCustomHeader('Location', Location)
    │
    └─ THorseResponse.Status(THTTPStatus.SeeOther)
           │
           ├─ THTTPStatus.ToInteger
           │      └─ Ord(SeeOther) = 303
           │
           └─ FWebResponse.StatusCode := 303
```

### データフロー図

```
[入力]                    [処理]                         [出力]

リダイレクトURL ───▶ SetCustomHeader ───▶ Locationヘッダー
                          │
                          ▼
                    Statusメソッド
                          │
                          ▼
                    ToInteger変換 ───▶ HTTPステータスコード
                                              │
                                              ▼
                                       HTTPレスポンス
                                       - Status: 303
                                       - Location: URL
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | RedirectToメソッドの実装 |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | THTTPStatus列挙型とヘルパー |
