# 機能設計書 23-ヘッダー追加

## 概要

本ドキュメントは、HorseフレームワークにおけるHTTPレスポンスヘッダー追加機能の設計を記述する。THorseResponse.AddHeaderメソッドを使用して、カスタムHTTPヘッダーをレスポンスに追加する機能について詳述する。

### 本機能の処理概要

**業務上の目的・背景**：HTTPレスポンスヘッダーは、クライアントに対してさまざまなメタ情報を伝達するために使用される。キャッシュ制御、セキュリティヘッダー、カスタムアプリケーションヘッダーなど、柔軟なヘッダー設定が必要となる場面は多い。本機能により、開発者は任意のHTTPヘッダーをレスポンスに追加できる。

**機能の利用シーン**：CORS（Cross-Origin Resource Sharing）ヘッダーの設定、キャッシュ制御ヘッダー（Cache-Control、Expires等）の設定、セキュリティヘッダー（X-Content-Type-Options、X-Frame-Options等）の設定、カスタムアプリケーションヘッダー（X-Request-ID等）の設定、認証トークンヘッダーの設定などで使用される。

**主要な処理内容**：
1. ヘッダー名と値を受け取る
2. FWebResponseのSetCustomHeaderメソッドを呼び出す
3. カスタムヘッダーをレスポンスに追加する
4. 自身のインスタンスを返却してメソッドチェーンを可能にする

**関連システム・外部連携**：本機能はHTTPプロトコルの標準的なヘッダー仕様に従う。追加されたヘッダーはクライアント（ブラウザやHTTPクライアント）で参照・利用される。

**権限による制御**：本機能自体に権限制御機能は含まれない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（APIレスポンス機能） |

## 機能種別

HTTP制御機能 / ヘッダー操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AName | string | Yes | ヘッダー名 | - |
| AValue | string | Yes | ヘッダー値 | - |

### 入力データソース

アプリケーションコードからの直接指定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Custom Header | string | 追加されたカスタムヘッダー |

### 出力先

HTTPレスポンスヘッダーとしてクライアントに送信

## 処理フロー

### 処理シーケンス

```
1. AddHeaderメソッド呼び出し
   └─ ヘッダー名と値を受け取る
2. SetCustomHeaderの呼び出し
   └─ FWebResponse.SetCustomHeaderでヘッダーを追加
3. 自身のインスタンス返却
   └─ メソッドチェーンを可能にする
```

### フローチャート

```mermaid
flowchart TD
    A[AddHeader呼び出し] --> B[AName, AValue受け取り]
    B --> C[FWebResponse.SetCustomHeader呼び出し]
    C --> D[ヘッダー追加完了]
    D --> E[Self返却]
    E --> F[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-23-01 | フルーエントインターフェース | メソッドは自身のインスタンスを返却しメソッドチェーンを可能にする | 常時 |
| BR-23-02 | 同名ヘッダー上書き | 同じ名前のヘッダーが既に存在する場合の挙動はFWebResponseの実装に依存 | ヘッダー追加時 |
| BR-23-03 | バリデーションなし | ヘッダー名・値のバリデーションは行わない | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能ではデータベース操作は発生しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 本機能では明示的なエラー処理なし | - |

### リトライ仕様

リトライ機能は実装されていない。

## トランザクション仕様

本機能ではトランザクション処理は発生しない。

## パフォーマンス要件

- 単純なヘッダー追加のみのため、処理は即座に完了する
- 追加のI/O処理は発生しない

## セキュリティ考慮事項

- HTTPヘッダーインジェクション攻撃に注意：ユーザー入力をそのままヘッダー値として使用しないこと
- CRLFインジェクションの可能性があるため、ヘッダー値に改行文字が含まれないよう検証すべき
- セキュリティヘッダーの適切な設定を推奨（X-Content-Type-Options、X-Frame-Options等）

## 備考

- Delphi/FPC環境のTWebResponse/TResponseのSetCustomHeaderメソッドを利用
- 既存のヘッダーとの重複時の挙動は下層のWeb実装に依存

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

レスポンスクラスの基本構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Response.pas | `src/Horse.Response.pas` | THorseResponseクラスの定義（24-50行目）でFWebResponseフィールドを確認 |

**読解のコツ**: FWebResponseは条件コンパイルにより、FPC環境ではTResponse、Delphi環境ではTWebResponse型となる。これがHTTPレスポンスの実体を保持している。

#### Step 2: エントリーポイントを理解する

AddHeader機能の宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Response.pas | `src/Horse.Response.pas` | AddHeaderメソッドの宣言（42行目） |

**主要処理フロー**:
1. **42行目**: `function AddHeader(const AName, AValue: string): THorseResponse; virtual;`

#### Step 3: AddHeader実装の詳細を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Response.pas | `src/Horse.Response.pas` | AddHeaderメソッドの実装部分（63-67行目） |

**主要処理フロー**:
- **63-67行目**: AddHeader実装

```pascal
function THorseResponse.AddHeader(const AName, AValue: string): THorseResponse;
begin
  FWebResponse.SetCustomHeader(AName, AValue);
  Result := Self;
end;
```

#### Step 4: FWebResponseの理解

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Response.pas | `src/Horse.Response.pas` | FWebResponseのフィールド定義（26行目） |

**主要処理フロー**:
- **26行目**: 条件コンパイルによる型定義

```pascal
FWebResponse: {$IF DEFINED(FPC)}TResponse{$ELSE}TWebResponse{$ENDIF};
```

### プログラム呼び出し階層図

```
THorseResponse.AddHeader(Name, Value)
    │
    └─ FWebResponse.SetCustomHeader(Name, Value)
           │
           └─ TWebResponse/TResponse の内部実装
                  └─ CustomHeaders リストに追加
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ヘッダー名 (AName) ───▶ SetCustomHeader ───▶ HTTPレスポンスヘッダー
ヘッダー値 (AValue) ──▶                              │
                                                      ▼
                                              クライアントに送信
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | AddHeaderメソッドの実装 |
| Web.HTTPApp | Delphi標準ライブラリ | 外部ライブラリ | TWebResponse実装（Delphi環境） |
| HTTPDefs | FPC標準ライブラリ | 外部ライブラリ | TResponse実装（FPC環境） |
